"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import logging
import tempfile
import shutil
import tarfile
import re
import subprocess

from docker.errors import ImageNotFound, APIError

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.lib.dockerlib import DockerLib
from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.config import get_compose_path

logger = logging.getLogger(__name__)


REGISTRY_VOLUME_PATH    = r'/var/lib/'
REGISTRY_CONTAINER_NAME = r'registry'
REGISTRY_DEFAULT_PORT=5000
TMP_DIR_MIN_SIZE = 4 # GB


def start_docker_registry():
    """ Start the docker registry """

    docker = DockerLib()

    try:
        registry_version = Config().get('docker', 'registry_version')
    except Exception as e:
        raise errors.InvalidConfig(e)

    registry_name = REGISTRY_CONTAINER_NAME
    image_name = registry_name + ':' + registry_version

    def _recreate_registry():
        """Inner version called if registry need to be create/recreate """

        img = docker.docker.images.get(name=image_name)

        # stop and delete registry container
        docker.rm_container_if_running(registry_name)    

        # create & start
        registry_container = docker.docker.containers.create(
            image=img,
            name=registry_name,
            ports={REGISTRY_DEFAULT_PORT: docker.registry_port},
            volumes={REGISTRY_VOLUME_PATH:
                         {'bind': '/var/lib/registry', 'mode': 'rw'}},
            environment=[
                "REGISTRY_STORAGE_FILESYSTEM_ROOTDIRECTORY=/var/lib/registry",
                "REGISTRY_STORAGE_DELETE_ENABLED=true"],
            restart_policy={"MaximumRetryCount": 0, "Name": "always"})

        registry_container.start()

        logger.info(":: docker registry created: name=%s version=%s status=%s" %
                    (registry_container.name,
                     registry_version,
                     registry_container.status))

    try:

        if not os.path.exists(REGISTRY_VOLUME_PATH):
            os.makedirs(REGISTRY_VOLUME_PATH)

        containers_list = docker.docker.containers.list(all=True)

        if not any(c.name == registry_name for c in containers_list):
            logger.info(":: Registry container doesn't exist, will be created")
            _recreate_registry()

        else :

            currentPorts = docker.docker_api.inspect_container(registry_name)['HostConfig']['PortBindings']
            #Get current containerPort and hostPort, ie. 5000
            containerPort = currentPorts.keys()[0].split("/")[0]
            hostPort = currentPorts.get(currentPorts.keys()[0])[0]['HostPort']

            if ((containerPort != str(REGISTRY_DEFAULT_PORT)) or
            (hostPort != str(docker.registry_port)) or
            (any((ctn.name == registry_name and ctn.status != 'running') for ctn in containers_list))):

                logger.info(":: Registry parameters have been modified, registry will be recreated")
                # get all compose container names and remove all compose containers 
                # BUGFIX52250 
                cmd = "cd %s && sudo docker-compose -f ./compose_mdt.yaml ps --services" % get_compose_path()
                proc = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                (proc_output, proc_error) = proc.communicate()
                proc.wait()

                for c in proc_output.split('\n')[0:-1]:
                    docker.rm_container_if_running(c)

                # recreate registry container
                _recreate_registry()

    except ImageNotFound:
        logger.error('Registry image -%s- not found' % image_name)
        raise
    except APIError:
        logger.error('Starting registry failed: server Error')
        raise
    except errors.InitError as e:
        logger.error("ERROR: %s" % e)
        raise

def mount_registry_nfs(address, path):
    """
    Permanent NFS mount
    :param address:
    :param path:
    :return:
    """
    source = address + ':' + path
    target = os.path.join(REGISTRY_VOLUME_PATH, 'docker', 'registry')
    fstype = "nfs"

    # Add permanent mount (case reboot) if not already in /etc/fstab
    with open('/etc/fstab','ab+') as fstab:
        if source not in fstab.read():
            fstab.write("\n" + source + " " + target + " nfs defaults 0 0")
        fstab.close()

    if not os.path.exists(target):
        os.makedirs(target)

    logger.info(":: docker registry NFS Storage mount: source=%s target=%s" %
                (source,
                 target))
    try:
        proc = subprocess.Popen(["mount", "-t", fstype, source, target],
                         stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        proc.wait()
    except OSError as e:
        logger.error('Error while mounting NFS: %s' % e.strerror)
        raise
    except:
        raise


def load_docker_registry(path_tgz=None,path_dir=None,registry=None):
    """
    Upload containers to the Docker registry used by MDT (default or external)
    :param path_tgz:
    :param path_dir:
    :param registry:
    :return:
    """
    if path_dir:
        if os.path.exists(path_dir):
            for item in os.listdir(path_dir):
                item_path = os.path.join(path_dir, item)
                if os.path.isdir(item_path):
                    load_docker_registry(path_dir=item_path,registry=registry)
                else:
                    load_targz(item_path,registry)
        else:
             raise errors.LoadError("Directory %s not found ! " % path_dir)

    elif path_tgz:
        load_targz(path_tgz,registry)


def load_targz(tgz,registry=None):
    """

    :param tgz:
    :param registry:
    :return:
    """
    # TODO use external registry -> address, port
    docker = DockerLib()
    tmp = None
    try:
        logger.info(":: loading containers from %s" % tgz)

        # Check if file exist
        if not os.path.exists(tgz):
            raise errors.LoadError("file %s not found ! " % tgz)

        tgz_size = os.stat(tgz).st_size
        tmp = _create_tmp_dir(tgz_size)
        if tmp is None:
            raise errors.LoadError("Need %s GB free space" % TMP_DIR_MIN_SIZE)

        # is a readable tar file ?
        if tarfile.is_tarfile(tgz) is not True:
            raise errors.LoadError("%s is not a tar file" % tgz)
        # open and extract
        tar = tarfile.open(tgz)

        def _load(_img, _img_path):
            # load image(s)  locally
            images = docker.load_tgz_image(_img_path)
            # push them on registry only if they are loaded in the previous step
            # it means that there isn't already image with the same tag
            # and of course the registry is up
            for image in images:
                #If repository identified in image name, remove repository info by renaming image
                if "/" in image:
                    oldrepo,image_new=image.rsplit('/', 1)
                    full_name,img_tag=image_new.split(':')
                    img = docker.docker.images.get(image)
                    img.tag(full_name, img_tag)
                    docker.docker.images.remove(image)
                #If no repo info in name, split as usual
                else:
                    full_name,img_tag=image.split(':')
                docker.push_image(full_name, img_tag)
            else:
                logger.info("")

        multi_cont = True
        for member in tar.getmembers():
            member_ext = os.path.splitext(member.name)[1]
            if member.isfile() and \
               member_ext != '' and \
               (re.match(member_ext, '?gz') or
                re.match(member_ext, '.tar') or
                re.match(member_ext, '.tgz')):

                # extract the image
                img = _extract_tarfile(tar, member, tmp)
                img_path = os.path.join(tmp, img)
                _load(img, img_path)

                # make some space
                os.remove(img_path)
            elif member.isdir():
                continue
            else:
                #not an archive with multiple containers archives ?
                multi_cont = False
                break

        # close tar obj
        tar.close()

        # try to load archive directly as a container
        if not multi_cont:
            _load(tgz, os.path.join('', tgz))

    except Exception as e:
        __cleanup(tmp)
        raise #errors.LoadError(e)
    finally:
        __cleanup(tmp)


def __cleanup(tmp):
    if tmp is not None and os.path.exists(tmp):
        shutil.rmtree(tmp)


def _extract_tarfile(tar, member, tmp):

    logger.debug("extracting image: %s " % member.name)

    if member.name.endswith('.tar') or \
            member.name.endswith('.tar.gz') or \
            member.name.endswith('.tgz'):

        # check if there is not ../ in file path
        if re.match(r'\.\./', member.name):
            raise errors.LoadError(
                "suspicious filename with /../ in path")

        # check if filename not begin with /
        if re.match('^/', member.name):
            raise errors.LoadError(
                "suspicious filename start with /")

        # extract image
        logger.info("  -> extracting %s" % member.name)
        tar.extract(member, path=tmp)

        return member.name
    else:
        raise errors.LoadError(
            "containers in archive must be .tar - .tar.gz or .tgz file")


def _create_tmp_dir(size):
    """ Create a temp directory to extract tar.gz file """
    # TODO check /tmp space otherwise use ~/

    def check_free(tmpdir):
        _ret = False
        f = os.statvfs(tmpdir)
        free_size = f.f_frsize * f.f_bavail
        logger.debug(" free space on %s =  %s" % (tmpdir, free_size))
        if free_size > size:
            _ret = True
        return _ret

    # check free size in /tmp /var or /home
    tmp_list = [tempfile.gettempdir(), '/var', os.environ['HOME']]
    for _tmp in tmp_list:
        if check_free(_tmp):
            return tempfile.mkdtemp(dir=_tmp)

    return None
