#!/bin/bash



##################################################
### Install MDT                                ###
##################################################

SCRIPTNAME="`basename "$0"`"
SCRIPTDIR=$(dirname "$(readlink -f "$0")")

LOGFILE="mdt_$SCRIPTNAME.log"

REPO_NAME=mdt_repo
REPO_FILE=/etc/yum.repos.d/mdt.repo
REPO_DIR=rpm_repo
RESOURCES_DIR=resources

DOCKER_OVERLAY_FS="true"

HELM=helm-v*-linux-amd64.tar.gz

__exit_val=0


#######################
# Functions
#######################


show_help() {
   cat << EOF
Usage: $SCRIPTNAME [-h help]
Install MDT. Options are
    -h|--help             : Display this help and exit
    -f|--no-overlay-fs    : Disable docker overlayFS backend
EOF
}


# print error and exit
die ()
{
    local __str="$1"
    echo $__str
    exit 1
}


__cleanup()
{
#   echo -e "\nCleaning up..."

   echo -e "\nExiting..."
   exit $__exit_val
}


trap __cleanup SIGHUP SIGINT SIGTERM


create_local_repo()
{
   cat <<EOM >$REPO_FILE
[$REPO_NAME]
name=Local MDT repository
baseurl=file://$SCRIPTDIR/$REPO_DIR
enabled=1
gpgcheck=0
EOM
}


## BUG 58729 - LRO
## Docker must be installed at install step and not and the init step, because registry yum is
## available by container MDT bootstrap httpd
## Update installation to be consistent with Kubespray Docker installation
install_docker ()
{
    # Install Docker version in REPO_NAME
    yum --disablerepo=* --enablerepo=$REPO_NAME -y install --setopt=obsoletes=0 docker-ce
    usermod -aG docker root

    # Service file
    mkdir -p /etc/systemd/system/docker.service.d
    cat << EOF > /etc/systemd/system/docker.service
[Unit]
Description=Docker Application Container Engine
Documentation=http://docs.docker.com
After=network.target docker-storage-setup.service
Wants=docker-storage-setup.service

[Service]
Type=notify
ExecReload=/bin/kill -s HUP $MAINPID
Delegate=yes
KillMode=process
ExecStart=/usr/bin/dockerd \\
          \$DOCKER_OPTS \\
          \$DOCKER_STORAGE_OPTIONS \\
          \$DOCKER_NETWORK_OPTIONS \\
          \$DOCKER_DNS_OPTIONS \\
          \$INSECURE_REGISTRY
LimitNOFILE=1048576
LimitNPROC=1048576
LimitCORE=infinity
TimeoutStartSec=1min
# restart the docker process if it exits prematurely
Restart=on-failure
StartLimitBurst=3
StartLimitInterval=60s

[Install]
WantedBy=multi-user.target

EOF

    ## Configuration file
    cat << EOF > /etc/systemd/system/docker.service.d/docker-options.conf
[Service]
Environment="DOCKER_OPTS=--log-opt max-size=50m --log-opt max-file=5"

EOF

    # Install Docker compose
    cp /opt/mfvp/deploypattern/ansible/files/docker-compose /usr/bin/docker-compose
    chmod 0744 /usr/bin/docker-compose

    ## BUG 61208 - LRO
    ## Backup Docker configuration files, which are deleted by reset kube cluster
    cp /etc/systemd/system/docker.service.d/docker-options.conf /etc/systemd/system/docker.service.d/docker-options.conf_bk
    cp /etc/systemd/system/docker.service /etc/systemd/system/docker.service_bk

    systemctl daemon-reload
    systemctl enable docker
    systemctl start docker
    systemctl status docker | grep running > /dev/null 2>&1
    [ $? -eq 0 ] || die "Error: [DOCKER] Docker service is not started"
}



#######################
# Main
#######################


exec > >(tee -a $LOGFILE)
exec 2>&1

cd $SCRIPTDIR
echo "============>"
echo "============> $(date)"
echo "============>"
echo


INSTALL_CONF=$RESOURCES_DIR/install.conf
[ -f $INSTALL_CONF ] && source $INSTALL_CONF || exit_on_error "$INSTALL_CONF is not a file"

cat /etc/redhat-release | grep $SUPPORTED_REDHAT_RELEASE #2>&1 > /dev/null
if [ $? -ne 0 ]; then
    die "Error: Red Hat release not supported: $SUPPORTED_REDHAT_RELEASE "
fi


while [ $# -gt 0 ]; do
   case "$1" in
      -h|--help)
         show_help;
         exit 1
         ;;
      -f|--no-overlay-fs)
         DOCKER_OVERLAY_FS="false"
         ;;
      *)
         echo "Invalid option"
         show_help
         exit 1
         ;;
   esac
   shift
done

echo -e "\n***** Creating local repository $REPO_NAME ..."
create_local_repo


echo -e "\n***** Installing $RPM_NAME package ..."
find $REPO_DIR -name "$RPM_NAME*.rpm" -exec yum --disablerepo=* --enablerepo=$REPO_NAME -y install {} \;

echo -e "\n***** Installing helm package ..."
tar -zxf $RESOURCES_DIR/$HELM && mv ./linux-amd64/helm /usr/bin/helm
rm -rf RESOURCES_DIR/linux-amd64

echo -e "\n***** Installing docker ..."
install_docker

echo -e "\n***** Configure Ansible ..."
mkdir -p /etc/ansible/
cp /opt/mfvp/deploypattern/ansible/files/ansible.conf  /etc/ansible/ansible.cfg
touch /var/log/ansible.log
chmod 0666 /var/log/ansible.log

echo -e "\n***** create path for static files ..."
mkdir -p /var/www/ericsson/static/  && chown daemon:daemon /var/www/ericsson/static/

rm -rf $REPO_FILE
echo
echo

exec 1>&- 2>&-

kill -TERM $$
