"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import sys
import signal
import argparse
import logging
import logging.handlers
import subprocess
import os
import shutil
import tarfile
from ericsson.deploypattern.tools.mdtcli import exec_cmd_shell

from ericsson.deploypattern.lib import errors
from ericsson.deploypattern.lib import ansibleplay
from ericsson.deploypattern.lib import config as config_lib
from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.inventory import inventory_command
from ericsson.deploypattern.plugins.deployplugins import load_plugin
from ericsson.deploypattern.plugins.pattern import pattern
from ericsson.deploypattern.tools.dfninit import mgnt_server_init, check_init
from ericsson.deploypattern.tools.yum import load_yum_repo
from ericsson.deploypattern.tools.apt import load_deb_repo
from ericsson.deploypattern.tools.registry import load_docker_registry
from ericsson.deploypattern.tools import helm
from ericsson.deploypattern.tools import bundle
from ericsson.deploypattern.tools import mdtcli
from ericsson.deploypattern.version import __version__

class LoggingLevelFilter(logging.Filter):

    def __init__(self, min_level=logging.NOTSET, max_level=logging.CRITICAL):
        super(logging.Filter, self).__init__()
        self.min_level = min_level
        self.max_level = max_level

    def filter(self, record):
        return record.levelno \
            if self.min_level <= record.levelno <= self.max_level \
            else False

# create logger
LOGGER = logging.getLogger()
LOGGER.setLevel(logging.INFO)

# Console Handlerr for stderr
CH_ERR = logging.StreamHandler(stream=sys.stderr)
CH_ERR.addFilter(LoggingLevelFilter(min_level=logging.ERROR))
CH_ERR.setLevel(logging.ERROR)
# Console Handler for stdout
CH_OUT = logging.StreamHandler(stream=sys.stdout)
CH_OUT.addFilter(LoggingLevelFilter(min_level=logging.DEBUG, max_level=logging.WARNING))
CH_OUT.setLevel(logging.DEBUG)

LOGGER.addHandler(CH_ERR)
LOGGER.addHandler(CH_OUT)

# File Handler
FORMATTER = logging.Formatter('%(asctime)s:'
                              '%(levelname)s:'
                              '%(filename)s:'
                              '%(funcName)s:'
                              ' %(message)s')
F_H = logging.handlers.RotatingFileHandler(
    '/tmp/deploypattern.log', 'a', 1000000, 1)
# F_H = logging.FileHandler('/var/log/deploypattern.log')
F_H.setLevel(logging.INFO)
F_H.setFormatter(FORMATTER)
LOGGER.addHandler(F_H)

# For Ansible Log
# Ansible add FileHandler to root logger
# Search ansible logger with its key: p=<PID ansible> u=<user> |
# Don't propagate the log to parent logger (root), not to have twice log written
# Add your Handler ansible logger to have Ansible log
for key in logging.Logger.manager.loggerDict.keys():
    if ' | ' in key:
        logging.Logger.manager.loggerDict[key].propagate = False
        logging.Logger.manager.loggerDict[key].addHandler(F_H)

DFN_CONF_FILE = '/etc/deploypattern.conf'


class ArgumentParser(argparse.ArgumentParser):
    """
    Override error method in argparse.ArgmentParser
    """

    def error(self, message):
        self.print_help(sys.stderr)
        self.exit(2, '%s: error: %s\n' % (self.prog, message))


def run_pattern(deploy_pattern):
    """ Parse deployment pattern

    Launch the plugin related to the task type
    And run it

    """
    tasks_list = deploy_pattern['deployment_tasks']

    for task in tasks_list:
        LOGGER.info("\n:: %s ::", task['name'])
        LOGGER.info("--------------------------")

        # get task type
        __deployment_type = task['type']

        # load the plugin
        plugin = load_plugin(__deployment_type)
        plugin(task)


def redirect_to_cli(command):
    """
    To manage Help
    :param command:
    :return:
    """
    def handle(parsed_args, unknown_args):
        """
        Handle to manage Help
        :param parsed_args:
        :param unknown_args:
        :return handle:
        """
        if unknown_args.count("-h") != 0 or unknown_args.count("--help") != 0:
            cmd = 'sudo docker exec --user root mdt-cli mdt {} {}'.format(
                command, ' '.join(unknown_args))
            returncode, stdout, stderr = exec_cmd_shell(cmd)
            if returncode == 0:
                LOGGER.info(stdout)
                exit(0)
            else:
                LOGGER.error("stderr : %s", stderr)
                exit(1)
            try:
                unknown_args.remove("-h")
                unknown_args.remove("-help")
            except ValueError:
                pass
    return handle


def parse_cmdline():
    """
    Parse command line arguments
    """
    parser = ArgumentParser(prog="mdt",
                            usage='mdt [mdt_options] [command] [resource] [command_options]',
                            description=('''
    MDT: MediaFirst Deployment Toolkit
    ==================================
    Tool to deploy Kubernetes cluster and products

                                     '''),
                            formatter_class=argparse.RawDescriptionHelpFormatter)

    # Optional arguments for the MDT
    parser.add_argument('--version', action='version',
                        version="- %(prog)s {version} -"
                        .format(version=__version__))

    parser.add_argument("--debug",
                        action="store_true",
                        help="Enable Debug output messages")

    parser.add_argument("-c", "--config",
                        dest="conf",
                        help="configuration file; "
                        "default is " + DFN_CONF_FILE)

    # Subparser commands
    subparsers = parser.add_subparsers(dest='cmd')
    subparsers.required = False
    # del_parser = subparsers.add_parser('delete',
    #                                   usage='mdt [mdt_options] delete [resource] [command_options].',
    #                                   help='Delete resource')
    deploy_parser = subparsers.add_parser('deploy',
                                          usage='mdt [mdt_options] deploy [resource] [command_options]',
                                          help='Deploy resource')
    get_parser = subparsers.add_parser('get',
                                       usage='mdt [mdt_options] get [resource] [command_options]',
                                       help='Get resource')
    init_parser = subparsers.add_parser('init',
                                        description=('''
!!!! Command available only for MDT bootstrap !!!!

Initialize the MDT software to setup the environment appropriately.
For allinone installation the local disk will be used instead of NFS server
                                        '''),
                                        formatter_class=argparse.RawDescriptionHelpFormatter,
                                        usage='mdt [mdt_options] init  [-i IP_ADDRESS] [-k SSH_KEY] [-g REGISTRY] [-m HELM_REPO_URL] [-d REGISTRY_NFS_ADDRESS] [-p REGISTRY_NFS_PATH] [-u USER] [-r {yum,deb}] [-a ANSIBLE_PATH] [-c 3RDPARTY_PATH] [--http-port HTTP_PORT] [--skip_load] [--stop]',
                                        help='MDT initialisation')
    load_parser = subparsers.add_parser('load',
                                        usage='mdt [mdt_options] load [resource] [command_options]',
                                        help='Load resources')
    reset_parser = subparsers.add_parser('reset',
                                         usage='mdt [mdt_options] reset [resource] [command_options]',
                                         help='Reset ressources')
    set_parser = subparsers.add_parser('set',
                                       usage='mdt [mdt_options] set [resource] [command_options]',
                                       help='Set ressources')
    unset_parser = subparsers.add_parser('unset',
                                         usage='mdt [mdt_options] unset [resource] [command_options]',
                                         help='Unset ressources')
    add_val_parser = subparsers.add_parser('add',
                                           usage='mdt [mdt_options] add [resource] [command_options]',
                                           help='Add ressources')
    delete_parser = subparsers.add_parser('delete',
                                          usage='mdt [mdt_options] delete [resource] [command_options]',
                                          help='Delete ressources')
    upgrade_parser = subparsers.add_parser('upgrade',
                                           usage='mdt [mdt_options] upgrade [resource] [command_options]',
                                           help='Upgrade ressources')
    audit_parser = subparsers.add_parser('audit',
                                         usage='mdt [mdt_options] audit [resource] [command_options]',
                                         help='Audit ressources')
    # Command deploy
    # -----------------
    deploy_sub_parser = deploy_parser.add_subparsers(dest='cmd_deploy')
    deploy_helm_parser = deploy_sub_parser.add_parser('helm',
                                                      description='Install Helm repository on MDT node and Helm server tiller, on one master.',
                                                      usage='mdt [mdt_options] deploy helm')

    deploy_kubecluster_parser = deploy_sub_parser.add_parser('kube-cluster',
                                                             description=('''
Deploy the kube-cluster, Kubernetes on the nodes masters and workers. The inventory, list of nodes, is given by the mdt-matrix file in the config bundle folder.
The configuration of Kubernetes cluster as HA registry, HA proxy, load balancer ... are given in kube-config file in the config bundle folder.
                                                '''),
                                                             formatter_class=argparse.RawDescriptionHelpFormatter,
                                                             usage='mdt [mdt_options] deploy kube-cluster')

    deploy_products_parser = deploy_sub_parser.add_parser(
        'products', add_help=False)
    deploy_products_parser.set_defaults(
        func=redirect_to_cli('deploy products'))

    deploy_volumes_parser = deploy_sub_parser.add_parser('volumes',
                                                         description=('''
Deploy storage classes from kube-volumes file in config bundle folder.
A storage class is defined with 4 parameters:
* name
* type
* address
* path
                                                '''),
                                                         formatter_class=argparse.RawDescriptionHelpFormatter,
                                                         usage='mdt [mdt_options] deploy volumes')

    # --> mdt deploy helm
    # no parameter

    # --> mdt deploy kube-cluster
    # no parameter

    # --> mdt deploy products [-p profile_name]
#    deploy_products_parser.add_argument("-p", "--profile",
#                                        dest="profile_name",
#                                        default=None,
#                                        help="Specific profile"
#                                        )

    # --> mdt deploy volumes
    # no parameter

    # Command reset
    # ----------------
    reset_sub_parser = reset_parser.add_subparsers(dest='cmd_reset')
    reset_product_parser = reset_sub_parser.add_parser(
        'products', add_help=False)
    reset_product_parser.set_defaults(func=redirect_to_cli('reset products'))

    reset_kubecluster_parser = reset_sub_parser.add_parser('kube-cluster',
                                                           description=('''
Reset the Kubernetes cluster on all nodes and clean load balancers. 
The inventory, list of nodes, is given by the mdt-matrix file in the config bundle folder.
                                                '''),
                                                           formatter_class=argparse.RawDescriptionHelpFormatter,
                                                           usage='mdt [mdt_options] reset kube-cluster')
    reset_volumes_parser = reset_sub_parser.add_parser('volumes',
                                                       description=('''
    Reset all volumes to the state before using the deploy command.
                                                    '''),
                                                       formatter_class=argparse.RawDescriptionHelpFormatter,
                                                       usage='mdt [mdt_options] reset volumes')

    # Command get
    # ---------------
    get_sub_parser = get_parser.add_subparsers(dest='cmd_get')
    get_label_parser = get_sub_parser.add_parser('label', add_help=False)
    get_label_parser.set_defaults(func=redirect_to_cli('get label'))

    get_cfg_parser = get_sub_parser.add_parser('cfg', add_help=False)
    get_cfg_parser.set_defaults(func=redirect_to_cli('get cfg'))

    get_chart_parser = get_sub_parser.add_parser('chart', add_help=False)
    get_chart_parser.set_defaults(func=redirect_to_cli('get chart'))

    get_context_parser = get_sub_parser.add_parser('context', add_help=False)
    get_context_parser.set_defaults(func=redirect_to_cli('get context'))

    # Command load
    # -------------
    load_sub_parser = load_parser.add_subparsers(dest='cmd_load')
    load_cfg_parser = load_sub_parser.add_parser('cfg', add_help=False)
    load_cfg_parser.set_defaults(func=redirect_to_cli('load cfg'))

    # Command audit
    # -------------
    audit_sub_parser = audit_parser.add_subparsers(dest='cmd_audit')
    audit_kube_cluster_parser = audit_sub_parser.add_parser('kube-cluster',
                                                            description=('''
Perform an audit to the nodes that will be used in the kube-cluster.
You can run the audit for all kube-cluster nodes or for a single node with -n  parameter.
                                                  '''),
                                                            formatter_class=argparse.RawDescriptionHelpFormatter,
                                                            usage='mdt [mdt_options] audit kube-cluster [-n <node>]')

    audit_kube_cluster_parser.add_argument("-n", "--node",
                                           dest="node",
                                           default=None,
                                           help="Node to audit"
                                           )

    # US 45690 - LRO
    load_chart_parser = load_sub_parser.add_parser('chart', add_help=False)
    load_chart_parser.set_defaults(func=redirect_to_cli('load chart'))

    load_container_parser = load_sub_parser.add_parser('container',
                                                       description=('''
Load containers from CONTAINER_TGZ archive file or CONTAINER_DIR directory path.
The containers are pushed to the MDT Docker registry (default).
See NFS configuration to share registry with kube HA Docker registry.
                                                  '''),
                                                       formatter_class=argparse.RawDescriptionHelpFormatter,
                                                       usage='mdt [mdt_options] load container (-f CONTAINER_TGZ | -d CONTAINER_DIR)')

    load_rpm_parser = load_sub_parser.add_parser('rpm',
                                                 description=('''
Load rpm from RPM_TGZ archive file.
The rpm are pushed to the local yum repository.
                                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] load rpm (-f RPM_TGZ )')

    load_deb_parser = load_sub_parser.add_parser('deb',
                                                 description=('''
Load deb from DEB_TGZ archive file.
The deb are pushed to the local deb repository.
                                                  '''),
                                                 formatter_class=argparse.RawDescriptionHelpFormatter,
                                                 usage='mdt [mdt_options] load deb (-f DEB_TGZ )')

    # --> mdt load rpm -f <tar.gz>
    load_rpm_parser.add_argument("-f", "--file",
                                 dest="rpm_tgz",
                                 help="Path to rpm tgz archive to load ")

    # --> mdt load deb -f <tar.gz>
    load_deb_parser.add_argument("-f", "--file",
                                 dest="deb_tgz",
                                 help="Path to deb tgz archive to load ")

    # --> mdt load container [--force] -f <tar.gz> | -d <folder> [-r registry]
    load_container_exclusive_parser = load_container_parser.add_mutually_exclusive_group(
        required=True)
    load_container_exclusive_parser.add_argument("-f", "--file",
                                                 dest="container_tgz",
                                                 default=None,
                                                 help="Path of the container archive file to be loaded")
    load_container_exclusive_parser.add_argument("-d", "--directory",
                                                 dest="container_dir",
                                                 default=None,
                                                 help="Path of container directory to be loaded")
    # TODO when uncommnet don't forget to use this parameter in load_docker_registry function call
    # load_container_parser.add_argument("-r", "--registry",
    #                                    dest="registry",
    #                                    default=None,
    # help="Load containers in specific registry")

    # Command init
    # -------------
    init_parser.add_argument("--debug",
                             action="store_true",
                             help="Enable Debug output messages")
    init_parser.add_argument("-i ", "--ip-address",
                             dest="ip_address",
                             help="IP address of the MDT to use for cluster nodes ")
    init_parser.add_argument("-k ", "--ssh-key",
                             dest="ssh_key",
                             help="Full path to the SSH key to use for cluster node SSH connections")
    init_parser.add_argument("-g ", "--docker-registry",
                             dest="registry",
                             help="Only if external insecure Docker registry is used: <host>:<port> ")
    init_parser.add_argument("-m ", "--helm-repository",
                             dest="helm_repo_url",
                             help="Only if external Helm repository is used: <host>:<port> ")
    init_parser.add_argument("-d ", "--registry-nfs-address",
                             dest="registry_nfs_address",
                             help="NFS address for registry storage")
    init_parser.add_argument("-p ", "--registry-nfs-path",
                             dest="registry_nfs_path",
                             help="Remote NFS path "
                             "(with docker/registry/ path)")
    init_parser.add_argument("-u ", "--user",
                             dest="user",
                             help="Linux user to connect to the remote nodes, default is " + Config().get('ansible', 'remote_user'))
    init_parser.add_argument("-r ", "--repo",
                             dest="repo",
                             choices=["yum", "apt"],
                             help="Repository to use yum (rpm packages) or apt(deb packages), default yum")
    init_parser.add_argument("-a ", "--ansible-path",
                             dest="ansible_path",
                             help="Ansible yaml path, default " + Config().get('ansible', 'path'))
    init_parser.add_argument("-s ", "--stop",
                             action="store_true",
                             dest="stop",
                             help="Stop all containers launched by MDT bootstrap")
    init_parser.add_argument("--skip-load",
                             action="store_true",
                             dest="skip_load",
                             help="Skip load MDT and 3rd party containers to init MDT bootstrap")
    init_parser.add_argument("--http-port",
                             dest="http_port",
                             help="HTTP port to access MDT containers, default " + Config().get('global', 'http_port'))
    init_parser.add_argument("-c", "--container-3rdparty-path",
                             default=None,
                             dest="thirdparty_path",
                             help="Path of the 3rd party containers to load, "
                                  "default /opt/mfvp/deploypattern/resources/filename.tgz")

    # Command set
    # ----------------
    set_sub_parser = set_parser.add_subparsers(dest='cmd_set')
    set_node_parser = set_sub_parser.add_parser('node',
                                                description='set node on kube-matrix',
                                                usage='mdt [mdt_options] set node')
    set_node_parser.add_argument("-n", "--node",
                                 dest="set_node_value",
                                 help="node to set")
    set_node_parser.add_argument("-t", "--type",
                                 dest="set_node_value_type",
                                 help="type of node to set")

    # Command unset
    # ----------------
    unset_sub_parser = unset_parser.add_subparsers(dest='cmd_unset')
    unset_node_parser = unset_sub_parser.add_parser('node',
                                                    description='unset node on kube-matrix',
                                                    usage='mdt [mdt_options] unset node')
    unset_node_parser.add_argument("-n", "--node",
                                   dest="unset_node_value",
                                   help="node to unset")

    # Command add
    # ----------------
    add_sub_parser = add_val_parser.add_subparsers(dest='cmd_add')
    add_profile_parser = add_sub_parser.add_parser('profile', add_help=False)
    add_profile_parser.set_defaults(func=redirect_to_cli('add profile'))

    # Command delete
    # ----------------
    delete_sub_parser = delete_parser.add_subparsers(dest='cmd_delete')
    delete_profile_parser = delete_sub_parser.add_parser('profile', add_help=False)
    delete_profile_parser.set_defaults(func=redirect_to_cli('delete profile'))
    ## US 45690 - LRO
    delete_chart_parser = delete_sub_parser.add_parser('chart', add_help=False)
    delete_chart_parser.set_defaults(func=redirect_to_cli('delete chart'))

    # Command upgrade
    # ----------------
    upgrade_sub_parser = upgrade_parser.add_subparsers(dest='cmd_upgrade')
    upgrade_products_parser = upgrade_sub_parser.add_parser('products', add_help=False)
    upgrade_products_parser.set_defaults(func=redirect_to_cli('upgrade products'))

    # inventory_parser = subparsers.add_parser('inventory',
    #                                          help='Inventory management (MDT <1.5)')
    # inventory_parser.add_argument("--debug",
    #                               action="store_true",
    #                               help="Enable Debug output messages")
    # inventory_parser.add_argument("-l", "--list",
    #                               action="store_true",
    #                               help="List current inventory")
    # inventory_parser.add_argument("-f", "--format",
    #                               dest="format",
    #                               default="ansible",
    #                               help="list in format ansible/yaml")
    # inventory_parser.add_argument("-i", "--import",
    #                               dest="_import",
    #                               help="import inventory")
    # inventory_parser.add_argument("-e", "--export",
    #                               dest="export",
    #                               help="export inventory to ansible format")

    parsed_args, unknown_args = parser.parse_known_args()

    if hasattr(parsed_args, 'func'):
        parsed_args.func(parsed_args, unknown_args)

    if not sys.argv[1:]:
        parser.print_help()
        parser.exit(2)

    return parsed_args, unknown_args


def signal_handler(signal, frame):
    """ Catch signal """
    logging.info(':: exiting ...')
    sys.exit(0)


def __get_cfg():
    """
    # US 55043
    # Because deploy kube cluster is temporary done by front-end
    # get config bundle from database
    # and save it to predefined folder /opt/mfvp/deploypattern/bundle

    :return:
    """
    try:
        os.mkdir('/opt/mfvp/deploypattern/bundle', 0755)
    except Exception as error:
        pass
    mdtcli.call_mdt_cli('get', 'cfg', ['-o', '/opt/mfvp/deploypattern/bundle/bundle.tgz'])
    try:
        tar = tarfile.open('/opt/mfvp/deploypattern/bundle/bundle.tgz')
        tar.extractall('/opt/mfvp/deploypattern/bundle/')
        tar.close()
    except Exception as msg:
        LOGGER.error("Get bundle failed: %s", msg.message)
        return 1

def __clean_cfg():
    """

    :return:
    """
    try:
        shutil.rmtree('/opt/mfvp/deploypattern/bundle', ignore_errors=True)
    except Exception as msg:
        LOGGER.error("Clean bundle failed: %s", msg.message)
        return 1

def main():
    """ Main """

    # setup signal handler
    signal.signal(signal.SIGINT, signal_handler)
    # parse cmdline args
    args, unknown_args = parse_cmdline()

    if args.debug:
        formatter = logging.Formatter('%(levelname)s:'
                                      '%(filename)s:'
                                      '%(funcName)s:'
                                      ' %(message)s')
        CH_OUT.setFormatter(formatter)
        LOGGER.setLevel(logging.DEBUG)
        CH_OUT.setLevel(logging.DEBUG)
        F_H.setLevel(logging.DEBUG)

    # Command mdt init
    # ---------------------
    # init config
    if args.conf:
        config = Config(args.conf)
    else:
        config = Config(DFN_CONF_FILE)

    # Launch init process
    # Available only for MDT bootstrap
    if args.cmd == 'init':
        # TODO Check if MDT bootstrap
        try:
            mgnt_server_init(ipaddr=args.ip_address,
                             sshkey=args.ssh_key,
                             registry=args.registry,
                             registry_nfs_addr=args.registry_nfs_address,
                             registry_nfs_path=args.registry_nfs_path,
                             helm_repo=args.helm_repo_url,
                             user=args.user,
                             repo=args.repo,
                             ansible_path=args.ansible_path,
                             skip_load=args.skip_load,
                             stop=args.stop,
                             thirdparty_path=args.thirdparty_path,
                             http_port=args.http_port)
        except Exception as msg:
            LOGGER.error("Initialization failed: %s", msg.message, exc_info=True)
            return 1

    # check if init as been done before going on
    try:
        check_init(config)
    except Exception as msg:
        LOGGER.error("Init must be done before: %s", msg.message)
        return 1

    # Command mdt deploy
    # --------------------------
    if args.cmd == 'deploy':

        if args.cmd_deploy == 'kube-cluster':
            try:
                __get_cfg()
                bundle.Bundle().deploy_kubecluster()
                __clean_cfg()

            except Exception as msg:
                __clean_cfg()
                LOGGER.error("Deploying Kube cluster failed: %s", msg.message)
                return 1

        if args.cmd_deploy == "helm":
            try:
                inventory = bundle.Bundle().get_inventory()
                kube_vars_dict = bundle.Bundle().get_vars_kube_config()
                # Deploy Helm server (container tiller) only on
                #  one master with deployment (replicas=1)
                # Add service to expose Helm server
                result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="deploy_helm",
                    vars=config_lib.get_config_vars_dict(),
                    extra_vars=kube_vars_dict)
                LOGGER.debug("result : %s", result)
                if result != 0:
                    raise errors.AnsibleError(
                        "Ansible playbook failed : deploy_helm ")

            except Exception as msg:
                LOGGER.error("Install Helm failed: %s", msg.message)
                return 1

        if args.cmd_deploy == 'products':
            try:
                mdtcli.call_mdt_cli('deploy', 'products', unknown_args)
            except Exception as msg:
                LOGGER.error("Install products failed: %s", msg.message)
                return 1

        if args.cmd_deploy == "volumes":
            try:
                __get_cfg()
                bundle.Bundle().deploy_volumes()
                __clean_cfg()
            except Exception as msg:
                LOGGER.error("Install volumes failed: %s", msg.message)
                return 1

    # Command mdt reset
    # -------------------------
    if args.cmd == 'reset':
        if args.cmd_reset == 'products':
            try:
                # Reset all products
                mdtcli.call_mdt_cli('reset', 'products', unknown_args)
            except Exception as msg:
                LOGGER.error("Reset products failed: %s", msg.message)
                return 1

        if args.cmd_reset == 'kube-cluster':
            try:
                __get_cfg()
                bundle.Bundle().reset_kubecluster()
                __clean_cfg()
            except Exception as msg:
                __clean_cfg()
                LOGGER.error("Reset Kube cluster failed: %s", msg.message)
                return 1
            # Update mdt_api_products with Docker and Helm parameters after
            # Kube cluster reset
            # TODO: is it useful ?

        if args.cmd_reset == 'volumes':
            try:
                __get_cfg()
                bundle.Bundle().reset_volumes()
                __clean_cfg()
            except Exception as msg:
                LOGGER.error("Reset volumes failed: %s", msg.message)
                return 1

    # Command mdt load
    # -------------------------
    if args.cmd == 'load':
        if args.cmd_load == 'cfg':
            try:
                # load config bundle files:
                mdtcli.call_mdt_cli('load', 'cfg', unknown_args)
            except Exception as msg:
                LOGGER.error("Loading MDT bundle failed: %s", msg.message)
                return 1

        if args.cmd_load == 'chart':
            try:
                # load charts :
                mdtcli.call_mdt_cli('load', 'chart', unknown_args)
            except Exception as msg:
                LOGGER.error("Loading chart failed: %s", msg.message)
                return 1

        if args.cmd_load == 'container':
            try:
                load_docker_registry(
                    args.container_tgz, args.container_dir)
# args.container_tgz, args.container_dir, args.registry) TODO use when
# regitry will be used

            except Exception as msg:
                LOGGER.error("Loading Containers failed: %s", msg.message)
                return 1

        if args.cmd_load == 'rpm':
            try:
                load_yum_repo(args.rpm_tgz)

            except Exception as msg:
                LOGGER.error("Loading Rpm failed: %s", msg.message)
                return 1

        if args.cmd_load == 'deb':
            try:
                load_deb_repo(args.deb_tgz)

            except Exception as msg:
                LOGGER.error("Loading deb failed: %s", msg.message)
                return 1

    # Command mdt-cli get
    # ----------------------
    if args.cmd == 'get':
        if args.cmd_get == 'label':
            try:
                # get label
                mdtcli.call_mdt_cli('get', 'label', unknown_args)
            except Exception as msg:
                LOGGER.error("Retreive label failed: %s", msg.message)
                return 1

        if args.cmd_get == 'cfg':
            try:
                # get config bundle files:
                mdtcli.call_mdt_cli('get', 'cfg', unknown_args)
            except Exception as msg:
                LOGGER.error("Retreive bundle failed: %s", msg.message)
                return 1

        if args.cmd_get == 'chart':
            try:
                # get chart list:
                mdtcli.call_mdt_cli('get', 'chart', unknown_args)
            except Exception as msg:
                LOGGER.error("Retreive chart failed: %s", msg.message)
                return 1

        if args.cmd_get == 'context':
            try:
                # get chart list:
                mdtcli.call_mdt_cli('get', 'context', unknown_args)
            except Exception as msg:
                LOGGER.error("Retreive context failed: %s", msg.message)
                return 1

    # Command mdt set
    # -------------------------
    if args.cmd == 'set':
        if args.cmd_set == 'node':
            try:
                bundle.Bundle().set_node(args.set_node_value, args.set_node_value_type)
            except Exception as msg:
                LOGGER.error("Set node failed: %s", msg.message)
                return 1

    # Command mdt unset
    # -------------------------
    if args.cmd == 'unset':
        if args.cmd_unset == 'node':
            try:
                bundle.Bundle().unset_node(args.unset_node_value)
            except Exception as msg:
                LOGGER.error("Unset node failed: %s", msg.message)
                return 1

    # Command mdt audit
    # -------------------------
    if args.cmd == 'audit':
        if args.cmd_audit == 'kube-cluster':
            try:
                __get_cfg()
                bundle.Bundle().audit_kubecluster(args.node)
                __clean_cfg()
            except Exception as msg:
                LOGGER.error("Audit Kube-cluster failed: %s", msg.message)
                return 1

    # Command mdt add
    # -------------------------
    if args.cmd == 'add':
        if args.cmd_add == 'profile':
            try:
                mdtcli.call_mdt_cli('add', 'profile', unknown_args)
            except Exception as msg:
                LOGGER.error("Add profile failed: %s", msg.message)
                return 1

    # Command mdt delete
    # -------------------------
    if args.cmd == 'delete':
        if args.cmd_delete == 'profile':
            try:
                mdtcli.call_mdt_cli('delete', 'profile', unknown_args)
            except Exception as msg:
                LOGGER.error("Delete profile failed: %s", msg.message)
                return 1
        if args.cmd_delete == 'chart':
            try:
                mdtcli.call_mdt_cli('delete', 'chart', unknown_args)
            except Exception as msg:
                LOGGER.error("Delete chart failed: %s", msg.message)
                return 1

    # Command mdt upgrade
    # -------------------------
    if args.cmd == 'upgrade':
        if args.cmd_upgrade == 'products':
            try:
                mdtcli.call_mdt_cli('upgrade', 'products', unknown_args)
            except Exception as msg:
                LOGGER.error("Upgrade products failed: %s", msg.message)
                return 1

    if unknown_args:
        LOGGER.error("Unknown arguments : %s", unknown_args)
        exit(2)
    # LOGGER.info("\n")
    return 0


if __name__ == '__main__':
    main()
