"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""

from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import os
import sys
import logging
import requests
from collections import namedtuple

from ansible.parsing.dataloader import DataLoader
from ansible.vars.manager import VariableManager
from ansible.inventory.manager import InventoryManager
from ansible.inventory.group import Group
from ansible.inventory.host import Host
from ansible.executor.playbook_executor import PlaybookExecutor
from ansible.playbook.play import Play
from ansible.executor.task_queue_manager import TaskQueueManager
from ansible.plugins.callback import CallbackBase
from ansible.utils.display import Display
from ansible import constants as C
from ericsson.deploypattern.lib.ansible_callback import EventsCallback

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.lib.config import Config, get_config_vars_dict
from ericsson.deploypattern.lib.utils import load_yaml, save_yaml
from ericsson.deploypattern.lib.inventory import build_inventory
from ericsson.deploypattern.lib.inventory import save_yaml_inventory
from ericsson.deploypattern.lib.inventory import get_saved_node_list
LOGGER = logging.getLogger(__name__)


class AnsiblePlay(object):
    """
    Class AnsiblePlay
    """

    def __init__(self,
                 host_list=None,
                 extra_vars=None,
                 options=None,
                 output=False):

        # initialize needed objects
        self.loader = DataLoader()
        self.passwords = {}

        self.display = Display()
        self.options = options

        if hasattr(self.options, 'verbosity'):
            self.display.verbosity = self.options.verbosity
        else:
            self.display.verbosity = 0

        # create inventory and pass to var manager
        # Inventory is just a list of IP addresses or list of dictionaries
        # with groups, subgroups and nodes (name, ip address)
        self.inventory = None

        if host_list is None:
            host_list = get_saved_node_list()
            if host_list is None:
                LOGGER.error("No inventory found")
                raise errors.AnsibleError("No inventory found")

        if isinstance(host_list, list):
            for elt in host_list:
                if isinstance(elt, dict):
                    self.inventory = InventoryManager(loader=self.loader,
                                                      # variable_manager=self.variable_manager,
                                                      # host_list=None)
                                                      sources=None)
                    build_inventory(self.inventory, host_list)
                    save_yaml_inventory(host_list)
                    break

        # Case host_list is not None or list, then it is path ?
        if self.inventory is None:
            self.inventory = InventoryManager(loader=self.loader,
                                              # variable_manager=self.variable_manager,
                                              # host_list=host_list)
                                              sources=host_list)
        self.variable_manager = VariableManager(
            loader=self.loader, inventory=self.inventory)

        if extra_vars is not None:
            self.variable_manager.extra_vars = extra_vars

        # if LOGGER.getEffectiveLevel() != 10 or output:
        #     self.results_callback = OnFailedCB()
        # else:
        self.results_callback = "default"
        #
        # if output:
        #     self.results_callback = OnOutput()

    def play(self, play_source=None):
        """
        Method play of class AnsiblePlay
        It is used to play task, roles or playbook
        """

        tqm = None
        try:
            play = Play().load(play_source,
                               variable_manager=self.variable_manager,
                               loader=self.loader)
            # actually run it
            tqm = TaskQueueManager(
                inventory=self.inventory,
                variable_manager=self.variable_manager,
                loader=self.loader,
                options=self.options,
                passwords=self.passwords,
                stdout_callback=self.results_callback,
            )
            result = tqm.run(play)
        except Exception as e:
            #            LOGGER.error('ansible play failed')
            raise
        finally:
            if tqm is not None:
                tqm.cleanup()

        return result


class AnsibleRunner(object):
    """
    Class AnsibleRunner
    """

    def __init__(self, connection='ssh', module_path=None, become=False,
                 become_method=None, become_user=None, check=None,
                 extra_vars=None, private_key_file=None, remote_user=None,
                 verbosity=5, listtasks=None, listtags=None, listhosts=None,
                 syntax=None, tags=[], skip_tags=[], diff=False):

        Options = namedtuple('Options',
                             ['connection', 'forks', 'module_path', 'become',
                              'become_method', 'become_user', 'become_ask_pass',
                              'become_prompt', 'check',
                              'extra_vars', 'private_key_file', 'remote_user',
                              'verbosity', 'listhosts', 'listtasks', 'listtags',
                              'syntax', 'ssh_extra_args', 'tags', 'skip_tags', 'diff'])

        try:
            if private_key_file is None:
                private_key_file = Config().get('ansible', 'ssh_key')
            if remote_user is None:
                remote_user = Config().get('ansible', 'remote_user')
        except Exception as e:
            LOGGER.error("get ansible config failed")
            raise

        # force no check TODO add it as option
        ssh_extra_args = '-o StrictHostKeyChecking=no'

        # if remote_user != 'root':
        become = True
        become_method = 'sudo'
        become_user = 'root'
        become_ask_pass = False

        self.options = Options(
            connection=connection,
            forks=100,
            private_key_file=private_key_file,
            remote_user=remote_user,
            module_path=module_path,
            become=become,
            become_method=become_method,
            become_user=become_user,
            become_ask_pass=become_ask_pass,
            become_prompt=False,
            check=check,
            extra_vars=extra_vars,
            verbosity=verbosity,
            listhosts=listhosts,
            listtasks=listtasks,
            listtags=listtags,
            syntax=syntax,
            ssh_extra_args=ssh_extra_args,
            tags=tags,
            skip_tags=skip_tags,
            diff=diff
        )

    def add_tag(self, tag):
        """
        Add tag to Ansible Runner
        :param tag:
        :return:
        """
        LOGGER.debug("add Ansible tag")
        tags_list = getattr(self.options, 'tags')
        tags_list.append(tag)
        self.options = self.options._replace(tags=tags_list)

    def add_skip_tag(self, tag):
        """
        Add skip tag to Ansible Runner
        :param tag:
        :return:
        """
        LOGGER.debug("add Ansible sqkip_tag")
        tags_list = getattr(self.options, 'skip_tags')
        tags_list.append(tag)
        self.options = self.options._replace(skip_tags=tags_list)

    def playtask(self, task=None, args=None, name=None, host_list=None,
                 group=None, vars=None, extra_vars=None, output=False):
        """
        Execute a task
        """
        self._add_all_vars(extra_vars)
        play = AnsiblePlay(host_list=host_list,
                           extra_vars=extra_vars,
                           options=self.options,
                           output=output)
        if group:
            hosts = group
        else:
            hosts = host_list

        play_source = dict(
            name=name,
            hosts=hosts,
            gather_facts='yes',
            vars=vars,
            tasks=[dict(action=dict(module=task, args=args))])

        return play.play(play_source=play_source)

    def playroles(self, roles=None, name=None, host_list=None, group=None,
                  vars=None, extra_vars=None):
        """
        Execute a Role
        """
        self._add_all_vars(extra_vars)
        play = AnsiblePlay(host_list=host_list,
                           extra_vars=extra_vars,
                           options=self.options)

        if group:
            hosts = group
        else:
            hosts = host_list

        play_source = dict(
            name=name,
            hosts=hosts,
            gather_facts='yes',
            vars=vars,
            roles=[dict(name=role) for role in roles]
        )
        return play.play(play_source=play_source)

    def playbook(self, playbooks=None, host_list=None, extra_vars=None,
                 vars=None, manage_callback=False,
                 next_status="deploymentInProgress", nodes=[]):
        """
        Execute a playbook
        """
        self._add_all_vars(extra_vars)

        try:
            play = AnsiblePlay(host_list=host_list,
                               extra_vars=extra_vars,
                               options=self.options)
            if vars != None:
                play.variable_manager.options_vars = vars

            pbex = PlaybookExecutor(playbooks=[construct_playbook(playbooks)],
                                    inventory=play.inventory,
                                    variable_manager=play.variable_manager,
                                    loader=play.loader,
                                    options=self.options,
                                    passwords=play.passwords)

            if manage_callback:
                pbex._tqm._stdout_callback = EventsCallback(next_status, nodes)
            else:
                pbex._tqm._stdout_callback = play.results_callback

            # Clear pattern cache for inventory to update "all" pattern
            pbex._inventory.clear_pattern_cache()

            results = pbex.run()
        except errors.FileNotFound as e:
            LOGGER.error('Playbook not found %s: ', playbooks)
            raise
        except Exception as e:
            LOGGER.error('ansible playbook run failed')
            raise
        return results

    def _add_all_vars(self, varsdict):
        """
        Add all variables without overwrite existing variables in varsdict
        all means default variables
        varsdict means user variable (not to overwrite)
        :param varsdict:
        :return:
        """
        import json
        if varsdict is None:
            return
        ansible_path = Config().get('ansible', 'path')
        _ff = os.path.join(ansible_path, 'group_vars', 'all')
        allvars = {}
        for ext in ['.yml', '.yaml']:
            _f = _ff + ext
            if os.path.exists(_f):
                allvars = load_yaml(_f)
                LOGGER.debug(json.dumps(allvars, indent=4))
                for key, item in allvars.iteritems():
                    if key not in varsdict.keys():
                        varsdict[key] = item
                    # Don't overwrite current variable but save it in all.yaml
                    else:
                        allvars[key] = varsdict[key]
                save_yaml(_f, allvars, style='"')
                break

    def file_task(self, nodes, cmd):
        """
        Method file_task
        """
        try:
            LOGGER.info("file task  %s", cmd)

            result = self.playtask(
                host_list=nodes,
                task="file",
                args=cmd
            )
            if result != 0:
                msg = "Ansible failed task=file args=%s nodes=%s" % \
                      (cmd, nodes)
                raise errors.AnsibleError(msg)
        except Exception as e:
            raise

        return result

    def remote_command(self, nodes, cmd, ignore_errors=False, output=False):
        """
        Method remote_command
        """
        try:
            LOGGER.info("exec command=%s", cmd)
            extra_vars = {'ignore_errors': ignore_errors}
            result = self.playtask(
                host_list=nodes,
                task="command",
                args=cmd,
                extra_vars=extra_vars,
                output=output
            )
            if result != 0:
                msg = "Ansible failed task=command args=%s nodes=%s" % \
                      (cmd, nodes)
                raise errors.AnsibleError(msg)
        except Exception as e:
            raise
        return result

    def copy_file(self, nodes, file, dest, task='copy'):
        """
        Copy compose files on remote nodes list
        """
        if not os.path.exists(file):
            raise errors.FileNotFound('%s: ' % file)

        try:
            LOGGER.info("copy file=%s dest=%s", file, dest)
            result = self.playtask(
                host_list=nodes,
                task=task,
                args={'src': file, 'dest': dest},
                extra_vars=get_config_vars_dict())
            if result != 0:
                msg = "Ansible failed task=%s file=%s dest=%s, nodes=%s" % \
                      (task, file, dest, nodes)
                raise errors.AnsibleError(msg)
        except Exception as e:
            raise

        return result

    def create_dir(self, nodes, directory):
        """
        Create directory on remote nodes list
        """
        mk_cmd = 'path=' + directory + ' ' + 'state=directory'
        self.file_task(nodes, mk_cmd)


def run_playbook(playbook_name):
    """
    Launch playbook on local host
    """
    # TODO: use playrole instead
    # check if localhost and add var :
    # ansible_python_interpreter: "/usr/bin/python"
    #~ playbook = construct_playbook(playbook_name)
    extra_vars = get_config_vars_dict()
    result = AnsibleRunner(). \
        playbook(host_list=[],
                 playbooks=playbook_name,
                 extra_vars=get_config_vars_dict())
    return result


def get_ansible_path():
    """
    Retrieve path where ansible files can be found
    """
    pb_path = None
    if Config().has_option('ansible', 'path'):
        pb_path = Config().get('ansible', 'path')

    if not os.path.exists(pb_path):
        pb_path = os.path.join(os.path.dirname(sys.argv[0]), 'ansible')

    if not os.path.exists(pb_path):
        LOGGER.error("Ansible path Not Found : %s ", pb_path)
        raise errors.FileNotFound('%s ' % pb_path)

    return str(pb_path)


def construct_playbook(name):
    """
    Add extension .yml to playbook name
    """
    pb_path = get_ansible_path()
    if os.path.splitext(name)[1] == '':
        pb_name = name + '.yaml'
        playbook = os.path.join(pb_path, pb_name)
        if not os.path.exists(playbook):
            pb_name = name + '.yml'
            playbook = os.path.join(pb_path, pb_name)
            if not os.path.exists(playbook):
                raise errors.FileNotFound('%s: ' % playbook)
    else:
        pb_name = name
        playbook = os.path.join(pb_path, pb_name)
        if not os.path.exists(playbook):
            raise errors.FileNotFound('%s: ' % playbook)

    return playbook
