"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

try:
    from configparser import ConfigParser
except ImportError:
    from ConfigParser import ConfigParser

import os
import sys
import logging

import ericsson.deploypattern.lib.errors as errors

LOGGER = logging.getLogger(__name__)

DEFAULT = {
    'global': {
        'mgnt_ip_addr': '127.0.0.1',
        'path': '',
        'http_port': '80'
    },
    'ansible': {
        'ssh_key': "",
        'remote_user': "centos",
        'path': ''
    },

    'docker': {
        'registry_address': '',
        'registry_port': '5001',
        'registry_sub': '',
        'registry_version': '2.6.2dck1',
        'registry_nfs_ip': '',
        'registry_nfs_path': '/var/lib/docker/registry',
        'bridge': '172.21.0.1'
    },

    'repo': {
        'path': '/var/www/ericsson',
        'name': 'mdt_repo',
        'type': 'first'
    },

    'helm_repo': {
        'address': 'http://chartmuseum:8080',
        'path': '/var/www/ericsson',
        'name': 'helm_repo'
    },

    'compose': {
        'path': ''
    },
    'bundle': {
        'path': ''
    }
}


class Config(object):
    """ Class to handle config from file

    If file is not pass in argument the default value
    is take
    """
    _state = {}
    _conf = None

    def __new__(cls, *args, **kwargs):
        self = super(Config, cls).__new__(cls)
        self.__dict__ = cls._state
        return self

    def __init__(self, conf=None):
        if self._conf is None:
            self.parser = ConfigParser()
            self.parser.read_dict(DEFAULT)
            if conf:
                self._conf = conf
                self.parser.read(conf)

        # assume global path for ansible, compose etc...
        # located where the binaire is if not set
        if not self.get('global', 'path'):
            pkg_path = self.__retrive_pkg_path()
            self.set('global', 'path', pkg_path)
        else:
            pkg_path = self.get('global', 'path')

        if not self.get('compose', 'path'):
            compose_path = os.path.join(pkg_path, 'compose')
            self.set('compose', 'path', compose_path)

        if not self.get('ansible', 'path'):
            ansible_path = os.path.join(pkg_path, 'ansible')
            self.set('ansible', 'path', ansible_path)

        if not self.get('bundle', 'path'):
            bundle_path = os.path.join(pkg_path, 'bundle')
            self.set('bundle', 'path', bundle_path)

    def __getattr__(self, attr):
        if hasattr(self.parser, attr):
            def wrapper(*args, **kw):
                return getattr(self.parser, attr)(*args, **kw)
            return wrapper
        raise AttributeError(attr)

    def get_file_conf(self):
        """

        :param:
        :return:
        """
        return self._conf

    def get_default_conf(self):
        return DEFAULT

    def set_registry(self, registry):
        """
        Set insecure Docker registry yo use instead to the default Docker
        MDT registry.
        :param registry: (IP_ADDRESS | HOSTNAME)[:PORT]
        :return:
        """
        if registry == "local":
            addr = self.get('global', 'mgnt_ip_addr')
            port = DEFAULT.get('docker').get('registry_port')
            subrepo = ""
        elif ':' in registry:
            if '/' in registry:
                full_addr, subrepo = registry.split('/', 1)
                addr, port = full_addr.split(':', 1)
            else:
                addr, port = registry.split(':', 1)
                subrepo = ""
        else:
            if '/' in registry:
                addr, subrepo = registry.split('/', 1)
            else:
                addr = registry
                subrepo = ""
            port = "80"
        self.set('docker', 'registry_address', addr)
        self.set('docker', 'registry_port', port)
        self.set('docker', 'registry_sub', subrepo)

    @staticmethod
    def __retrive_pkg_path():
        """
        get the path of deploypattern package
        :param:
        :return:
        """
        realpath = os.path.realpath(sys.argv[0])
        bindir = os.path.dirname(realpath)
        pkgdir = os.path.dirname(bindir)
        return pkgdir


def get_config_vars_dict():
    """
    get the config var as dict
    :param:
    :return:
    """
    dict_vars = {
        'mgnt_ip_addr': Config().get('global', 'mgnt_ip_addr'),
        'path': Config().get('global', 'path'),
        'http_port': Config().get('global', 'http_port'),

        'repo_path': Config().get('repo', 'path'),
        'repo_name': Config().get('repo', 'name'),
        'repo_type': Config().get('repo', 'type'),

        'bundle_path': Config().get('bundle', 'path'),

        'helm_repo_address': Config().get('helm_repo', 'address'),
        'helm_repo_path': Config().get('helm_repo', 'path'),
        'helm_repo_name': Config().get('helm_repo', 'name'),

        'docker_bridge': Config().get('docker', 'bridge'),
        'docker_bridge_cidr': Config().get('docker', 'bridge')[:-1] + '0/16',
        'registry_nfs_ip': Config().get('docker', 'registry_nfs_ip'),
        'registry_nfs_path': Config().get('docker', 'registry_nfs_path'),
        'registry_port': Config().get('docker', 'registry_port'),
        'docker_registry': '',

        'ansible_path': Config().get('ansible', 'path'),
        'role_path': os.path.join(Config().get('ansible', 'path'), 'roles'),
        'compose_path': Config().get('compose', 'path'),
        'ansible_python_interpreter': "/usr/bin/python"

    }

    if Config().get('docker', 'registry_sub') == '':
        dict_vars['docker_registry'] = \
            Config().get('docker', 'registry_address') + ':' \
                            + Config().get('docker', 'registry_port')
    else:
        dict_vars['docker_registry'] = \
            Config().get('docker', 'registry_address') + ':' \
                            + Config().get('docker', 'registry_port') + '/' \
                            + Config().get('docker', 'registry_sub')

    return dict_vars


def get_ssh_key_file():
    """

    :param:
    :return:
    """
    return Config().get('ansible', 'ssh_key')


def get_remote_user():
    """

    :param:
    :return:
    """
    return Config().get('ansible', 'remote_user')


def get_helm_charts_server():
    """

    :param:
    :return:
    """
    return Config().get('helm_repo', 'path')


def get_helm_charts_reponame():
    """

    :param:
    :return:
    """
    return Config().get('helm_repo', 'name')


def get_mdt_node():
    """

    :param:
    :return:
    """
    return Config().get('global', 'mgnt_ip_addr')

def get_docker_node():
    """

    :param:
    :return:
    """
    return Config().get('docker', 'registry_address')


def get_http_port():
    """

    :param:
    :return:
    """
    return Config().get('global', 'http_port')


def get_tiller_server():
    """

    :param:
    :return:
    """
    return Config().get('tiller', 'server')


def get_tiller_port():
    """

    :param:
    :return:
    """
    return Config().get('tiller', 'port')


def get_compose_path():
    """

    :param:
    :return:
    """
    return Config().get('compose', 'path')


def get_docker_registry_url():
    """

    :param:
    :return:
    """
    if Config().get('docker', 'registry_sub') == '':
        return Config().get('docker', 'registry_address') + ':' + \
               Config().get('docker', 'registry_port')
    return Config().get('docker', 'registry_address') + ':' + \
           Config().get('docker', 'registry_port') + '/' + \
           Config().get('docker', 'registry_sub')

def get_repo_path():
    """

    :param:
    :return:
    """
    return Config().get('repo', 'path')

def get_repo_name():
    """

    :param:
    :return:
    """
    return Config().get('repo', 'name')
