"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import logging
import gzip
import re
import tarfile
import json
import requests
import subprocess
import re
import ericsson.deploypattern.lib.errors as errors

from docker.errors import ImageNotFound, APIError

from docker import DockerClient, APIClient
from ericsson.deploypattern.lib.config import Config

LOGGER = logging.getLogger(__name__)


class DockerLib(object):
    """ Docker client class wrapper """

    def __init__(self, host=None, port=None):
        """ Connect to the docker daemon """

        # To access remote docker daemon via TCP -> more difficult
        # Our Docker installation via Ansible doesn't configure TCP option
        # if Config().get('docker', 'registry_address') == \
        #        Config().get('global', 'mgnt_ip_addr'):
        #    base_url = 'unix://var/run/docker.sock'
        # else:
        #    base_url = "http://%s:%s" % \
        #               (Config().get('docker', 'registry_address'), \
        #                Config().get('docker', 'registry_port'))

        # Access to server daemon docker local
        base_url = 'unix://var/run/docker.sock'
        LOGGER.debug("docker base_url: %s ", base_url)

        try:
            # TODO: use only DockerClient and not low level API
            self.docker = DockerClient(base_url=base_url, timeout=240)
            self.docker_api = APIClient(base_url=base_url, timeout=240)
        except Exception as msg:
            LOGGER.error('No docker daemon running ? ')
            raise

        try:
            self.registry_address = Config().get('docker', 'registry_address')
            self.registry_port = Config().get('docker', 'registry_port')
        except Exception as msg:
            LOGGER.error('Config value not found')
            raise

        self.registry = self.registry_address + ":" + self.registry_port

    def clean_image(self, name, tag):
        """
        Clean an image and all its tags loaded locally,
        when we push a container on Docker registry (by instance)
        :param name:
        :param tag:
        :return:
        """
        img = self.docker.images.get(name + ':' + tag)
        # make sure the clean will not delete more tags than necessary
        regex = re.compile(r".*:%s|.*:latest" % re.escape(tag))
        for __tag in img.tags:
            try:
                match = regex.match(__tag)
                if match.group(0) == __tag:
                    self.docker.images.remove(__tag, force=True)
            except AttributeError:
                # ignore error when regex is not ok
                pass

    def load_tgz_image(self, image, force=False):
        """
        Load locally image(s) from tar.gz file
        if force, load image(s) in all cases
        if not force, load image if it is not already push on the Docker registry
        docker-py bug, cannot handle large file... Use docker load cli instead

        :param image: path file of tar.gz
        :param force: boolean (default False)
        :return: list of loaded images [ "name:tag" ..]
        """
        try:
            LOGGER.info("  -> loading image: %s ", image)
            if os.path.splitext(image)[-1] not in ['.gz', '.tar', '.tgz']:
                LOGGER.error('Unknown extension %s... exiting', os.path.splitext(image)[-1])
                return []

            # Search all tags (image:version) in manifest file
            tar = tarfile.open(image)
            manifest = json.load(tar.extractfile("manifest.json"))
            tar.close()
            tags = [manifest[x]["RepoTags"][0] for x in range(len(manifest))]
            # Force load image(s) in local
            if force:
                subprocess.call(['sudo', 'docker', 'load', '-i', '{}'.format(image)])
                return tags
            # For each image to load, search tags in Docker registry
            # if there is at least one image not pushed, load image
            load = False
            for tag in tags:
                try:
                    versions = requests.get("http://" + self.registry + "/v2/" + \
                                            tag.split(":")[0] + "/tags/list")
                    if tag.split(":")[1] not in json.loads(versions.text)["tags"]:
                        load = True
                        break
                except:
                    load = True
                    break

            # Case: at least one image is not already push or version not in the list
            if load:
                subprocess.call(['sudo', 'docker', 'load', '-i', '{}'.format(image)])
                return tags
            # Case
            LOGGER.info("  -> Not loading image: %s, because already load ", image)
            return []
        # Case Incorrect registry URL
        except requests.exceptions.ConnectionError as msg:
            raise errors.LoadError("ERROR: Check if Docker registry is up")
        except Exception as msg:
            LOGGER.error("loading image %s failed ", image)
            raise

    def push_image(self, name, tag):
        """
        Push an image (name:tag) on the current Docker registry
        The image is previously loaded on local, it means that the Docker
        registry is up and the image not already push on it
        :param name:
        :param tag: version of the image
        :return:
        """
        repository = os.path.join(self.registry, name)
        try:
            LOGGER.debug("get image %s:%s", name, tag)
            img = self.docker.images.get(name + ':' + tag)
            # Tag to push
            LOGGER.debug(" - tag image: %s:%s and latest", repository, tag)
            img.tag(repository=repository, tag=tag)
            img.tag(repository=repository, tag="latest")
            # Push
            cr = self.docker.images.push(repository,
                                         tag=tag,
                                         insecure_registry=True)
            if not 'error' in cr:
                LOGGER.info("  -> pushing image: %s:%s %s \n", repository, tag, img.short_id)
                self.docker.images.push(repository, tag="latest", insecure_registry=True)
            else:
                # Clean local images
                self.clean_image(name, tag)
                raise errors.LoadError("ERROR pushing image %s:%s %s. Check if the Docker registry is up" \
                                       % (repository, tag, img.short_id))

        except ImageNotFound:
            LOGGER.error('image -%s:%s- not found', name, tag)
            raise
        except Exception as msg:
            LOGGER.error("pushing image %s:%s failed ", name, tag)
            raise

        try:
            self.clean_image(name, tag)
        except Exception as msg:
            LOGGER.error("removing tmp image %s:%s failed ", name, tag)
            pass

    def delete_image(self, name, version):
        """
        Remove image from the Docker registry
        :param name:
        :param version:
        :return:
        """
        pass

    def get_image_id(self, image):
        """
        Get id of the image from its tag
        :param image: tag of the image -> name:version or repository/name:version
        :return:
        """
        try:
            img = self.docker.images.get(image)
        except Exception as msg:
            LOGGER.error("getting image %s failed ", image)
            raise
        return img.id

    def is_running_from_image(self, image):
        """ Check if a container is running from image """
        return self.docker_api.containers(filters={'ancestor': image})

    def is_running(self, name):
        """ Check if container name is running  """
        return self.docker_api.containers(filters={'name': name})

    def rm_container_if_running(self, name):
        """

        :param name:
        :return:
        """
        cont_list = self.docker.containers.list(all=True)
        for container in cont_list:
            LOGGER.debug("Container: %s - status: %s", container.name, container.status)
            if container.name == name:
                try:
                    # stop it first if running
                    if container.status == r'running':
                        LOGGER.debug("stopping: %s ", container.name)
                        container.stop()

                    container = self.docker.containers.get(container.id)

                    # remove it finally
                    LOGGER.debug("removing: %s - %s", container.name, container.status)
                    container.remove(force=True)

                except Exception as msg:
                    LOGGER.debug("Error on removing container: %s ", container.name)
                    # do not raise error: bypass error in devmapper loop
                    # docker filesystem backend
                    pass
                break

    @staticmethod
    def get_name_tag_from_archive(image):
        """ Get name and tag from a tar.gz image file """
        LOGGER.debug("image archive: %s ", image)

        version_regexp = [r"(v\d+\.\d+\.\d+-\d+\D{3}\d+)",
                          r"(\d+\.\d+\.\d+-\d+\D{3}\d+)",
                          r"(\d+\.\d+\.\d+\D{3}\d+)",
                          r"(v\d+\.\d+\.\d+-beta\.\d+)",
                          r"(v\d+\.\d+\.\d+-amd64)",
                          r"(\d+\.\d+\.\d+-kubeadm)",
                          r"(\d+\.\d+\.\d+-management-alpine)",
                          r"(\d+\.\d+\.\d+-management)",
                          r"(\d+\.\d+\-alpine)",
                          r"(v\d+\.\d+\.\d+-\d+)",
                          r"(\d+\.\d+\.\d+-\d+)",
                          r"(v\d+\.\d+\.\d+)",
                          r"(\d+\.\d+\.\d+)",
                          r"(\d+\.\d+)"]

        __found = False
        for __re in version_regexp:
            if re.search(__re, os.path.basename(image)):
                img = re.split(__re,
                               os.path.basename(image))
                LOGGER.debug("image found: %s:%s ", img[1], img[0])
                __found = True
                break

        if __found:
            img_tag = img[1]
            img_name = img[0]
        else:
            raise errors.ImageNameError("could not get name:tag for image: %s"
                                        % image)

        # remove '-' character between name and version
        if img[0][-1:] == '-':
            img_name = img[0][:-1]

        return img_name, img_tag
