"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""

from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import os
import sys
import logging
import re

from ansible.parsing.dataloader import DataLoader
from ansible.inventory.manager import InventoryManager
from ansible.inventory.host import Host

from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.config import get_ssh_key_file, get_remote_user
from ericsson.deploypattern.lib.utils import save_yaml, load_yaml

import ericsson.deploypattern.lib.errors as errors

LOGGER = logging.getLogger(__name__)

SAVE_INVENTORY_FILE = "cluster_inventory.yaml"


def inventory_command(cmd):
    """

    :param cmd:
    :return:
    """
    if cmd.list:
        list_inventory(cmd.format)
    if cmd._import:
        save_inventory(cmd._import)
    if cmd.export:
        export_inventory(cmd.export)


def build_inventory(inventory=None, node_list=None):
    """
    Build or Update Inventory object from node list with group
    nodes:
       - group1:
           - name: host1
             access_ip: addr1
           - name: host2
             access_ip: addr2
       - group2:
           - name: host3
             access_ip: addr3
       - group3:
           - children: group1
           - children: group2
       - name: host4
         access_ip: addr4
       - name: host5
         access_ip: addr5

    Correction with Ansible >= 2.5

    US 53470 - LRO
    Correction: inventory doesn't take any inventory variables !!
    For Kubespray, we add access_ip variable, used when there is no ansible_default_ip variable.
    we add to localhost, to have also access_ip variable
    :param node_list:
    :return: Inventory
    """
    re_ip = re.compile('\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$')
    if inventory is None:
        inventory = InventoryManager(loader=DataLoader(), sources=None)

    host_list = []
    for node in node_list:
        # Case node outside any group
        # Create Host add it to host list
        # Add host only wit its name
        if 'name' in node and 'access_ip' in node:
            name, access_ip = get_ip_and_name(node)
            host_list.append({'name': name, 'access_ip': access_ip})
        # Case group
        # with eventually children group
        else:
            for group, nodegrouplist in node.iteritems():
                # With Ansible >= 2.5 add group with group name, no more
                # group object
                inventory.add_group(group)
                inventory._inventory.add_child('all', group)
                # Add Host or child to this group
                for node_group in nodegrouplist:
                    # Case sub group
                    if 'children' in node_group:
                        child = node_group['children']
                        if not isinstance(child, str):
                            raise errors.PatternError(
                                "bad structure in pattern: "
                                "bad value for key \'children\' in group %s" \
                                % group)
                        inventory._inventory.add_child(group, child)
                    # Case node to add to group
                    # Create Host, add to inventory and as child to group
                    else:
                        name, address = get_ip_and_name(node_group)
                        inventory._inventory.add_host(name)
                        inventory._inventory.add_child(group, name)
                        if re_ip.match(address):
                            inventory._inventory.set_variable(name, 'access_ip', address)
                            inventory._inventory.set_variable(name, 'ip', address)

    # Add nodes without group, add them to 2 groups
    # all and also ungrouped (default) groups
    for host in host_list:
        inventory._inventory.add_host(host['name'], 'all')
        inventory._inventory.add_host(host['name'], 'ungrouped')
        if re_ip.match(host['access_ip']):
            inventory._inventory.set_variable(host['name'], 'access_ip', host['access_ip'])
            inventory._inventory.set_variable(host['name'], 'ip', host['access_ip'])

    # Add localhost with its IP address
    inventory._inventory.add_host('localhost')
    inventory._inventory.set_variable('localhost', 'access_ip', '127.0.0.1')
    inventory._inventory.set_variable('localhost', 'ip', '127.0.0.1')
    return inventory


def get_ip_and_name(host):
    """

    :param host:
    :return:
    """
    _name = None
    _address = None
    try:
        for key, value in host.iteritems():
            if key == 'name':
                if not isinstance(value, str):
                    raise errors.PatternError(
                        "bad structure in pattern: "
                        "bad value for key \'name\' in group")
                _name = value
            elif key == 'access_ip':
                if not isinstance(value, str):
                    raise errors.PatternError(
                        "bad structure in pattern: "
                        "bad value for key \'access_ip\' in group")
                _address = value
    except Exception as err:
        raise errors.PatternError("bad structure in pattern: %s" % err.message)
    return _name, _address


def create_new_host(address, name):
    """

    :param address:
    :param name:
    :return:
    """
    LOGGER.debug(":: Creating Host: %s:%s", name, address)
    new_host = Host(name=name)
    new_host.address = address
    new_host.set_variable('ansible_host', address)

    private_key_file = get_ssh_key_file()
    if private_key_file:
        new_host.set_variable('ansible_ssh_private_key_file',
                              private_key_file)
    remote_user = get_remote_user()
    if remote_user:
        new_host.set_variable('ansible_user',
                              remote_user)
    return new_host




def list_inventory(format=None):
    """

    :param format:
    :return:
    """
    if format == 'ansible':
        list_inventory_ansible_format()
    elif format == 'yaml':
        list_inventory_yaml_format()
    else:
        sys.stderr.write("\n >>> Unknown format: %s" % format)


def list_inventory_ansible_format():
    """

    :return:
    """
    node_list = get_saved_node_list()
    if node_list is not None:
        inventory_obj = build_inventory(node_list=node_list)
        export_inventory_obj(inventory_obj, sys.stdout)
        sys.stderr.write("\n")
    else:
        sys.stderr.write("\n >>> No inventory saved found !\n\n")


def list_inventory_yaml_format():
    """

    :param cmd:
    :return:
    """
    node_list_file = get_saved_inventory_path()
    if os.path.exists(node_list_file):
        with open(node_list_file) as fd:
            data = fd.read()
            sys.stdout.write(data)
    else:
        sys.stderr.write("\n >>> No inventory saved found !\n\n")


def export_inventory(_file):
    """

    :param _file:
    :return:
    """
    node_list = get_saved_node_list()
    if node_list is not None:
        inventory_obj = build_inventory(node_list=node_list)
        with open(_file, 'w+') as fd:
            export_inventory_obj(inventory_obj, fd)
        fd.close()
    else:
        sys.stderr.write("\n >>> No inventory saved found !\n\n")


def save_inventory(_file):
    """

    :param _file:
    :return:
    """
    node_list = load_yaml(_file)
    save_yaml_inventory(node_list)


def get_saved_node_list():
    """

    :return:
    """
    data = None
    inventory_path = get_saved_inventory_path()
    if os.path.exists(inventory_path):
        data = load_yaml(inventory_path)
    return data


def get_saved_inventory_path():
    """

    :return:
    """
    path = Config().get('global', 'path')
    return os.path.join(path, 'inventory', SAVE_INVENTORY_FILE)


def export_inventory_obj(inventory, fd):
    """

    :param inventory:
    :param fd:
    :return:
    """
    ungrouped = inventory.get_group('ungrouped')
    for host in ungrouped.hosts:
        fd.write("%s ansible_host=%s\n" % (host.name, host.address))

    for grp in inventory.get_group_dict():
        if grp != 'all' and grp != 'ungrouped':
            group = inventory.get_group(grp)

            fd.write("\n[%s]\n" % group)
            if group.hosts:
                for host in group.hosts:
                    fd.write("%s ansible_host=%s\n" % (host.name, host.address))

            if group.child_groups:
                fd.write("\n[%s:children]\n" % group)
                for child in group.child_groups:
                    fd.write("%s\n" % child)

            if group.vars:
                fd.write("\n[%s:vars]\n" % group)
                for var in group.vars:
                    for key, value in var:
                        fd.write("%s=%s\n" % (key, value))


def save_yaml_inventory(node_list, _file=None):
    """

    :param node_list:
    :param _file:
    :return:
    """
    if _file is None:
        _file = SAVE_INVENTORY_FILE

    path = Config().get('global', 'path')
    inventory_path = os.path.join(path, 'inventory')
    if not os.path.exists(inventory_path):
        os.makedirs(inventory_path)

    full_path = os.path.join(inventory_path, _file)
    LOGGER.debug("Saving inventory in %s", full_path)
    save_yaml(full_path, node_list)
