"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import logging
import tarfile
import subprocess
import re
import shlex

try:
    from subprocess import DEVNULL # py3k
except ImportError:
    import os
    DEVNULL = open(os.devnull, 'wb')

import ericsson.deploypattern.lib.errors as errors
from ericsson.deploypattern.lib.config import Config

LOGGER = logging.getLogger(__name__)

KEY_GEN_CONF = "key-gen.conf"
CREATEREPO = "debian_repo.sh"


def __get_repo_path():
    """

    :return:
    """
    repo = Config().get('repo', 'path') + '/' + \
           Config().get('repo', 'name')
    return repo


def __get_ansible_path():
    """

    :return:
    """
    return Config().get('ansible', 'path')


def __get_script():
    """

    :return:
    """
    script_path = os.path.join(__get_ansible_path(), "files")
    return os.path.join(script_path, CREATEREPO)


def __get_key_conf():
    """

    :return:
    """
    return os.path.join(__get_ansible_path(), "files", KEY_GEN_CONF)


def create_apt_repo():
    """

    :return:
    """
    update_apt_repo(create=True)


def update_apt_repo(create=False):
    """ Create a local deb repository """
    try:
        repo_path = __get_repo_path()
        createrepo = __get_script()

        if not os.path.exists(repo_path):
            os.makedirs(repo_path)

        cmd = createrepo
        cmd += " -r " + repo_path + " -g " + __get_key_conf()
        if create:
            LOGGER.info(":: creating APT repository in %s", repo_path)
            cmd += " -c "

        LOGGER.debug(cmd)
        subprocess.call(shlex.split(cmd),
                        stdout=DEVNULL, stderr=subprocess.STDOUT)
    except Exception as msg:
        raise errors.InitError(msg)


def load_deb_repo(tgz):
    """ Load an apt repository """
    try:
        os.path.exists(tgz)
        repo_path = __get_repo_path()
        if not os.path.exists(repo_path):
            raise errors.LoadError("Did you call init before ?")

        LOGGER.info(":: updating APT repository %s", repo_path)

        if tarfile.is_tarfile(tgz) is not True:
            raise errors.LoadError("%s is not a tar file" % tgz)

        tar = tarfile.open(tgz)
        for rpm_file in tar.getmembers():
            if rpm_file.isfile():
                LOGGER.debug(" - > tar members %s", rpm_file.name)
                if rpm_file.name.lower().endswith('.deb'):
                    # check if there is not ../ in file path
                    if re.match(r'\.\./', rpm_file.name):
                        raise errors.LoadError(
                            "suspicious filename with /../ in path")

                    # check if filename not begin with /
                    if re.match('^/', rpm_file.name):
                        raise errors.LoadError(
                            "suspicious filename start with /")

                    LOGGER.info(" - > extracting DEB %s", rpm_file.name)
                    tar.extract(rpm_file, path=repo_path)
                else:
                    raise errors.LoadError("is %s contains dep ?" % tgz)
        tar.close()
        update_apt_repo()

    except Exception as msg:
        raise errors.LoadError(msg)
        #raise
