"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import logging
import os
import socket
import subprocess
import glob
import time
import requests
import shutil

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.config import get_compose_path
from ericsson.deploypattern.lib.config import get_docker_registry_url
from ericsson.deploypattern.tools.registry import start_docker_registry
from ericsson.deploypattern.tools.registry import load_docker_registry
from ericsson.deploypattern.tools.registry import mount_registry_nfs
from ericsson.deploypattern.tools.helm import create_helm_repo_dir
from ericsson.deploypattern.tools.yum import create_yum_repo
from ericsson.deploypattern.tools.apt import create_apt_repo
from ericsson.deploypattern.lib.ansibleplay import run_playbook

LOGGER = logging.getLogger(__name__)




def __configure_docker():
    """

    BUG 58729 - LRO
    Just update configuration file to add insecure-registry options and restart docker service
    :return:
    """
    ## Add option insecure_registry
    registry = get_docker_registry_url().replace("/","")
    with open("/etc/systemd/system/docker.service.d/docker-options.conf", "r") as config_file:
        config = config_file.read()
    config = config.replace("DOCKER_OPTS=", "DOCKER_OPTS=--insecure-registry=" + registry+ " ")
    with open("/etc/systemd/system/docker.service.d/docker-options.conf", "w") as config_file:
      config_file.write(config)
    config_file.close()
    # Reload docker daemon and restart it
    subprocess.Popen(
        "systemctl daemon-reload", shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    proc = subprocess.Popen(
        "systemctl restart docker", shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    if proc.returncode != 0:
        (stdout, stderr) = proc.communicate()
        LOGGER.debug(":: stdout=%s", stdout)
        LOGGER.debug(":: stderr=%s", stderr)
        if stderr:
            raise SystemExit(stderr)


def __validate_sshkey_conf(config):
    """

    :param config:
    :return:
    """
    key = config.get('ansible', 'ssh_key')
    LOGGER.debug(":: check ssh-key : %s", key)
    if key.startswith('~'):
        key = os.path.expanduser(key)
    if not os.path.exists(key):
        raise errors.FileNotFound('%s: ' % key)
    return True


def __validate_ipaddr_conf(config):
    """

    :param config:
    :return:
    """
    addr = config.get('global', 'mgnt_ip_addr')
    try:
        socket.inet_aton(addr)
    except socket.error:
        raise errors.InvalidConfig("mgnt_ip_addr in config not valid")
    return True


def __update_config(config):
    """

    :param config:
    :return:
    """
    with open(config.get_file_conf(), 'wb+') as configfile:
        config.write(configfile)


def __exec_cmd_shell(cmd, error_msg=None):
    """

    :param cmd:
    :param error_msg:
    :return:
    """
    process = subprocess.Popen(cmd, shell=True, stderr=subprocess.PIPE,
                               stdout=subprocess.PIPE)
    (stdout, stderr) = process.communicate()
    LOGGER.debug(":: rc=%s for cmd='%s'", process.returncode, cmd)
    if process.returncode != 0:
        LOGGER.info(stdout)
        LOGGER.info(stderr)
        if error_msg:
            raise errors.InitError(error_msg)
    return (process.returncode, stdout, stderr)

def __check_docker_running():
    """

    :return:
    """
    __exec_cmd_shell("rpm -q docker-ce",
                     "docker is NOT installed")
    (ret, _, _) = __exec_cmd_shell("systemctl status docker")
    if ret != 0:
        __exec_cmd_shell("sudo systemctl restart docker",
                         "docker does NOT restart")
        LOGGER.info("docker is restarted")
    else:
        LOGGER.info("docker is running")

def __check_nfs_mount_point():
    """

    :return:
    """
    if Config().get('docker', 'registry_nfs_ip'):
        source = Config().get('docker', 'registry_nfs_ip') + ':' +\
                 Config().get('docker', 'registry_nfs_path')
        target = r'/var/lib/docker/registry'
        # warning nfs type can ne nfs and nfs4
        cmd = "sudo mount -l | grep '%s on %s type nfs'" % (source, target)
        __exec_cmd_shell(cmd, "nfs registry is NOT correctly mount")
        if os.path.exists(target) and os.path.isdir(target):
            if not os.listdir(target):
                raise errors.InitError("nfs registry %s is empty" % target)
            else:
                LOGGER.debug("nfs registry %s is not empty", target)
        else:
            raise errors.InitError("nfs registry %s don't exists" % target)
        LOGGER.info("nfs registry is available")


def __check_docker_registry_responding():
    """

    :return:
    """
    url = "http://%s/v2/_catalog" % get_docker_registry_url()
    try:
        requests.get(url)
    except requests.ConnectionError:
        if Config().get('global', 'mgnt_ip_addr') != \
                Config().get('docker', 'registry_address'):
            raise errors.InitError("distant docker repository %s is NOT responding" %
                                   get_docker_registry_url())

        # check registry container is available on local repo docker
        __exec_cmd_shell("docker images | grep -w registry",
                         "registry container not loaded in local repo docker")

        # check local registry is started
        (rc, out, error) = __exec_cmd_shell("docker inspect --format '{{json .State.Running }}' registry")
        # try to restart local registry
        if "false" in out:
            __exec_cmd_shell("docker restart registry",
                             "docker registry doesn't restart")
        elif "No such object: registry" in error:
            LOGGER.info("no registry running, registry will be recreate")
            start_docker_registry()

        # check again local registry is started
        (_, out, _) = __exec_cmd_shell("docker inspect --format '{{json .State.Running }}' registry",
                                       "docker inspect does NOT answer correctly")
        # try to restart local registry
        if "false" in out:
            raise errors.InitError("docker registry is NOT restarting")
        else:
            LOGGER.info("docker registry is restarted")

        # check again docker registry is responding
        ret = __wait_connected_url(url, timeout=5)
        if ret.status_code != 200:
            raise errors.InitError("docker registry is always NOT responding")
        else:
            LOGGER.debug("%s is responding", url)
    else:
        LOGGER.info("docker repository is responding")

def __check_docker_compose_mdt():
    """
    Ensure Docker-compose containers are running
    :return:
    """
    cmd = "cd %s && sudo docker-compose -f ./compose_mdt.yaml up -d" % \
          get_compose_path()
    (_, _, err) = __exec_cmd_shell(cmd,  "Docker compose is not up")
    # In case of docker-compose stdout is empty anad stderr have all information,
    LOGGER.info(err)

def __wait_connected_url(url, timeout=None):
    """

    :param url:
    :param timeout:
    :return:
    """
    start_time = time.time()
    while 1:
        try:
            ret = requests.get(url)
            return ret
        except requests.ConnectionError:
            LOGGER.debug("%s is NOT responding", url)
            time.sleep(0.5)
        finally:
            current_time = time.time()
            if (current_time - start_time) >= timeout:
                raise errors.InitError("%s is NOT available" % url)

def __check_req_head_url_ok(url):
    """

    :param url:
    :return:
    """
    ret = __wait_connected_url(url, timeout=5)
    if ret.status_code != 200:
        raise errors.InitError("%s is NOT responding" % url)
    else:
        LOGGER.debug("%s is responding", url)

def __check_repo_type(repo_type="repo"):
    """

    :param repo_type:
    :return:
    """
    ip_addr = Config().get('global', 'mgnt_ip_addr')
    port = Config().get('global', 'http_port')
    path = Config().get(repo_type, 'path')
    name = Config().get(repo_type, 'name')
    # check if directory exist
    target = path + '/' + name
    LOGGER.debug("repo dir = %s", target)
    if os.path.exists(target) and os.path.isdir(target):
        if not os.listdir(target):
            LOGGER.info("%s is empty (%s)", name, target)
        else:
            LOGGER.debug("%s is not empty (%s)", name, target)
    else:
        raise errors.InitError("%s don't exists (%s)" % (name, target))
    # check if repo is responding
    url = "http://%s:%s/%s/" % (ip_addr, port, name)
    try:
        __check_req_head_url_ok(url)
    except errors.InitError:
        __exec_cmd_shell("sudo docker restart httpd",
                         "docker httpd is NOT restarting")
        __check_req_head_url_ok(url)
    LOGGER.info("%s is responding", name)

def __load_docker_in_local_registry(dck_3rd_party_file=None, error_3rdparty=None):
    """

    :param dck_3rd_party_file:
    :param error_3rdparty:
    :return:
    """
    # If not skip load, load MDT and 3rd party containers in local Docker registry
    # The archive for MDT containers is set at installation
    load_docker_registry(os.path.join(Config().get('global', 'path'),
                                      'resources', 'mdt-containers_mdt.tgz'))
    # The archive for 3rdparty containers is by default to global_path/resources
    #   or given by parameter
    if dck_3rd_party_file is None:
        # Search archive file
        try:
            dck_3rd_party_file = glob.glob(
                os.path.join(Config().get('global', 'path'),
                             'resources', 'mdt-containers_3rdparty*'))[0]
        except IndexError as err:
            msg = "There is no 3rd party container archive in %s" % \
                  os.path.join(Config().get('global', 'path'), 'resources')
            if error_3rdparty:
                raise errors.InitError(msg)
            else:
                LOGGER.warning("warning: %s", msg)
    # check dck_3rd_party_file because it can be None when error_3rdparty != None
    if dck_3rd_party_file != None:
        if os.path.isfile(dck_3rd_party_file):
            load_docker_registry(dck_3rd_party_file)
        else:
            msg = "%s is not a valid 3rd party container archive file" % \
                  dck_3rd_party_file
            if error_3rdparty:
                raise errors.InitError(msg)
            else:
                LOGGER.warning("warning: %s", msg)

def check_init(config):
    """

    :param config:
    :return:
    """
    try:
        __validate_ipaddr_conf(config)
        __validate_sshkey_conf(config)
    except Exception as msg:
        raise msg


def mgnt_server_init(ipaddr=None,
                     sshkey=None,
                     registry=None,
                     registry_nfs_addr=None,
                     registry_nfs_path=None,
                     helm_repo=None,
                     user=None,
                     repo=None,
                     ansible_path=None,
                     skip_load=False,
                     stop=False,
                     thirdparty_path=None,
                     http_port=None
                    ):
    """
    usage: mdt init [-h] [--debug] [-i IP_ADDRESS] [-k SSH_KEY] [-g REGISTRY]
                [-d REGISTRY_NFS_ADDRESS] [-p REGISTRY_NFS_PATH] [-u USER]
                [-r REPO] [-a ANSIBLE_PATH] [-c 3RDPARTY_PATH]
                [--http-port HTTP_PORT] [--skip_load] [--stop]

    optional arguments:
        -h, --help            show this help message and exit
        --debug               Enable Debug output messages
        -i IP_ADDRESS, --ip-address IP_ADDRESS
                        IP address of the MDT to use for cluster nodes
        -k SSH_KEY, --ssh-key SSH_KEY
                        ssh key to use for cluster node connections
        -g REGISTRY:PORT, --docker-registry REGISTRY:PORT
                        Specify Docker registry to use
        -m REPOSITORY:PORT, --helm-repository EPOSITORY:PORT
                        Specify Helm repository to use
        -d REGISTRY_NFS_ADDRESS, --registry-nfs-address REGISTRY_NFS_ADDRESS
                        Specify a NFS address for registry storage
        -p REGISTRY_NFS_PATH, --registry-nfs-path REGISTRY_NFS_PATH
                        Specify remote NFS path (with docker/registry/ path)
        -u USER, --user USER  Specify user to use
        -r REPO, --repo REPO  Specify repository to use yum or deb
        -a ANSIBLE_PATH, --ansible-path ANSIBLE_PATH
                        Ansible yaml path
        -c THIRD_PARTY_PATH, --container-3rdparty-path THIRD_PARTY_PATH
                        Path of the 3rd party containers to load, default current directory
        --http-port PORT
                        HTTP port to access MDT containers
        -s, --stop
                        Stop all containers launched by MDT bootstrap
        --skip-load
                        kip load MDT and 3rd party containers to init MDT bootstrap

    :return:
    """
    ## BUG 61208 - LRO
    ## Restore backup Docker configuration files if they exist
    if os.path.exists("/etc/systemd/system/docker.service_bk"):
        shutil.copy("/etc/systemd/system/docker.service_bk",
                    "/etc/systemd/system/docker.service")
    if os.path.exists("/etc/systemd/system/docker.service.d/docker-options.conf_bk"):
        shutil.copy("/etc/systemd/system/docker.service.d/docker-options.conf_bk",
                    "/etc/systemd/system/docker.service.d/docker-options.conf")


    if ipaddr is None and \
       sshkey is None and \
       registry is None and \
       registry_nfs_addr is None and \
       registry_nfs_path is None and \
       helm_repo is None and \
       user is None and \
       repo is None and \
       ansible_path is None and \
       http_port is None and \
       skip_load is False and \
       stop is False:
        LOGGER.info("== init without parameters")
        try:
            config = Config()
            __check_docker_running()
            __check_nfs_mount_point()
            __check_docker_registry_responding()
            if config.get('docker', 'registry_address') == \
                  config.get('global', 'mgnt_ip_addr'):
                __load_docker_in_local_registry(thirdparty_path)
            __check_docker_compose_mdt()
            __check_repo_type("repo")
            __check_repo_type("helm_repo")
        except (errors.InitError, errors.LoadError) as msg:
            LOGGER.error("ERROR: %s", msg)
            exit(1)
        return

    if stop:
        # Stop all containers launched by MDT bootstrap
        LOGGER.info("== Stop MDT containers with docker-compose")
        try:
            cmd = "cd %s && sudo docker-compose -f ./compose_mdt.yaml ps | grep ' Up ' " % \
                  get_compose_path()
            (ret, _, err) = __exec_cmd_shell(cmd)
            if ret == 0:
                cmd = "cd %s && sudo docker-compose -f ./compose_mdt.yaml down " % \
                      get_compose_path()
                (_, _, err) = __exec_cmd_shell(cmd,
                                               "docker compose is NOT down")
                LOGGER.info(err)
            else:
                LOGGER.info("MDT bootstrap already stop")
            # In case of docker-compose stdout is empty
            # and stderr have all information,
        except errors.InitError as msg:
            LOGGER.error("ERROR: %s", msg)
            exit(1)
        return

    LOGGER.info("== initializing MDT server ==")
    LOGGER.info("\n")
    ## LRO
    ## Test if init is possible, it means no cluster deployed
    ## Kubespray creates folder /artifacts/kubectl on MDT node
    if os.path.isfile("/artifacts/kubectl"):
        LOGGER.warning("Kube cluster is deployed. Reset it before init again")
        exit(0)

    # Initialize configuration with parameters
    try:
        # get current config
        config = Config()
        if ipaddr is not None:
            config.set('global', 'mgnt_ip_addr', ipaddr)
        __validate_ipaddr_conf(config)

        if sshkey is not None:
            config.set('ansible', 'ssh_key', sshkey)
        __validate_sshkey_conf(config)

        if user is not None:
            config.set('ansible', 'remote_user', user)

        if http_port is not None:
            config.set('global', 'http_port', http_port)

        if ansible_path:
            config.set('ansible', 'path', ansible_path)

        if registry is not None:
            config.set_registry(registry)
        else:
            # assume local MDT registry by default
            if not config.get('docker', 'registry_address'):
                config.set('docker', 'registry_address',
                           config.get('global', 'mgnt_ip_addr'))

        if registry_nfs_addr == "local":
            # Set back to default values nfs parameters
            default_address = config.get_default_conf().get('docker').get('registry_nfs_ip')
            default_path = config.get_default_conf().get('docker').get('registry_nfs_path')
            config.set('docker', 'registry_nfs_ip', default_address)
            config.set('docker', 'registry_nfs_path', default_path)

        elif bool(registry_nfs_addr) != bool(registry_nfs_path):
            raise errors.InvalidConfig("--registry-nfs-address and "
                                       "--registry-nfs-path must both be set ")
        elif registry_nfs_addr and registry_nfs_path:
            config.set('docker', 'registry_nfs_ip', registry_nfs_addr)
            config.set('docker', 'registry_nfs_path', registry_nfs_path)

        if helm_repo == 'local':
            config.set('helm_repo', 'address',
                config.get_default_conf().get('helm_repo').get('address'))
        elif helm_repo:
            if not helm_repo.startswith("http://"):
                config.set('helm_repo', 'address', 'http://' + helm_repo)
            else:
                config.set('helm_repo', 'address', helm_repo)

        repo_saved = config.get('repo', 'type')

        if repo is not None:
            config.set('repo', 'type', repo)
        elif repo_saved == 'first':
            repo = 'yum'
            config.set('repo', 'type', 'yum')

        # write new config in file
        __update_config(config)

    except Exception:
        LOGGER.error("updating config failed")
        raise

    try:
        # Create the yum RPM repository or apt DEB repository (by default yum)
        if repo_saved == 'first':
            if repo == "yum":
                create_yum_repo()
            elif repo == "apt":
                create_apt_repo()
        elif repo_saved == "yum" and repo == "apt":
            create_apt_repo()
        elif repo_saved == "apt" and repo == "yum":
            create_yum_repo()

        # Simply create directory that will host Helm charts later
        if config.get('helm_repo', 'address') == \
                config.get_default_conf().get('helm_repo').get('address'):
            create_helm_repo_dir()

        # Configure all necessary services: docker
        __configure_docker()

        try:
            # mount registry storage if nfs address provided
            if config.get('docker', 'registry_nfs_ip') != '':
                mount_registry_nfs(config.get('docker', 'registry_nfs_ip'),
                    config.get('docker', 'registry_nfs_path'))

            # If Docker registry is not external
            if config.get('docker', 'registry_address') == \
                    config.get('global', 'mgnt_ip_addr'):
                # Load local image for registry and Start the docker registry
                registry_path = glob.glob(
                    os.path.join(config.get('global', 'path'),
                                 'resources', 'registry*'))[0]
                cmd = "gunzip -c " + registry_path + " | docker load "
                (_, out, err) = __exec_cmd_shell(cmd,
                                                 "local registry NOT loaded correctly")
                LOGGER.info(out)
                start_docker_registry()
                if not skip_load:
                    __load_docker_in_local_registry(thirdparty_path, True)
            else:
                # check local registry is running
                (_, out, _) = __exec_cmd_shell("docker ps --filter \"name=registry\" --filter \"status=running\" -q",
                                               "docker ps --filter... does NOT answer correctly")
                if out:
                    # try to stop local registry
                    __exec_cmd_shell("docker stop registry",
                                     "docker registry doesn't stop")
        except (errors.InitError, errors.LoadError) as msg:
            LOGGER.error("ERROR: %s", msg)
            exit(1)

        # Run all MDT containers
        LOGGER.info(":: Run MDT containers with docker-compose")
        # Update Compose environment variables: HTTP_PORT, DOCKER_REGISTRY and HELM_REPO
        with open(os.path.join(get_compose_path(), '.env'), 'r+') as env_file:
            env = env_file.read()
            env_file.seek(0)
            for line in env_file:
                if line.startswith('HTTP_PORT='):
                    env = env.replace(line.strip(), "HTTP_PORT=%s" %
                                      Config().get('global', 'http_port'))
                if line.startswith('DOCKER_REGISTRY='):
                    env = env.replace(line.strip(), "DOCKER_REGISTRY=%s/" %
                                      get_docker_registry_url())
                if line.startswith('MDT_HELM_REPO='):
                    env = env.replace(line.strip(), "MDT_HELM_REPO=%s/" %
                                      Config().get('helm_repo', 'address'))
            env_file.seek(0)
            env_file.truncate()
            env_file.write(env)
            env_file.close()

        # Add volume for mdt-api-products for future helm certificates after
        # kube deployment, then create future folder
        try:
            cmd = "sudo mkdir -p /root/.helm"
            (_, _, err) = __exec_cmd_shell(cmd, "Helm repo not created")
        except errors.InitError as msg:
            LOGGER.error("ERROR: %s", msg)
            exit(1)

        # Add volume for mdt-api-kube for future kube deployment
        # to save kubectl command and config file
        try:
            cmd = "sudo mkdir -p /artifacts"
            (_, _, err) = __exec_cmd_shell(cmd,
                                           "/artifacts for kubectl MDT not created")
        except errors.InitError as msg:
            LOGGER.error("ERROR: %s", msg)
            exit(1)

        try:
            cmd = "cd %s && sudo docker-compose -f ./compose_mdt.yaml up -d " % \
                  get_compose_path()
            (_, _, err) = __exec_cmd_shell(cmd, "docker compose is NOT up")
            # In case of docker-compose stdout is empty anad stderr have all information,
            LOGGER.info(err)
        except errors.InitError as msg:
            LOGGER.error("ERROR: %s", msg)
            exit(1)

    except Exception as msg:
        raise
