#!/bin/bash

set -e


SCRIPTNAME="`basename "$0"`"
SCRIPTDIR=`dirname "$(readlink -f "$0")"`

APT_REPO_PATH=${APT_REPO_PATH:-/var/www/ericsson/mdt_repo}
GPG_CONF=${GPG_CONF:-${SCRIPTDIR}/key-gen.conf}
create="no"

show_help() {
   cat << EOF
Usage: $SCRIPTNAME [-h help]
Install MDT. Options are
    -h|--help         : Display this help and exit
    -r|--repo-path    : APT repository path to update
    -g|--gpg-conf     : GPG config file for key generation
    -c|--create       : Create an APT repository
EOF
}

exit_on_error ()
{
    local __str="$1"
    echo "[ERROR]: $__str"
    __exit_val=1
    kill -TERM $$
}


while [ $# -gt 0 ]; do
echo "$1 $2"
   case "$1" in
      -h|--help)
         show_help;
         exit 1
         ;;
      -r|--repo-path)
         APT_REPO_PATH="${2}"
         shift
         ;;
      -g|--gpg-conf)
         GPG_CONF="${2}"
         shift
         ;;
      -c|--create)
         create="yes"
         ;;
      *)
         echo "Invalid option"
         show_help
         exit 1
         ;;
   esac
   shift
done


if grep -i "centos" /etc/os-release >/dev/null; then
    yum install -y epel-release
    yum install -y rng-tools dpkg-dev
fi


if [[ ${create} == "yes" ]]; then
    mkdir -p ${APT_REPO_PATH}
    if [[ -f ${GPG_CONF} ]]; then
        if ! gpg -k | grep "mfvp@ericsson.com"; then
            echo "Creating GPG key, could take a while ..."
            rngd -r /dev/urandom  # generate entropy

            gpg --batch --gen-key ${GPG_CONF} || \
            exit_on_error "Could create GPG key with conf file ${GPG_CONF}"
            gpg --armor --export mfvp@ericsson.com >  ${APT_REPO_PATH}/gpg.pub
            # kill -9 $(pidof rngd)
            pkill -9 rngd
        else
            echo "GPG key already exists"
        fi
   else
        exit_on_error "Could not found GPG key conf file ${GPG_CONF}"
   fi
fi


echo "Updating APT repo in ${APT_REPO_PATH}"
cd ${APT_REPO_PATH}  || exit_on_error "Could not cd into ${APT_REPO_PATH}"

#lets create the package index

which dpkg-scanpackages ||  exit_on_error "Could not find dpkg-scanpackages from dpkg-dev package"

dpkg-scanpackages -m . > Packages
cat Packages | gzip -9c > Packages.gz

#create the Release file
PKGS=$(wc -c Packages)
PKGS_GZ=$(wc -c Packages.gz)
cat > Release << EOF
Architectures: all
Date: $(date -R)
MD5Sum:
 $(md5sum Packages  | cut -d" " -f1) $PKGS
 $(md5sum Packages.gz  | cut -d" " -f1) $PKGS_GZ
SHA1:
 $(sha1sum Packages  | cut -d" " -f1) $PKGS
 $(sha1sum Packages.gz  | cut -d" " -f1) $PKGS_GZ
SHA256:
 $(sha256sum Packages | cut -d" " -f1) $PKGS
 $(sha256sum Packages.gz | cut -d" " -f1) $PKGS_GZ
EOF

#and sign it
rm Release.gpg || true
gpg --batch --yes -abs -o Release.gpg Release
cd -

exit 0
