from __future__ import (absolute_import, division, print_function)

__metaclass__ = type

import sys
import json
import datetime
import logging
import requests

from ericsson.deploypattern.lib.config import get_ssh_key_file, get_remote_user
from ericsson.deploypattern.lib.utils import ssh_command


from ansible.plugins.callback.default import CallbackModule

# callbacks are places in Ansible execution that can be plugged into
# for added functionality
# There are expected callback points that can be registered against to trigger
# custom actionsat those points
# Callbacks can bu utilized to change how things are displayed on screen,
# to update a central status system on progress...

DOCUMENTATION = """
    callback: Kube API Notifier
    type: kube-api-status-notifier
    short_description: send ansible events to kube API
    description:
      - Use this callback to send ansible events to kube API
    version_added: "2.7.2"
    extends_documentation_fragment:
      - default_callback
    options:
      url:
        description: Url to connect to kube API server
        env:
          - name: MDT_GLOBAL_KUBE_API_ADDRESS
        default: localhost
"""


LOGGER = logging.getLogger(__name__)


class EventsCallback(CallbackModule):

    CALLBACK_NAME = "Kube API Notifier"
    CALLBACK_TYPE = "kube-api-status-notifier"
    CALLBACK_VERSION = 1.0

    def __init__(self, next_status, category, nodes=[], last_playbook=True):
        super(EventsCallback, self).__init__()
        # log is initialized without a web framework name
        self.failed_nodes = []
        self.start_time = datetime.datetime.utcnow()
        self.status = next_status
        # IF a list of Nodes is defined, use it, otherwise all nodes are used
        self.nodes = nodes
        # Can be "deploy", "reset", "audit"
        self.category = category
        # In case we have many playbooks to complete the deployments,
        # set last_playbook to false
        self.last_playbook = last_playbook

    def _dump_results(self, result, indent=None, sort_keys=True, keep_invocation=False):
        """
        Prepare result as json data
        :param result: ansible task results
        :param indent: json indent
        :param sort_keys: the output of dictionaries will be sorted by key
        :param keep_invocation:
        :return:
        """
        save = {}
        for key in ["stdout_lines", "stderr_lines"]:
            save[key] = result[key] if key in result else []
        for key in ["module_stdout", "module_stderr"]:
            save[key] = result[key].splitlines() if key in result else []
        save["message"] = result["msg"] if "msg" in result else ""
        return save

    def v2_playbook_on_stats(self, stats):
        """
        This callback is called once at the end of the playbook run
        (stats for statistics)
        """
        super(EventsCallback, self).v2_playbook_on_stats(stats)
        end_time = datetime.datetime.utcnow()
        runtime = end_time - self.start_time

        for host in stats.processed.keys():
            if (self.nodes != [] and host in self.nodes) or self.nodes == []:
                # if self.nodes not empty means add/remove node
                if host in self.failed_nodes:
                    # status = "FAILED"
                    # if the initial state is deletionInProgress and KO the
                    # host stays to remove pendingDeletion
                    if self.status == "deleteInProgress":
                        self.status = "pendingDeletion"
                    # if the initial state is auditInProgress set to
                    # auditFailed
                    elif self.status == "auditInProgress":
                        self.status = "auditFailed"
                    # in others cases: full cluster deployment or nodes to add
                    # and KO it means that status is deploymentFailed
                    else:
                        self.status = "deploymentFailed"
                else:
                    # status = "OK"
                    # if the initial state is deletionInProgress,
                    # and OK it means node removed from Kube cluster
                    # then indicate to mdt kube api to delete it
                    if self.status == "deleteInProgress":
                        self.delete_node(host)
                        continue
                    # if the initial state is auditInProgress set to auditOK
                    elif self.status == "auditInProgress" and not self.last_playbook:
                        self.status = "auditOK"
                    # in others cases: full cluster deployment or nodes to add
                    # and OK it means that status is deployed
                    else:
                        self.status = "deployed"
                if self.status == "deployed" and not self.last_playbook:
                    # if last_playbook is false, we need to wait before sending
                    # deployed status to the api.
                    continue
                self.send_status_to_api(
                    data=None, host=host, progress_indicator=False)

    def v2_runner_on_ok(self, result, **kwargs):
        """
         This callback is called upon a successful run of a module

        :param result:
        :param kwargs:
        :return:
        """
        super(EventsCallback, self).v2_runner_on_ok(result, **kwargs)
        # in case of reset cluster
        if self.status in ["configured", "deleteInProgress"]:
            self.send_status_to_api(
                data=None, host=result._host.name, progress_indicator=False)
        else:
            timestamp = datetime.datetime.utcnow()
            data = {
                "status": "OK",
                "category": self.category,
                "event_type": "task",
                "host_name": result._host.name,
                "host_address": result._host.address,
                "task_name": result._task.get_name(),
                "timestamp": "%s" % timestamp,
                "result": self._dump_results(result._result)
            }

            self.send_status_to_api(data=data, host=result._host.name)

    def v2_playbook_on_import_for_host(self, result, imported_file):
        """

        :param result:
        :param imported_file:
        :return:
        """
        super(EventsCallback, self).v2_playbook_on_import_for_host(
            result, imported_file)
        timestamp = datetime.datetime.utcnow()
        data = {
            "status": "IMPORTED",
            "category": self.category,
            "event_type": "import",
            "host_name": result._host.name,
            "host_address": result._host.address,
            "timestamp": "%s" % timestamp,
            "imported_file": imported_file
        }

        self.send_status_to_api(data=data, host=result._host.name)

    def v2_playbook_on_not_import_for_host(self, result, missing_file):
        """

        :param result:
        :param missing_file:
        :return:
        """
        super(EventsCallback, self).v2_playbook_on_not_import_for_host(
            result, missing_file)
        timestamp = datetime.datetime.utcnow()
        data = {
            "status": "NOT IMPORTED",
            "category": self.category,
            "event_type": "import",
            "host_name": result._host.name,
            "host_address": result._host.address,
            "missing_file": missing_file,
            "timestamp": "%s" % timestamp
        }

        self.send_status_to_api(data, result._host.name)

    def v2_runner_on_failed(self, result, ignore_errors=False):
        """

        :param result:
        :param ignore_errors:
        :return:
        """
        super(EventsCallback, self).v2_runner_on_failed(result, ignore_errors)
        timestamp = datetime.datetime.utcnow()
        # in case of audit the fails are ignored, but the final status is
        # failed.
        if not ignore_errors or self.status == "auditInProgress":
            self.failed_nodes.append(result._host.name)
            self.failed_nodes.append(result._host.address)
            data = {
                "status": "FAILED",
                "category": self.category,
                "event_type": "task",
                "host_name": result._host.name,
                "host_address": result._host.address,
                "task_name": result._task.get_name(),
                "timestamp": "%s" % timestamp,
                "result": self._dump_results(result._result)
            }

            self.send_status_to_api(data, result._host.name)

    def v2_runner_on_unreachable(self, result, **kwargs):
        """

        :param result:
        :param kwargs:
        :return:
        """
        super(EventsCallback, self).v2_runner_on_unreachable(result, **kwargs)
        timestamp = datetime.datetime.utcnow()
        data = {
            "status": "UNREACHABLE",
            "category": self.category,
            "event_type": "task",
            "host_name": result._host.name,
            "host_address": result._host.address,
            "task_name": result._task.get_name(),
            "timestamp": "%s" % timestamp,
            "result": self._dump_results(result._result)
        }
        self.failed_nodes.append(result._host.name)
        self.failed_nodes.append(result._host.address)
        self.send_status_to_api(data, result._host.name)

    def send_status_to_api(self, data, host, progress_indicator=True):
        """

        :param data:
        :param host:
        :param progress_indicator:
        :return:
        """
        # don't store information for tasks executed on localhost
        key = get_ssh_key_file()
        user = get_remote_user()
        # don't store information for tasks executed on localhost
        if host not in ["localhost", "127.0.0.1"]:
            # if list of nodes is not empty, compare with the host
            if (self.nodes != [] and host in self.nodes) or self.nodes == []:
                # if the node is defined
                if progress_indicator:
                    result_dict = {"status": {"state": "%s" % self.status,
                                              "progressIndicator": data}}
                else:
                    result_dict = {"status": {
                        "state": self.status}}
                try:
                    headers = {"Content-Type": "application/json"}
                    url = "http://127.0.0.1/api/mdt/kube/cluster/nodes/%s/status" % host
                    result_json = json.dumps(result_dict)
                    result = requests.put(
                        url, data=result_json, headers=headers)
                    if result.status_code != 200:
                        # US 53470 - LRO
                        # in matrix host could be IP address or hostname
                        # parameter host is hostname
                        # If previous request fails it means that IP Address is in matrix
                        # then search IP for hostname and try again
                        ip = ssh_command(
                            host, key, user, "hostname -I | awk '{print $1}'", False)
                        url = "http://127.0.0.1/api/mdt/kube/cluster/nodes/%s/status" % ip[0:-2]
                        result_json = json.dumps(result_dict)
                        result = requests.put(
                            url, data=result_json, headers=headers)
                        if result.status_code != 200:
                            LOGGER.warn("Failed to update status : status_code = %s  message = %s ",
                                        result.status_code, result.text)
                except requests.ConnectionError:
                    pass

    def delete_node(self, host):
        """

        :param host:
        :return:
        """
        # don't store information for taks executed on localhost
        if host not in ["localhost", "127.0.0.1"]:
            if (self.nodes != [] and host in self.nodes) or self.nodes == []:
                try:
                    url = "http://127.0.0.1/api/mdt/kube/cluster/nodes/%s" % host
                    result = requests.delete(url)
                    if result.status_code != 200:
                        LOGGER.warn("Failed to delete node: status_code = %s  message = %s ",
                                    result.status_code, result.text)
                except requests.ConnectionError:
                    LOGGER.warn("Failed to call api delete node: status_code = %s  message = %s ",
                                result.status_code, result.text)
