"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import sys
import logging
import csv
import fcntl
import termios
import struct
import re
import ctypes
import yaml
import paramiko
import tempfile
import shutil
import ericsson.deploypattern.lib.errors as errors

LOGGER = logging.getLogger(__name__)
logging.getLogger("paramiko").setLevel(logging.WARNING)


def load_yaml(_file):
    """ Load a yaml file into a dict """
    with open(_file, 'r') as f:
        try:
            yaml_dict = yaml.load(f) or {}
        except Exception as msg:
            raise errors.YamlError("Failed to load Yaml file: %s" % msg)
        finally:
            f.close()
    return yaml_dict


def save_yaml(_file, data, style=None, indent=2, flow_style=None):
    """ save a yaml dict into a file """
    with open(_file, 'w+') as f:
        try:
            yaml.dump(data, f, default_style=style, indent=indent,
                      default_flow_style=flow_style)
        except Exception as msg:
            logging.error("%s", msg)
            raise
        finally:
            f.close()


def csv2yaml(csv_file_path, yaml_file_path=None, style=None, indent=2, flow_style=None):
    """
    Convert csv file to yaml file
    :param csv_file:
    :param yaml_file:
    :return:
    """
    # If no yaml_file_path parameter, then change only the suffix
    # from csv_file_path
    if yaml_file_path is None:
        yaml_file_path = csv_file_path.rsplit('.')[0] + ".yaml"
    # Read csv file and convert it in DictReader
    # Convert DictReader object in list of dictionaries
    with open(csv_file_path, 'rb') as _csvfile:
        try:
            csv_dict = csv.DictReader(_csvfile)
            list_dict = []
            for row in csv_dict:
                list_dict.append(row)
        except Exception as msg:
            logging.error("%s", msg)
            raise
        finally:
            _csvfile.close()
    # Build and write yaml_file from the list of dictionaries
    with open(yaml_file_path, 'w+') as _yamlfile:
        try:
            yaml.dump(list_dict, _yamlfile, default_style=style, indent=indent,
                      default_flow_style=flow_style)
        except Exception as msg:
            logging.error("%s", msg)
            raise
        finally:
            _yamlfile.close()


def load_csv(csv_file_path):
    """
    Load csv_file and return yaml dictionary
    :param csv_file:
    :return:
    """
    # Read csv file and convert it in DictReader
    # Convert DictReader object in list of dictionaries
    with open(csv_file_path, 'rb') as _csvfile:
        try:
            csv_dict = csv.DictReader(_csvfile)
            list_dict = []
            for row in csv_dict:
                list_dict.append(row)
        except Exception as msg:
            logging.error("%s", msg)
            raise
        finally:
            _csvfile.close()

    return list_dict


def ssh_command(host, key, user, cmd, stream=False):
    try:
        ssh = paramiko.client.SSHClient()
        ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        ssh.load_system_host_keys()
        ssh.connect(host, username=user, key_filename=key)
    except paramiko.AuthenticationException:
        raise paramiko.AuthenticationException(
            "Authentication failed when connecting to %s" % host)
    except Exception as msg:
        raise errors.ConnectionError("Could not SSH to %s: %s" % (host, msg))

    channel = ssh.get_transport().open_session()

    # need to attach tty because of sudo restriction
    width, height = getTerminalSize()
    term = os.getenv('TERM')
    if term is not None:
        channel.get_pty(term=term, width=width, height=height)
    else:
        channel.get_pty(width=width, height=height)

    # remove logging handlers not File because of
    # tty options in paramiko mess
    _saved_handlers = []
    for hdlr in LOGGER.parent.handlers:
        if hdlr.__class__.__name__ is not 'FileHandler':
            _saved_handlers.append(hdlr)
            LOGGER.parent.removeHandler(hdlr)

    channel.exec_command(cmd)

    data = ""
    sys.stdout.flush()
    _out = channel.recv(1024)
    while _out:
        data += _out
        if stream:
            if _out:
                sys.stdout.write(_out)
                sys.stdout.flush()
        _out = channel.recv(1024)

    # filter before write in file because of escape chars from tty ...
    data = re.sub('.*Pulling.*', '', data)
    data = re.sub('.*Already.*', '', data)
    LOGGER.info("%s", data)

    for hdlr in _saved_handlers:
        LOGGER.parent.addHandler(hdlr)

    ssh.close()
    return data


def getTerminalSize():
    """

    :param:
    :return:
    """
    env = os.environ

    def ioctl_GWINSZ(fd):
        """

        :param:
        :return:
        """
        try:
            cr = struct.unpack('hh', fcntl.ioctl(fd, termios.TIOCGWINSZ,
                                                 '1234'))
        except:
            return
        return cr

    cr = ioctl_GWINSZ(0) or ioctl_GWINSZ(1) or ioctl_GWINSZ(2)
    if not cr:
        try:
            fd = os.open(os.ctermid(), os.O_RDONLY)
            cr = ioctl_GWINSZ(fd)
            os.close(fd)
        except:
            pass
    if not cr:
        cr = (env.get('LINES', 25), env.get('COLUMNS', 80))
    return int(cr[1]), int(cr[0])


# from:
# https://github.com/ox-it/python-libmount/blob/master/libmount/mounting.py
_libc = ctypes.cdll.LoadLibrary("libc.so.6")

# Simple wrapper around mount(2) and umount(2).
# Not thoroughly tested, and not very talkative.


class FLAGS:
    def __new__(self):
        raise NotImplementedError("This class is non-instantiable.")

    def flag_bits(count):
        """

        :param count:
        :return:
        """
        flag = 1
        for i in range(count):
            yield flag
            flag <<= 1

    (
        MS_RDONLY,  # 0
        MS_NOSUID,  # 1
        MS_NODEV,  # 2
        MS_NOEXEC,  # 3
        MS_SYNCHRONOUS,  # 4
        MS_REMOUNT,  # 5
        MS_MANDLOCK,  # 6
        MS_DIRSYNC,  # 7
        _, _,  # SKIP 8, 9
        MS_NOATIME,  # 10
        MS_NODIRATIME,  # 11
        MS_BIND,  # 12
        MS_MOVE,  # 13
        MS_REC,  # 14
        MS_SILENT,  # 15
        MS_POSIXACL,  # 16
        MS_UNBINDABLE,  # 17
        MS_PRIVATE,  # 18
        MS_SLAVE,  # 19
        MS_SHARED,  # 20
        MS_RELATIME,  # 21
        MS_KERNMOUNT,  # 22
        MS_I_VERSION,  # 23
        MS_STRICTATIME,  # 24
        _, _, _, _, _,  # SKIP 25-29
        MS_ACTIVE,  # 30
        MS_NOUSER,  # 31
    ) = flag_bits(32)

    del flag_bits, _

    MS_MGC_VAL = 0xc0ed0000
    MS_MGC_MSK = 0xffff0000


def mount(source, target, fstype, flags=0, data=None):
    """

    :param source:
    :param target:
    :param fstype:
    :param flags:
    :param data:
    :return:
    """
    flags = (flags & FLAGS.MS_MGC_MSK) | FLAGS.MS_MGC_VAL

    result = _libc.mount(ctypes.c_char_p(source),
                         ctypes.c_char_p(target),
                         ctypes.c_char_p(fstype),
                         ctypes.c_ulong(flags),
                         ctypes.c_char_p(data) if data is not None else 0)

    if result != 0:
        raise OSError(os.strerror(ctypes.get_errno()))


def umount(target):
    """

    :param target:
    :return:
    """
    result = _libc.umount(ctypes.c_char_p(target))

    if result != 0:
        raise OSError(os.strerror(ctypes.get_errno()))


# if __name__ == '__main__':
    #ssh_command('10.86.71.253', '/root/.ssh/id_rsa', 'root', 'ls /toto', stream=True)

def create_tmp_dir(size):
    """ Create a temp directory to extract tar.gz file """
    # TODO check /tmp space otherwise use ~/

    def check_free(tmpdir):
        """

        :param tmpdir:
        :return:
        """
        _ret = False
        stat = os.statvfs(tmpdir)
        free_size = stat.f_frsize * stat.f_bavail
        LOGGER.debug(" free space on %s =  %s", tmpdir, free_size)
        if free_size > size:
            _ret = True
        return _ret

    # check free size in /tmp /var or /home
    tmp_list = [tempfile.gettempdir(), '/var', os.environ['HOME']]
    for _tmp in tmp_list:
        if check_free(_tmp):
            return tempfile.mkdtemp(dir=_tmp)
    return None


def cleanup(tmp):
    """

    :param tmp:
    :return:
    """
    if tmp is not None and os.path.exists(tmp):
        shutil.rmtree(tmp)