"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import logging
import json

from ansible.errors import AnsibleError

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.plugins.deployplugins import register_plugin
from ericsson.deploypattern.lib.config import get_config_vars_dict
from ericsson.deploypattern.lib.ansibleplay import AnsibleRunner

LOGGER = logging.getLogger(__name__)


class Commissioning(object):
    """ Class to commission OS
    The commissioning is made through Ansible playbook
    """

    def __init__(self, cmd):
        LOGGER.debug("cmd : %s", json.dumps(cmd, indent=4))

        self.commonvars = []
        self.nodes = {}

        self.operation_list = cmd['operation']
        if 'nodes' in cmd:
            self.nodes = cmd['nodes']

        if 'vars' in cmd:
            self.commonvars = {key:value for var in cmd['vars'] for key, value in var.items()}

        self.ansible = AnsibleRunner()
        self._launch_operations()

    def _launch_operations(self):
        """
        Call ansible playbook for each products listed in the pattern
        """

        for ops in self.operation_list:
            LOGGER.debug("ops : %s", ops)

            group = None
            local_vars = {}
            _vars = {}

            name = ops['name']
            LOGGER.info(" -> %s", name)

            if 'group' in ops.keys():
                group = ops['group']

            # add config vars
            _vars.update(get_config_vars_dict())
            #print(_vars)

            # add localvars of an operation
            if 'vars' in ops.keys():
                local_vars = {key:value for var in ops['vars'] for key, value in var.items()}

            # add common vars of all operation
            if self.commonvars is not None:
                _vars.update(self.commonvars)

            LOGGER.debug(" -> vars: %s", _vars)

            try:
                obj = '_' + name.replace('-', '_')
                func = getattr(self, obj)
                LOGGER.debug("try to call func: %s", func)

                # if we call a task, operation vars (localvars)
                # are args of the task
                func(args=local_vars, _vars=_vars, group=group)
            except AttributeError:
                try:
                    LOGGER.debug("try to call role: %s", name)
                    # if we call a role or a playbook,
                    # add localvars to vars
                    _vars.update(local_vars)
                    result = self.ansible.playroles(
                        name=name,
                        host_list=self.nodes,
                        group=group,
                        roles=[name],
                        #vars=vars,
                        extra_vars=_vars)
                    LOGGER.debug("result : %s", result)
                    if result != 0:
                        raise errors.AnsibleError(
                            "Ansible playrole failed role=%s "
                            "vars=%s hosts=%s" % (name, _vars, self.nodes))
                except AnsibleError:
                    #res = run_playbook(name)
                    result = self.ansible.playbook(
                        playbooks=name,
                        host_list=self.nodes,
                        #vars=vars,
                        extra_vars=_vars)
                    if result != 0:
                        raise errors.AnsibleError(
                            "Ansible playbook run failed: %s "
                            "vars=%s hosts=%s" % (name, _vars, self.nodes))

    def _add_node(self, args=None, _vars=None, group=None):
        """
        Add node to K8S cluster
        """
        try:
            LOGGER.debug("add K8S node")
            self.ansible.add_tag("install-k8s-worker-ha")
            self.ansible.add_skip_tag("mdt")
            result = self.ansible.playbook(
                host_list=self.nodes,
                playbooks="install_k8s_HA",
                extra_vars=_vars
            )
            if result != 0:
                msg = "Add node failed"
                raise errors.AnsibleError(msg)
        except Exception as err:
            raise
        return result

    def _disable_selinux(self, args=None, _vars=None, group=None):
        """
        Disable selinux on remote nodes list
        """
        try:
            LOGGER.debug("disable selinux")

            args = {'policy': 'targeted', 'state': 'permissive'}
            result = self.ansible.playtask(
                host_list=self.nodes,
                group=group,
                task="selinux",
                args=args,
                extra_vars=_vars
            )
            if result != 0:
                msg = "disable selinux failed"
                raise errors.AnsibleError(msg)
        except Exception as err:
            raise
        return result

    def _disable_firewalld(self, args=None, _vars=None, group=None):
        """
        Disable firewalld on remote nodes list
        """
        LOGGER.debug("disable firewalld")
        args = {'name': 'firewalld', 'state': 'stopped', 'enabled': False}
        self._disable_systemd_service(_vars=_vars, args=args, group=group)

    def _disable_networkmanager(self, args=None, _vars=None, group=None):
        """
        Disable network manager on remote nodes list
        """
        LOGGER.debug("disable networkmanager")
        args = {'name': 'NetworkManager', 'state': 'stopped', 'enabled': False}
        self._disable_systemd_service(_vars=_vars, args=args, group=group)

    def _disable_systemd_service(self, _vars=None, group=None, args=None):
        """
        Disable service on remote nodes list
        """
        try:
            _vars.update({'ignore_errors': True})
            result = self.ansible.playtask(
                host_list=self.nodes,
                group=group,
                task="systemd",
                args=args,
                extra_vars=_vars
            )
            # if result != 0:
            #     msg = "disable firewalld failed"
            #     raise errors.AnsibleError(msg)
        except Exception as err:
            raise
        return result

    def _update_sysctl(self, _vars=None, group=None, args=None):
        """
        Set sysctl variable  on remote nodes list
        """
        try:
            _vars.update({'ignore_errors': True})
            result = self.ansible.playtask(
                host_list=self.nodes,
                group=group,
                task="sysctl",
                args=args,
                extra_vars=_vars
            )
        except Exception as err:
            raise
        return result

# register the plugin
register_plugin("commissioning", Commissioning)
