"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import sys
import logging
import tarfile
import shutil

import ericsson.deploypattern.lib.errors as errors

from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.ansibleplay import AnsibleRunner
from ericsson.deploypattern.lib.utils import load_yaml
from ericsson.deploypattern.lib.config import get_config_vars_dict

LOGGER = logging.getLogger(__name__)


PATTERN_CONFIG_FILE = 'pattern_config.yaml'
MASTER_PLAYBOOK     = 'configure_k8s.yaml'
GENERATE_PLAYBOOK   = 'generate_playbooks.yaml'

pattern_path = ''


def pattern(args):
    """

    :param args:
    :return:
    """
    if args.pattern_path:
        _set_pattern_path(args.pattern_path)
    if args.config:
        _load_config(args.config)
    if args.install:
        _install(args.install)
    elif args.list_config:
        _list_config()
    elif args.run:
        _launch()


def _install(pattern_tgz):
    """

    :param pattern_tgz:
    :return:
    """
    try:
        LOGGER.info(":: install pattern %s", pattern_tgz)

        # Check if file exist
        if not os.path.exists(pattern_tgz):
            raise errors.LoadError("file %s not found ! " % pattern_tgz)

        # is a readable tar file ?
        if tarfile.is_tarfile(pattern_tgz) is not True:
            raise errors.LoadError("%s is not a tar file" % pattern_tgz)

        pattern_path = _get_pattern_path()
        if not os.path.exists(pattern_tgz):
            os.mkdir(pattern_path)
        # open and extract
        tar = tarfile.open(pattern_tgz)
        tar.extractall(path=pattern_path)
        tar.close()

    except:
        raise


def _list_config():
    """

    :param:
    :return:
    """
    _pattern = _get_pattern_dir()
    config_file = _get_config_file(_pattern)
    if os.path.exists(config_file):
        with open(config_file) as fd:
            data = fd.read()
            print(data)
    else:
        sys.stderr.write("- No config found -\n")
        tpl = os.path.splitext(config_file)[0] + ".tpl"
        if os.path.exists(tpl):
            sys.stderr.write(">>Please, load config from this template :\n")
            with open(tpl) as fd:
                data = fd.read()
                sys.stdout.write(data)


def _load_config(config_file):
    """

    :param config_file:
    :return:
    """
    if os.path.exists(config_file):
        _pattern = _get_pattern_dir()
        shutil.copyfile(config_file, _get_config_file(_pattern))
    else:
        raise errors.LoadError("file %s not found ! " % config_file)


def _launch():
    """

    :param:
    :return:
    """
    _pattern = _get_pattern_dir()
    config_file = _get_config_file(_pattern)
    if not os.path.exists(config_file):
        raise errors.LoadError(" => config not found ! \n"
                               "Please load config first : "
                               "%s deploy-pattern -u pattern_config.yaml "
                               % sys.argv[0])

    conf_vars = load_yaml(config_file)
    common_vars = get_config_vars_dict()

    result = AnsibleRunner().playbook(
        playbooks=os.path.join(_pattern, GENERATE_PLAYBOOK),
        vars=common_vars,
        extra_vars=conf_vars)
    if result != 0:
        raise errors.AnsibleError(
            "Ansible playbook run failed: %s " % GENERATE_PLAYBOOK)

    result = AnsibleRunner().playbook(
        playbooks=os.path.join(_pattern, MASTER_PLAYBOOK),
        vars=common_vars,
        extra_vars=conf_vars)
    if result != 0:
        raise errors.AnsibleError(
            "Ansible playbook run failed: %s " % MASTER_PLAYBOOK)


def _get_pattern_dir():
    """

    :param:
    :return:
    """
    dir = []
    pattern_path = _get_pattern_path()
    for item in os.listdir(pattern_path):
        if os.path.isdir(os.path.join(pattern_path, item)):
            dir.append(item)

    if not dir:
        raise errors.LoadError(" => no pattern found !\n"
                               "Please install pattern first : "
                               "%s deploy-pattern -i pattern_.tar.gz "
                               % sys.argv[0])

    def _select():
        """

        :param:
        :return:
        """
        sys.stderr.write(":: select pattern : \n")
        i = 0
        for d in dir:
            sys.stderr.write("- %s [%s] \n" % (d, i))
            i += 1
        sys.stderr.write(" > pattern number : ")
        num = input()
        return num

    if len(dir) > 1:
        num = _select()
        if 0 < num >= len(dir):
            num = _select()
    else:
        num = 0

    return os.path.join(pattern_path, dir[num])


def _set_pattern_path(path):
    """

    :param:
    :return:
    """
    ## Global is so crappy but, need to rewrite everything
    ## so don"t care anymore about messy code ...
    ## ask to PO why they don't want to invest time in tools development ...
    global pattern_path
    pattern_path = path


def _get_pattern_path():
    """

    :param:
    :return:
    """
    if pattern_path == '':
        return os.path.join(Config().get('global', 'path'), 'pattern')

    return pattern_path


def _get_config_file(_pattern):
    """

    :param:
    :return:
    """
    return os.path.join(_pattern, PATTERN_CONFIG_FILE)
