"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.
 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)

import os
import glob
import logging
import shutil
import subprocess
import json
import re
import socket
import requests


from ericsson.deploypattern.lib import ansibleplay
from ericsson.deploypattern.lib.config import Config
from ericsson.deploypattern.lib.config import get_config_vars_dict, get_ssh_key_file, \
    get_remote_user, get_repo_path, get_repo_name
from ericsson.deploypattern.lib.utils import load_csv, load_yaml, ssh_command
from ericsson.deploypattern.tools import helm
from ericsson.deploypattern.lib import errors

LOGGER = logging.getLogger(__name__)


BUNDLE_FILE_LIST = ['kube-config', 'kube-volumes',
                    'kube-matrix', 'application-version-list']
BUNDLE_FOLDER_LIST = ['mdt-profiles', 'mdt-deploypackages']
BUNDLE_SUFFIX = ('.csv', '.yaml', '.yml')
NODE = 'node'
TYPE = 'type'
LOAD_BALANCER = "load-balancer"
KUBE_MASTER = "kube-master"
KUBE_NODE = "kube-node"
ETCD = 'etcd'
KUBE_ALL = 'k8s-cluster'
PROFILE = 'profile'
KEYS_MATRIX_LIST = [NODE, TYPE, PROFILE]
DPKG = 'deploypackage'
VERSION = 'version'
KEYS_PROFILE_LIST = [DPKG, VERSION]
KEYS_VOLUME_LIST = ['path', 'name', 'address', 'type']
THIRD_PART_LIST = ['mongo', 'rabbitmq', 'redis']


RE_IP = re.compile('\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$')


class Bundle(object):
    """
    From config bundle actions
    """

    def __init__(self):
        self._bundle_path = os.path.join(
            Config().get('global', 'path'), 'bundle')
        if not os.path.exists(self._bundle_path):
            os.makedirs(self._bundle_path)

    def get_inventory(self):
        """
        Get inventory from kube matrix file
        :return: dictionary of lists (group,list nodes)
        """
        LOGGER.info(":: Get list (group, node)")
        # Check matrix_path exists
        f_list = glob.glob(self._bundle_path + "/kube/kube-matrix*")
        if len(f_list) != 1:
            raise errors.BundleError(
                "Matrix file not found in config bundle! ")
        else:
            matrix_path = f_list[0]

        # Convert matrix file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(matrix_path)[1] == '.csv':
            matrix_dict_list = load_csv(matrix_path)
        else:
            matrix_dict_list = load_yaml(matrix_path)

        # Check file: keys are node, type, profile (optional)
        for matrix in matrix_dict_list:
            if (set(matrix.keys()) - set(KEYS_MATRIX_LIST)) == ([]):
                raise errors.BundleError("Key error in %s" % matrix_path)

        # Read files and create dictionaries list
        # one dictionary per node type
        # key = node type, value = list of nodes
        inventory_dict_list = []
        inventory_dict_list.append({KUBE_MASTER: []})
        inventory_dict_list.append({KUBE_NODE: []})
        inventory_dict_list.append({LOAD_BALANCER: []})
        inventory_dict_list.append({ETCD: []})
        inventory_dict_list.append({KUBE_ALL: []})

        key = get_ssh_key_file()
        user = get_remote_user()
        for matrix in matrix_dict_list:
            # matrix (node, type)
            # Search in dictionary list, the dictionary corresponding to the
            # node type
            index = [i for i, j in enumerate(inventory_dict_list) if j.keys()[
                0] == matrix[TYPE]]
            # Add the node to the list of nodes for this type
            # node = dictionary name, access_ip (current inventory used by
            # Ansible)
            # US 53470 - LRO
            # Add access_ip used by Kubespray if there is no ansible_default_ipv4
            # hostvars parameter, it can be the case when there is no default route
            # to infrastructure.
            # access_ip instead of ipaddress not used and not passed to inventory !
            # Set correct variable to inventory: <hostname>  access_ip=<ip_address>
            # In config bundle matrix, there are IP addresses or hostnames.
            # If IP address, then search the hostname
            # If hostname, search the IP address,
            if RE_IP.match(matrix[NODE]):
                # Case matrix with IP address
                # -> For 2.1
                #name = ssh_command(matrix[NODE],key,user,"hostname ", True)
                # inventory_dict_list[index[0]][matrix[TYPE]].append(
                #      {'name':name[0:-2], "access_ip": matrix[NODE]})
                inventory_dict_list[index[0]][matrix[TYPE]].append(
                    {'name': matrix[NODE], "access_ip": matrix[NODE]})
            else:
                # Case matrix with hostname
                # Search IP address, in /etc/hosts or with DNS
                # Translate the host/port argument into a sequence of 5-tuples
                # (with structure family, socktype, proto, canonname, sockaddr)
                # that contain all the necessary arguments for creating a
                # socket
                try:
                    res = socket.getaddrinfo(matrix[NODE], None, socket.AF_UNSPEC,
                                             socket.SOCK_STREAM, 0, socket.AI_PASSIVE)
                    family, socktype, proto, canonname, sockaddr = res[0]
                    ip, port = sockaddr
                    inventory_dict_list[index[0]][matrix[TYPE]].append(
                        {'name': matrix[NODE], "access_ip": ip})
                except Exception as msg:
                    LOGGER.error(
                        "Hostname %s unreachable, cannot gather hostname info", matrix[NODE])
                    return 1

        # copy master to node group to be able to deploy pod on them
        inventory_dict_list[1][KUBE_NODE].append({'children': KUBE_MASTER})
        inventory_dict_list[3][ETCD].append({'children': KUBE_MASTER})
        inventory_dict_list[4][KUBE_ALL].append({'children': KUBE_MASTER})
        inventory_dict_list[4][KUBE_ALL].append({'children': KUBE_NODE})
        return inventory_dict_list

    def get_vars(self):
        """
        Get K8S variable from kube-config fileo
        :return: dictionary var:value
        """
        #LOGGER.info(":: Get K8S variables ")

        # Check kube-config  exists
        f_list = glob.glob(self._bundle_path + "/kube/kube-config*")
        if len(f_list) != 1:
            raise errors.LabelError(
                "kube-config file not found in config bundle! ")
        else:
            kube_config_path = f_list[0]

        # Convert k8sconfig file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(kube_config_path)[1] == '.csv':
            kube_vars_dict = load_csv(kube_config_path)
        else:
            kube_vars_dict = load_yaml(kube_config_path)
        return kube_vars_dict

    def get_vars_kube_config(self):
        """

        :return:
        """
        # Build dictionary to pass to ansible playbook to overwrite default
        # values
        bundle_var_dict = self.get_vars()
        var_dict = {}
        for key, value in bundle_var_dict.items():
            var_dict[key] = value
        return var_dict

    def get_vars_common(self):
        """
        Get K8S variables from common file
        :return: dictionary var:value
        """
        #LOGGER.info(":: Get K8S common variables ")

        # Check common  exists
        f_list = glob.glob(self._bundle_path + "/common*")
        if len(f_list) != 1:
            raise errors.LabelError(
                "common-config file not found in config bundle! ")
        else:
            co_config_path = f_list[0]

        # Convert k8sconfig file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(co_config_path)[1] == '.csv':
            co_vars_dict = load_csv(co_config_path)
        else:
            co_vars_dict = load_yaml(co_config_path)
        return co_vars_dict

    def get_vars_common_config(self):
        """

        :return:
        """
        # Build dictionary for jinja template
        bundle_var_dict = self.get_vars_common()
        var_dict = {}
        for key, value in bundle_var_dict.items():
            var_dict[key] = value
        return var_dict

    def get_volumes_vars(self):
        """
        Get volumes variables from kube-volumes file
        :return: list of dictionary var:value
        """
        LOGGER.info(":: Get storage variables ")

        # Check k8s-volumes  exists
        f_list = glob.glob(self._bundle_path + "/kube/kube-volumes*")
        if len(f_list) != 1:
            raise errors.FileNotFound(
                "kube-volumes file not found in config bundle! ")
        else:
            k8s_volumes_path = f_list[0]

        # Convert kube-volumes file in list of dictionaries according to the format,
        # one per node
        if os.path.splitext(k8s_volumes_path)[1] == '.csv':
            k8s_volume_vars = load_csv(k8s_volumes_path)
        else:
            k8s_volume_vars = load_yaml(k8s_volumes_path)

        return k8s_volume_vars

    # Resource volume
    #================
    def deploy_volumes(self):
        """
        mdt deploy volumes

        Deploy storage classes from k8s-volumes file in config bundle folder.
        A storage class is defined with 4 parameters:
        * name
        * type
        * address
        * path
        :param:
        :return:
        """
        LOGGER.info(":: Deploy Volumes")
        inventory = self.get_inventory()
        k8s_volumes_vars = self.get_volumes_vars()
        if k8s_volumes_vars == {}:
            LOGGER.info("No volume to deploy")
            return
        config_extra_vars = get_config_vars_dict()
        for storage in k8s_volumes_vars:
            try:
                intersection = set(storage.keys()) & set(KEYS_VOLUME_LIST)
            except Exception as msg:
                raise errors.YamlError(
                    "The kube_volumes file is not correct: %s" % msg)
            in_file_only = set(storage.keys()) - set(KEYS_VOLUME_LIST)
            missing_in_file = set(KEYS_VOLUME_LIST) - set(storage.keys())
            if intersection == set(KEYS_VOLUME_LIST):
                if in_file_only != ([]):
                    LOGGER.warn(":: The following parameters will be ignored: %s",
                                in_file_only)
                new_vars_dic = {}
                new_vars_dic["storage_name"] = storage["name"]
                new_vars_dic["nfs_ip"] = storage["address"]
                new_vars_dic["nfs_path"] = storage["path"]
                new_vars_dic["storage_type"] = storage["type"]

                # Check if parameter is set for default storage class
                # in kube-volumes.yml
                # If parameter is not set, set it as "false" by default
                try:
                    storage["default"]
                except:
                    new_vars_dic["storage_default"] = "false"
                else:
                    new_vars_dic["storage_default"] = storage["default"]

                tmp_extra_vars = config_extra_vars
                tmp_extra_vars.update(new_vars_dic)
                result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="deploy_storage",
                    extra_vars=tmp_extra_vars)
                LOGGER.debug("result : %s", result)
                if result != 0:
                    # send status to api
                    api = "api/mdt/kube/volumes/%s/status" % storage["name"]
                    status = "deployedFailed"
                    self.send_status_to_api(status, api)

                    raise errors.AnsibleError(
                        "Ansible playbook failed :deploy_storage ")
            else:
                raise errors.YamlError(
                    "In the following storage class is not correct: %s. Missing the following parameters %s" %
                    (storage, list(missing_in_file)[0]))
            # send status to api
            api = "api/mdt/kube/volumes/%s/status" % storage["name"]
            status = "deployed"
            self.send_status_to_api(status, api)

    def reset_volumes(self):
        """
        mdt reset volumes

        Delete storage classes already deployed on the kubernetes cluster
        List of storage classes is gotten from the kube-API
        :param:
        :return:
        """

        LOGGER.info(":: Reset Volumes")
        inventory = self.get_inventory()
        k8s_volumes_vars = self.get_volumes_vars()
        if k8s_volumes_vars == {}:
            LOGGER.info("No volume to reset")
            return
        config_extra_vars = get_config_vars_dict()

        for storage in k8s_volumes_vars:
            try:
                intersection = set(storage.keys()) & set(KEYS_VOLUME_LIST)
            except Exception as msg:
                raise errors.YamlError(
                    "The kube_volumes file is not correct: %s" % msg)
            in_file_only = set(storage.keys()) - set(KEYS_VOLUME_LIST)
            missing_in_file = set(KEYS_VOLUME_LIST) - set(storage.keys())

            if intersection == set(KEYS_VOLUME_LIST):
                if in_file_only != ([]):
                    LOGGER.warn(":: The following parameters will be ignored: %s",
                                in_file_only)
                new_vars_dic = {}
                new_vars_dic["storage_name"] = storage["name"]
                new_vars_dic["nfs_ip"] = storage["address"]
                new_vars_dic["nfs_path"] = storage["path"]
                new_vars_dic["storage_type"] = storage["type"]

                tmp_extra_vars = config_extra_vars
                tmp_extra_vars.update(new_vars_dic)
                result = ansibleplay.AnsibleRunner().playbook(
                    host_list=inventory,
                    playbooks="reset_storage",
                    extra_vars=tmp_extra_vars)
                LOGGER.debug("result : %s", result)
                if result != 0:
                    # send status to api
                    api = "api/mdt/kube/volumes/%s/status" % storage["name"]
                    status = "deployedFailed"
                    self.send_status_to_api(status, api)

                    raise errors.AnsibleError(
                        "Ansible playbook failed :deploy_storage ")
            else:
                raise errors.YamlError(
                    "In the following storage class is not correct: %s. Missing the following parameters %s" %
                    (storage, list(missing_in_file)[0]))
            # send status to api
            api = "api/mdt/kube/volumes/%s/status" % storage["name"]
            status = "configured"
            self.send_status_to_api(status, api)

    # Resource kube-cluster
    #=======================
    def deploy_kubecluster(self):
        """
            mdt deploy kube-cluster

        Deploy the kube-cluster, Kubernetes on the nodes masters and workers.
        The inventory, list of nodes, is given by the kube-matrix file in the
        config bundle folder.
        The configuration of Kubernetes cluster is given in
        kube-config file in the config bundle folder.
        Before deploy cluster, you have to get the current config bundle and
        save it in /opt/mfvp/deploybundle/bundle
        :return:
        """

        # BZ 59801 - LRO
        # Check if Rpms are loaded before deploy kube-cluster, it is mandatory
        # step
        rpm_file_list = glob.glob(os.path.join(
            get_repo_path(), get_repo_name(), "*.rpm"))
        if not rpm_file_list:
            raise errors.BundleError(
                "ERROR: Load Rpms before deploy Kube cluster !")

        # ckeck if kubectl is installed
        # if no, it is the first deployment
        if os.path.isfile("/artifacts/kubectl"):
            is_first_deployment = False
        else:
            is_first_deployment = True

        if is_first_deployment:
            inventory = self.get_inventory()

            LOGGER.info(":: Deploy Kubernetes")

            # By pass Helm install by Kubepray
            # because helm server (tiller) can be called just inside the cluster
            # and mdt bootstrap needs to call it (outside the cluster)
            kube_vars = self.get_vars_kube_config()
            common_vars = self.get_vars_common()
            commont_dict = {}
            commont_dict['common_var'] = common_vars
            var_dict = {}
            var_dict.update(kube_vars)
            var_dict.update(commont_dict)
            var_dict['helm_enabled'] = False
            var_dict['helm_install'] = True

            # send deploymentInProgress to the api
            api = "api/mdt/kube/cluster/status"
            status = "deploymentInProgress"
            self.send_status_to_api(status, api)

            result = ansibleplay.AnsibleRunner().playbook(
                host_list=inventory,
                playbooks="pre_kubespray",
                vars=get_config_vars_dict(),
                extra_vars=var_dict)
            LOGGER.debug("result : %s", result)
            if result != 0:
                raise errors.AnsibleError(
                    "Ansible playbook failed: pre_kubespray ")

            result = ansibleplay.AnsibleRunner().playbook(
                host_list=inventory,
                playbooks="/opt/mfvp/deploypattern/ansible/kubespray/cluster",
                vars=get_config_vars_dict(),
                extra_vars=var_dict,
                manage_callback=True,
                next_status="deploymentInProgress",
                last_playbook=False,
                category="deploy")
            LOGGER.debug("result : %s", result)
            if result != 0:
                # send status to api
                api = "api/mdt/kube/cluster/status"
                status = "deploymentFailed"
                self.send_status_to_api(status, api)
                raise errors.AnsibleError("Ansible playbook failed: cluster ")

            result = ansibleplay.AnsibleRunner().playbook(
                host_list=inventory,
                playbooks="post_kubespray",
                vars=get_config_vars_dict(),
                extra_vars=var_dict,
                manage_callback=True,
                next_status="deploymentInProgress",
                category="deploy")
            LOGGER.debug("result : %s", result)

            if result != 0:
                # send status to api
                api = "api/mdt/kube/cluster/status"
                status = "deploymentFailed"
                self.send_status_to_api(status, api)
                raise errors.AnsibleError(
                    "Ansible playbook failed: post_kubespray ")
            # send status to api
            api = "api/mdt/kube/cluster/status"
            status = "deployed"
            self.send_status_to_api(status, api)
        else:
            # delete nodes if they are in state pendingDeletion
            self.delete_node_if_needed()
            # scale nodes if they are in state configured or DeployedFailed
            self.scale_node_if_needed()

    def reset_kubecluster(self):
        """
        Reset kube cluster: reset all nodes masters and workers
        Warning: not clean flannel and delete mdt containers
        :return:
        """
        # Delete namespaces
        url = "http://127.0.0.1/api/mdt/kube/cluster/namespaces"
        try:
            result = requests.delete(url)
            if result.status_code != 200:
                LOGGER.warn("Failed to Delete Namespaces : status_code = %s  message = %s ",
                            result.status_code, result.text)
        except requests.ConnectionError:
            LOGGER.warn("%s is NOT responding", url)

        inventory = self.get_inventory()
        LOGGER.info(":: Reset K8S")
        var_dict = self.get_vars_kube_config()
        conf_var_dict = get_config_vars_dict()
        # remove key 'role_path' before give configuration because role_path
        # isn't well setted
        conf_var_dict.pop('role_path')

        # Send status resetInProgress
        api = "api/mdt/kube/cluster/status"
        status = "configured"
        self.send_status_to_api(status, api)

        # Reset K8S cluster on all nodes a
        result = ansibleplay.AnsibleRunner().playbook(
            host_list=inventory,
            playbooks="/opt/mfvp/deploypattern/ansible/kubespray/reset",
            vars=conf_var_dict,
            extra_vars=var_dict,
            manage_callback=True,
            next_status="configured",
            category="reset")
        LOGGER.debug("result : %s", result)
        if result != 0:
            api = "api/mdt/kube/cluster/status"
            status = "deployed"
            self.send_status_to_api(status, api)
            raise errors.AnsibleError(
                "Ansible playbook failed : reset ")

        # Reset Haproxy
        result = ansibleplay.AnsibleRunner().playbook(
            host_list=inventory,
            playbooks="reset_haproxy",
            vars=get_config_vars_dict(),
            extra_vars=var_dict)
        LOGGER.debug("result : %s", result)
        if result != 0:
            raise errors.AnsibleError(
                "Ansible playbook failed: reset_haproxy ")

        # Clean artifacts folder on MDT
        shutil.rmtree('/artifacts', ignore_errors=True)

        # restart docker after reset
        #os.system('sudo systemctl restart docker')
        subprocess.call(['sudo', 'systemctl', 'restart', 'docker'])

    def send_status_to_api(self, status, api):
        """
        Send status to API
        :param status:
        :param api:
        :return:
        """
        headers = {"Content-Type": "application/json"}
        payload = {"status": {"state": status}}
        result_json = json.dumps(payload)
        url = "http://127.0.0.1/%s" % api
        try:
            result = requests.put(url, data=result_json, headers=headers)
            if result.status_code != 200:
                LOGGER.warn("Failed to update status : status_code = %s  message = %s ",
                            result.status_code, result.text)
        except requests.ConnectionError:
            pass

    def get_api(self, api):
        """
        Get from API
        :param api:
        :return:
        """
        headers = {"Content-Type": "application/json"}
        url = "http://127.0.0.1/%s" % api
        try:
            result = requests.get(url, headers=headers)
        except Exception as error_api:
            raise errors.CliError("get on %s is failed: %s" %
                                  (url, error_api.message))
        return result

    def set_api(self, api, payload):
        """
        Set to API
        :param api:
        :param payload:
        :return:
        """
        headers = {"Content-Type": "application/json"}
        payload_json = json.dumps(payload)
        url = "http://127.0.0.1/%s" % api
        try:
            result = requests.put(url, data=payload_json, headers=headers)
        except Exception as error_api:
            raise errors.CliError("put on %s is failed: %s" %
                                  (url, error_api.message))
        return result

    def del_api(self, api):
        """
        Delete to API
        :param api:
        :return:
        """
        headers = {"Content-Type": "application/json"}
        url = "http://127.0.0.1/%s" % api
        try:
            result = requests.delete(url, headers=headers)
        except Exception as error_api:
            raise errors.CliError("delete on %s is failed: %s" %
                                  (url, error_api.message))
        return result

    # Set node
    #=================
    def set_node(self, node_name=None, node_type=None):
        """
        set node with name and type in status configured, it's ready to deployed
        :return:
        """

        LOGGER.debug(":: Set node")
        if node_name is None:
            raise errors.CliError("node name must be defined")
        if node_type != KUBE_MASTER and node_type != KUBE_NODE:
            raise errors.CliError("node type must be %s or %s" %
                                  (KUBE_MASTER, KUBE_NODE))

        # retreive kube matrix throught api
        response = self.get_api("api/mdt/kube/matrix")
        if response.status_code != 200:
            raise errors.CliError("get kube matrix failed: code = %s msg = %s" %
                                  (response.status_code, response.text))
        LOGGER.debug("r = %s : r.json = %s", response, response.json())
        cfg = response.json()['kube_matrix']
        LOGGER.debug("cfg = %s", cfg)
        # check if node is already in kube matrix
        for current_dico in cfg:
            if current_dico['config']['node'] == node_name:
                raise errors.CliError("node %s already exist" % node_name)
        # set node in config
        cfg.append({"config": {"node": node_name, "type": node_type}})
        LOGGER.debug("cfg updated = %s", cfg)
        # update kube matrix throught api
        response = self.set_api("api/mdt/kube/matrix", {"kube_matrix": cfg})
        if response.status_code != 200:
            raise errors.CliError("update kube matrix failed: code = %s msg = %s" %
                                  (response.status_code, response.text))

        # retreive products matrix throught api
        response = self.get_api("api/mdt/products/matrix")
        if response.status_code != 200:
            raise errors.CliError("get products matrix failed: code = %s msg = %s" %
                                  (response.status_code, response.text))
        LOGGER.debug("r = %s : r.json = %s", response, response.json())
        cfg = response.json()['products_matrix']
        LOGGER.debug("cfg = %s", cfg)
        # check if node is already in products matrix
        for current_dico in cfg:
            if current_dico['config']['node'] == node_name:
                if current_dico['config']['profile'] != "":
                    raise errors.CliError(
                        "node %s already exist with none empty profile" % node_name)
        # set node in config
        cfg.append({"config": {"node": node_name, "profile": ""}})
        LOGGER.debug("cfg updated = %s", cfg)
        # update kube matrix throught api
        response = self.set_api(
            "api/mdt/products/matrix", {"products_matrix": cfg})
        if response.status_code != 200:
            raise errors.CliError("update products matrix failed: code = %s msg = %s" %
                                  (response.status_code, response.text))

    # Resource kube-cluster
    #=======================
    def audit_kubecluster(self, node_name=None):
        """
        Audit kube-cluster command. Use Kubespray playbooks to do this audit and send status to the api.
        :param: node_name (hostname or IP address) does not correspond to matrix
        :return: None
        """
        LOGGER.debug(":: Audit kube-cluster")

        ## LRO - review
        # Pb if node_name is hostname we don't set IP address in inventory

        # retreive nodes through api
        response = self.get_api("api/mdt/kube/cluster/nodes")
        if response.status_code != 200:
            raise errors.CliError("get cluster nodes failed: code = %s msg = %s" %
                                  (response.status_code, response.text))
        LOGGER.debug("r = %s : r.json = %s", response, response.json())
        nodes = response.json()
        LOGGER.debug("nodes = %s", nodes)
        # Variable used in case we specify a node_name
        node_list = []
        # Get inventory from the matrix
        inventory = self.get_inventory()
        LOGGER.debug("inventory = %s", inventory)
        if node_name is not None:
            node_found = False
            node_type = None
            # Check if the given node is in the cluster nodes
            # LRO - Add it to node_list for future callback
            # We assume that node parameter is passed like set in matrix or not ?
            # If parameter is hostname, serach its IP address to always check
            # existence with IP Address always in inventory with parameter
            # access_ip
            if not RE_IP.match(node_name):
                # node-name is hostname
                # Search IP address, in /etc/hosts or with DNS
                # Translate the host/port argument into a sequence of 5-tuples
                # (with structure family, socktype, proto, canonname, sockaddr)
                # that contain all the necessary arguments for creating a
                # socket
                res = socket.getaddrinfo(node_name, None, socket.AF_UNSPEC,
                                         socket.SOCK_STREAM, 0, socket.AI_PASSIVE)
                family, socktype, proto, canonname, sockaddr = res[0]
                node_ip, port = sockaddr
            else:
                node_ip = node_name
            if node_ip in [node['access_ip'] for node in inventory[0][KUBE_MASTER]]:
                node_found = True
                node_list.append(str(node_name))
                # We update inventory: remove node except masters
                kube_node_inventory = {
                    KUBE_NODE: [{'children': 'kube-master'}]}
                inventory[1][KUBE_NODE] = kube_node_inventory[KUBE_NODE]
            # for KUBE_NODE, there is a children KUBE-MASTER
            elif node_ip in [node['access_ip'] for node in inventory[1][KUBE_NODE] if not 'children' in node.keys()]:
                node_found = True
                node_list.append(str(node_name))
                # We update inventory
                # Build KUBE_NODE with always KUBE_MASTER and only the node
                # passed by parameter
                kube_node_inventory = {
                    KUBE_NODE: [{'children': 'kube-master'}]}
                kube_node_inventory[KUBE_NODE].append(
                    {'access_ip': str(node_ip),
                     'ip': str(node_ip),
                     'name': str(node_name)})
                inventory[1][KUBE_NODE] = kube_node_inventory[KUBE_NODE]

            if not node_found:
                raise errors.CliError(
                    "node name must be set in the kube-matrix before audit")

        LOGGER.debug("---- %s", inventory)
        var_dict = {"ignore_assert_errors": True}

        result = ansibleplay.AnsibleRunner().playbook(
            host_list=inventory,
            playbooks="audit",
            vars=get_config_vars_dict(),
            extra_vars=var_dict,
            manage_callback=True,
            next_status="auditInProgress",
            nodes=node_list,
            category="audit",
            last_playbook=False)
        LOGGER.debug("result : %s", result)
        if result != 0:
            raise errors.AnsibleError(
                "Ansible playbook failed: audit")
        result = ansibleplay.AnsibleRunner(tags=['asserts']).playbook(
            host_list=inventory,
            playbooks="/opt/mfvp/deploypattern/ansible/kubespray/cluster",
            vars=get_config_vars_dict(),
            extra_vars=var_dict,
            manage_callback=True,
            next_status="auditInProgress",
            nodes=node_list,
            category="audit")
        LOGGER.debug("result : %s" % result)
        if result != 0:
            raise errors.AnsibleError(
                "Ansible playbook failed: audit ")

    # Unset node
    #=================
    def unset_node(self, node_name=None):
        """
        unset node with name, it's ready to deployed
        :return:
        """

        LOGGER.debug(":: Unset node")
        if node_name is None:
            raise errors.CliError("node name must be defined")

        # retreive kube matrix throught api
        response = self.get_api("api/mdt/kube/matrix")
        if response.status_code != 200:
            raise errors.CliError("get kube matrix failed: code = %s msg = %s" %
                                  (response.status_code, response.text))
        LOGGER.debug("r = %s : r.json = %s", response, response.json())
        cfg = response.json()['kube_matrix']
        LOGGER.debug("cfg = %s", cfg)

        # check if node is not in kube matrix
        node_find = False
        for current_dico in cfg:
            if current_dico['config']['node'] == node_name:
                node_find = True
                # unset node in config
                cfg.remove(current_dico)
                break
        # The node can be present in UI with status pendingDeletion and
        # not be in Kube cluster matrix
        if not node_find:
            raise errors.CliError(
                "node %s doesn't exist or will be deleted" % node_name)
        LOGGER.debug("cfg updated = %s", cfg)

        # retreive products matrix throught api
        response = self.get_api("api/mdt/products/matrix")
        LOGGER.debug("r = %s : r.json = %s", response, response.json())
        cfg_products = response.json()['products_matrix']
        LOGGER.debug("cfg_products = %s", cfg_products)
        # When no profile, in the return matrix is empty string
        for current_dico in cfg_products:
            if current_dico['config']['node'] == node_name:
                if current_dico['config']['profile']:
                    raise errors.CliError("Profile %s already set on node %s" %
                                          (current_dico['config']['profile'], node_name))
                cfg_products.remove(current_dico)
                break
        # update kube matrix throught api
        response = self.set_api("api/mdt/kube/matrix", {"kube_matrix": cfg})
        if response.status_code != 200:
            raise errors.CliError("update kube matrix failed: code = %s  msg = %s" %
                                  (response.status_code, response.text))
        # update products matrix throught api
        response = self.set_api("api/mdt/products/matrix",
                                {"products_matrix": cfg_products})
        if response.status_code != 200:
            raise errors.CliError("update products matrix failed: code = %s  msg = %s" %
                                  (response.status_code, response.text))

    def scale_node_if_needed(self):
        """
        Add worker nodes into cluster
        Nodes to add have to be set in kube cluster before call to this function
        :return:
        """
        # Search all nodes with status configured or deploymentFailed
        # to add only them in kube cluster
        response = self.get_api(
            "api/mdt/kube/cluster/nodes?status=configured,deploymentFailed")
        nodes = response.json()
        if nodes:
            LOGGER.info(":: Scaling nodes")
            # US 45712 - LRO
            # node_list used by callback
            node_list = []
            kube_node_inventory = {KUBE_NODE: []}
            for node in nodes:
                if node["type"] == KUBE_NODE:
                    node_list.append(str(node["id"]))

                    # Send status resetInProgress
                    api = "api/mdt/kube/cluster/nodes/%s/status" % node["id"]
                    status = "deploymentInProgress"
                    self.send_status_to_api(status, api)

                    kube_node_inventory[KUBE_NODE].append(
                        {'access_ip': str(node["id"]),
                         'ip': str(node["id"]),
                         'name': str(node["id"])})

            # US 45712 - LRO
            # update inventory kube node group, with just nodes to add
            # don't add master nodes, they are already configured as worker
            # nodes
            inventory = self.get_inventory()
            inventory[1][KUBE_NODE] = kube_node_inventory[KUBE_NODE]
            LOGGER.debug("---- %s", inventory)

            var_dict = {}

            result = ansibleplay.AnsibleRunner().playbook(
                host_list=inventory,
                playbooks="pre_scale",
                vars=get_config_vars_dict(),
                extra_vars=var_dict)
            LOGGER.debug("result : %s" % result)
            if result != 0:
                raise errors.AnsibleError(
                    "Ansible playbook failed: pre_scale ")

            # run playbook
            result = ansibleplay.AnsibleRunner().playbook(
                host_list=inventory,
                playbooks="/opt/mfvp/deploypattern/ansible/kubespray/scale",
                vars=get_config_vars_dict(),
                extra_vars=var_dict,
                manage_callback=True,
                next_status="deploymentInProgress",
                nodes=node_list,
                category="deploy")

            LOGGER.info("-------result of LOGGER.info on result command:")
            LOGGER.info("result : %s", result)
            if result != 0:
                raise errors.AnsibleError(
                    "Ansible playbook failed: scale.yml ")

            # Send status deployed if ok
            self.send_status_to_api(status="deployed",
                                    api="api/mdt/kube/cluster/status")

    def delete_node_if_needed(self):
        """
        Delete node from cluster
        :return:
        """
        # Search all nodes with status pendindDeletion to delete them
        # in kube cluster
        response = self.get_api(
            "api/mdt/kube/cluster/nodes?status=pendingDeletion")
        nodes = response.json()
        if nodes:
            LOGGER.info(":: Deleting nodes")
            # US 45712 - LR0
            # node_list used by callback
            node_list = []
            kube_node_inventory = {KUBE_NODE: []}
            for node in nodes:
                if node["type"] == KUBE_NODE:
                    node_list.append(str(node["id"]))

                    # Send status resetInProgress
                    api = "api/mdt/kube/cluster/nodes/%s/status" % node["id"]
                    status = "deleteInProgress"
                    self.send_status_to_api(status, api)

                    kube_node_inventory[KUBE_NODE].append(
                        {'access_ip': str(node["id"]),
                         'ip': str(node["id"]),
                         'name': str(node["id"])})

            # update inventory kube node
            # US 45712 - LRO
            # Is kube-node group used in playbooks ?
            #kube_node_inventory[KUBE_NODE].append({'children': KUBE_MASTER})
            inventory = self.get_inventory()
            inventory[1][KUBE_NODE] = kube_node_inventory[KUBE_NODE]
            LOGGER.debug("---- %s", inventory)

            # US 45712 - LRO
            # node is a string of node names for Kubespray playbooks
            # to remove nodes, with "," as separator
            # node names are IP Addresses or hostnames known for Kubernetes,
            # it means like in Kube or products matrix
            # !! be careful we set id, for now it's OK because id corresponds
            # to IP Address or hostname according to the matrix
            # Convert list to string
            var_dict = {}
            var_dict['node'] = ",".join(node_list)
            LOGGER.debug("---- extra_vars : %s", var_dict)

            # US 45712 - LRO
            # remove key 'role_path' before give configuration because role_path
            # isn't well setted
            # !! test to remove it in function get_config_vars_dict
            conf_var_dict = get_config_vars_dict()
            conf_var_dict.pop('role_path')

            # run playbook
            result = ansibleplay.AnsibleRunner().playbook(
                host_list=inventory,
                playbooks="/opt/mfvp/deploypattern/ansible/kubespray/remove-node",
                vars=conf_var_dict,
                extra_vars=var_dict,
                manage_callback=True,
                next_status="deleteInProgress",
                nodes=node_list,
                category="reset")

            LOGGER.info("-------result of LOGGER.info on result command:")
            LOGGER.info("result : %s", result)
            if result != 0:
                raise errors.AnsibleError(
                    "Ansible playbook failed : remove-node.yml")

            # Send status deployed
            self.send_status_to_api(status="deployed",
                                    api="api/mdt/kube/cluster/status")
