"""
This software has been developed by Ericsson.

Copyright (c) 2016 Ericsson, Inc.

COPYRIGHT:
    This file is the property of Ericsson.
    It cannot be copied, used, or modified without obtaining
    an authorization from the authors or a mandated
    member of Ericsson.
    If such an authorization is provided, any modified version or
    copy of the software must contain this header.

 WARRANTIES:
    This software is made available by the authors in the hope
    that it will be useful, but without any warranty.
    Ericsson.com is not liable for any consequence related to the
    use of the provided software.
"""
from __future__ import (absolute_import, division, print_function)


import subprocess
import os
import logging
import sys
import time
import glob
import shutil
import ericsson.deploypattern.lib.errors as errors
from ericsson.deploypattern.lib.config import get_helm_charts_server, get_helm_charts_reponame


LOGGER = logging.getLogger(__name__)

DCK_CLI_CMD = "sudo docker exec --user root mdt-cli"


def exec_cmd_shell(cmd):
    """
    This function should be replaced by the one made in US51109

    :param cmd:
    :return:
    """
    proc = subprocess.Popen(
        cmd, shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    sys.stdout.write("working ")
    sys.stdout.flush()
    while True:
        if proc.poll() is not None:
            sys.stdout.write("\n")
            sys.stdout.flush()
            break
        else:
            sys.stdout.write(".")
            sys.stdout.flush()
        time.sleep(0.5)
    (stdout, stderr) = proc.communicate()
    LOGGER.debug(":: rc=%s for cmd='%s'", proc.returncode, cmd)
    if proc.returncode != 0:
        LOGGER.debug(":: stdout=%s", stdout)
        LOGGER.debug(":: stderr=%s", stderr)
        if stderr:
            raise SystemExit(stderr)
    return (proc.returncode, stdout, stderr)


def confirm(text, default=False, show_default=True):
    """
    Prompt for confirmation (yes/no)
    :param text: the question to ask
    :param default: the default value of the input
    :param show_default: show the default/available values in the prompt
    :return: True or False
    """
    prompt = text
    if show_default:
        default_str = '[Y/n]' if default else '[y/N]'
        prompt += ' {} '.format(default_str)

    while True:
        try:
            choice = raw_input(prompt).lower()
        except (KeyboardInterrupt, EOFError):
            return False

        if not choice:
            return default
        elif choice in ('y', 'yes'):
            return True
        elif choice in ('n', 'no'):
            return False

        print("Please enter 'yes'/'y' or 'no'/'n'.\n")


def get_index_of_arg(param_list, short_arg=None, long_arg=None):
    """
    This function return index for a list of param like
    this ['-o', 'test_file.txt', '-d', 'aDirectory', ...]
    :param param_list: the list of param/value
    :param short_arg: argument to find in short version (ex : -o)
    :param long_arg: argument to find in long version (ex:--output)
    :return: Index of argument
    """
    if param_list is None:
        param_list = []
    index = None

    if short_arg is not None and param_list.count(short_arg) > 0:
        index = param_list.index(short_arg)
    if long_arg is not None and param_list.count(long_arg) > 0:
        index = param_list.index(long_arg)

    return index


class Cfg(object):
    """
     Config bundle class
    """

    def __init__(self):
        self.resource = type(self).__name__.lower()

    def get(self, optional_param=None):
        """
        Get the entire configuration bundle currently applied and export all
        files as a tgz archive. The output archive is created in the path
        given in command line with option -o.
        :param optional_param:
        :return:
        """
        index = get_index_of_arg(optional_param, '-o', '--output')
        if index is None:
            raise errors.CliError("-o option must be used")
        opt_file = optional_param[index + 1]
        opt_cont_file = "/tmp/file.tgz"
        # Run mdt cli command
        docker_cmd = "%s mdt get %s -o %s" % (
            DCK_CLI_CMD, self.resource, opt_cont_file)

        exec_cmd_shell(docker_cmd)
        # Get file from container
        docker_cmd = "sudo docker cp mdt-cli:%s %s" % (
            opt_cont_file, opt_file)
        exec_cmd_shell(docker_cmd)
        # Clean folder
        docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
            DCK_CLI_CMD, opt_cont_file)
        exec_cmd_shell(docker_cmd)

        if index is not None:
            index = get_index_of_arg(optional_param, '-o', '--output')
            del optional_param[index + 1]
            del optional_param[index]

    def load(self, optional_param=None):
        """
        Load the content of the given config bundle from CONF_BUNDLE_TGZ archive
        file or CONF_BUNDLE_DIR directory path. If there is already a config bundle
        loaded, warning message is displayed. Use the option --force to force the loading.
        :param optional_param:
        :return:
        """
        # Case Load with file parameter
        index = get_index_of_arg(optional_param, '-f', '--file')
        if index is not None:
            # case option file exists
            try:
                opt_file = optional_param[index + 1]
                opt_cont_file = "/tmp/file.tgz"
                if os.path.isfile(opt_file):
                    # Load file to container
                    docker_cmd = "sudo docker cp %s mdt-cli:%s" % (
                        opt_file, opt_cont_file)
                    exec_cmd_shell(docker_cmd)
                    # Run mdt cli command
                    docker_cmd = "%s mdt load %s -f %s" % (
                        DCK_CLI_CMD, self.resource, opt_cont_file)
                    if optional_param.count('--force') > 0:
                        index_force = get_index_of_arg(
                            optional_param, None, '--force')
                        del optional_param[index_force]
                        docker_cmd = "%s --force" % (docker_cmd)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.info("%s", out)
                else:
                    # case option file does not exist
                    LOGGER.error("load %s failed: file %s not found",
                                 self.resource, opt_file)
                    raise errors.FileNotFound("config bundle not found")
                index = get_index_of_arg(optional_param, '-f', '--file')
                del optional_param[index + 1]
                del optional_param[index]

            finally:
                # Clean folder
                try:
                    docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                        DCK_CLI_CMD, opt_cont_file)
                    exec_cmd_shell(docker_cmd)
                except (SystemExit, NameError) as msg:
                    # file not found in container. No need to send a message
                    # for this
                    pass

        # Case Load with folder parameter
        index = get_index_of_arg(optional_param, '-d', '--directory')
        if index is not None:
            # case option folder exists
            try:
                opt_dir = optional_param[index + 1]
                opt_cont_dir = "/tmp/cfg_dir"
                if os.path.isdir(opt_dir):
                    # Load folder to container
                    docker_cmd = "sudo docker cp %s mdt-cli:%s" % (
                        opt_dir, opt_cont_dir)
                    exec_cmd_shell(docker_cmd)
                    # Run mdt cli command
                    docker_cmd = "%s mdt load %s -d %s " % (
                        DCK_CLI_CMD, self.resource, opt_cont_dir)
                    if optional_param.count('--force') > 0:
                        index_force = get_index_of_arg(
                            optional_param, None, '--force')
                        del optional_param[index_force]
                        docker_cmd = "%s --force" % (docker_cmd)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.info("%s", out)
                else:
                    # case option folder doesn't exist
                    LOGGER.error("load %s failed: folder %s not found",
                                 self.resource, opt_dir)
                    raise errors.DirectoryNotFound("config bundle not found")
                index = get_index_of_arg(optional_param, '-d', '--directory')
                del optional_param[index + 1]
                del optional_param[index]

            finally:
                # Clean folder
                try:
                    docker_cmd = "%s python3.6 -c \"import shutil; shutil.rmtree('%s')\"" % (
                        DCK_CLI_CMD, opt_cont_dir)
                    exec_cmd_shell(docker_cmd)
                except (SystemExit, NameError) as msg:
                    # file not found in container. No need to send a message
                    # for this
                    pass


class Label(object):
    """
    Label class
    """

    def __init__(self):
        self.resource = type(self).__name__.lower()

    def get(self, optional_param=None):
        """
        Get all the labels from all the nodes or from one node by defining
        the node name with the option -n. Labels are displayed in yaml format.
        If option -o is set, then the result will also be saved into the file.
        :param optional_param:
        :return:
        """
        index_node = get_index_of_arg(optional_param, '-n', '--node')
        index_file = get_index_of_arg(optional_param, '-o', '--output')
        opt_cont_file = "/tmp/label.yml"

        if index_node is None and index_file is None:
            # Run mdt cli command
            docker_cmd = "%s mdt get %s" % (DCK_CLI_CMD, self.resource)
            # ouput of docker cmd is redirected on stderr
            (_, out, _) = exec_cmd_shell(docker_cmd)
            LOGGER.info("%s", out)
        elif index_file is None:
            node = optional_param[index_node + 1]
            # Get file from container
            docker_cmd = "%s mdt get %s -n %s" % (
                DCK_CLI_CMD, self.resource, node)
            # ouput of docker cmd is redirected on stderr
            (_, out, _) = exec_cmd_shell(docker_cmd)
            LOGGER.info("%s", out)
        elif index_node is None:
            opt_file = optional_param[index_file + 1]
            # Get file from container
            docker_cmd = "%s mdt get %s -o %s" % (
                DCK_CLI_CMD, self.resource, opt_cont_file)
            exec_cmd_shell(docker_cmd)

            # copy file
            docker_cmd = "sudo docker cp mdt-cli:%s %s" % (
                opt_cont_file, opt_file)
            exec_cmd_shell(docker_cmd)

            # Clean file
            docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                DCK_CLI_CMD, opt_cont_file)
            exec_cmd_shell(docker_cmd)
        else:
            node = optional_param[index_node + 1]
            opt_file = optional_param[index_file + 1]
            # Get file from container
            docker_cmd = "%s mdt get %s -n %s -o %s" % (
                DCK_CLI_CMD, self.resource, node, opt_cont_file)
            exec_cmd_shell(docker_cmd)
            # copy file
            docker_cmd = "sudo docker cp mdt-cli:%s %s" % (
                opt_cont_file, opt_file)
            exec_cmd_shell(docker_cmd)
            # Clean file
            docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                DCK_CLI_CMD, opt_cont_file)
            exec_cmd_shell(docker_cmd)

        # Remove arguments who are processed
        if index_node is not None:
            index_node = get_index_of_arg(optional_param, '-n', '--node')
            del optional_param[index_node + 1]
            del optional_param[index_node]
        if index_file is not None:
            index_file = get_index_of_arg(optional_param, '-o', '--output')
            del optional_param[index_file + 1]
            del optional_param[index_file]


class Products(object):
    """ Products class
    """

    def __init__(self):
        self.resource = type(self).__name__.lower()

    def deploy(self, optional_param=None):
        """
         Deploy all products
        :return:
        """
        # US 59036 - LRO
        # Add option -p profile
        index = get_index_of_arg(optional_param, '-p', '--profile')

        docker_cmd = "%s mdt deploy %s " % (DCK_CLI_CMD, self.resource)
        if index is not None:
            # case option profile exists
            # Run mdt cli command
            profile = optional_param[index + 1]
            docker_cmd = "%s -p %s" % (docker_cmd, profile)
        # ouput of docker cmd is redirected on stderr
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)

        # Remove arguments who are processed
        if index is not None:
            index = get_index_of_arg(optional_param, '-p', '--profile')
            del optional_param[index + 1]
            del optional_param[index]

    def reset(self, optional_param=None):
        """
        Reset all products
        :return:
        """
        # US 59036 - LRO
        # Add option -p profile
        index = get_index_of_arg(optional_param, '-p', '--profile')

        docker_cmd = "%s mdt reset %s " % (DCK_CLI_CMD, self.resource)
        if index is not None:
            # case option profile exists
            profile = optional_param[index + 1]
            docker_cmd = "%s -p %s" % (docker_cmd, profile)
        # ouput of docker cmd is redirected on stderr
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)

        # Remove arguments who are processed
        if index is not None:
            index = get_index_of_arg(optional_param, '-p', '--profile')
            del optional_param[index + 1]
            del optional_param[index]

    def upgrade(self, optional_param=None):
        """
        upgrade products
        :return:
        """
        index_prepare = get_index_of_arg(optional_param, '--prepare')
        index_latest = get_index_of_arg(optional_param, '--latestCharts')
        index_file = get_index_of_arg(optional_param, '-o', '--output')
        index_clean = get_index_of_arg(optional_param, '--clean')
        index_profile = get_index_of_arg(optional_param, '-p', '--profile')

        if index_prepare is not None:
            if index_latest is None:
                raise errors.CliError(
                    "--latestCharts option must be used with --prepare")
        if index_latest is not None:
            if index_prepare is None:
                raise errors.CliError(
                    "--prepare option must be used with --latestCharts")
        if index_clean is not None:
            if index_prepare is not None:
                raise errors.CliError(
                    "--prepare option must not be used with --clean")
            if index_latest is not None:
                raise errors.CliError(
                    "--latestCharts option must not be used with --clean")
            if index_file is not None:
                raise errors.CliError(
                    "-o/--output option must be not used with --clean")
            if index_profile is not None:
                raise errors.CliError(
                    "-p/--profile option must be not used with --clean")
        if index_profile is not None:
            if index_prepare is not None:
                raise errors.CliError(
                    "--prepare option must not be used with -p/--profile")
            if index_latest is not None:
                raise errors.CliError(
                    "--latestCharts option must not be used with -p/--profile")

        docker_cmd = "%s mdt upgrade %s" % (DCK_CLI_CMD, self.resource)
        if index_file is not None:
            opt_file = optional_param[index_file + 1]
            # if not os.path.isfile(opt_file):
            #     raise errors.FileNotFound("file %s not found" % opt_file)
            opt_cont_file = "/tmp/upgrade.yaml"
            # Get file from container
            docker_cmd = "%s -o %s" % (docker_cmd, opt_cont_file)
        if index_prepare is not None and index_latest is not None:
            prepare = optional_param[index_prepare]
            latest = optional_param[index_latest]
            docker_cmd = "%s %s %s" % (docker_cmd, prepare, latest)
        if index_clean is not None:
            clean_option = optional_param[index_clean]
            docker_cmd = "%s %s" % (docker_cmd, clean_option)
        if index_profile is not None:
            # Check filename is defined
            if (index_profile + 1) >= len(optional_param):
                raise errors.CliError("profile name must be defined")
            profile_option = optional_param[index_profile]
            profile_name = optional_param[index_profile + 1]
            docker_cmd = "%s %s %s" % (
                docker_cmd, profile_option, profile_name)
        # ouput of docker cmd is redirected on stderr
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)
        if index_file is not None:
            opt_file = optional_param[index_file + 1]
            # copy file
            docker_cmd = "sudo docker cp mdt-cli:%s %s" % (
                opt_cont_file, opt_file)
            exec_cmd_shell(docker_cmd)
            # Clean file
            docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                DCK_CLI_CMD, opt_cont_file)
            exec_cmd_shell(docker_cmd)

        # Remove arguments who are processed
        if index_prepare is not None:
            index_prepare = get_index_of_arg(optional_param, '--prepare')
            del optional_param[index_prepare]
        if index_latest is not None:
            index_latest = get_index_of_arg(optional_param, '--latestCharts')
            del optional_param[index_latest]
        if index_file is not None:
            index_file = get_index_of_arg(optional_param, '-o', '--output')
            del optional_param[index_file + 1]
            del optional_param[index_file]
        if index_clean is not None:
            index_clean = get_index_of_arg(optional_param, '--clean')
            del optional_param[index_clean]
        if index_profile is not None:
            index_profile = get_index_of_arg(optional_param, '-p', '--profile')
            del optional_param[index_profile + 1]
            del optional_param[index_profile]

    # US 59826 - LRO
    def status(self, optional_param=None):
        """
        Get status of a release, or all releases for deploy package or all releases for a profile
        mdt [mdt_options] status products (-r RELEASE_NAME|-p PROFILE|-d DPYPKG_NAME:DPYPKG_VERSION) [-o OUTPUT_FILE]
        :param optional_param:
        :return:
        """
        index_release = get_index_of_arg(optional_param, '-r', '--release')
        index_deploypkg = get_index_of_arg(optional_param, '-d', '--deploy-package')
        index_profile = get_index_of_arg(optional_param, '-p', '--profile')
        index_file = get_index_of_arg(optional_param, '-o', '--output')
        opt_status_file = "/tmp/status.yaml"

        # exclusive options: -r, -d, -p
        if (index_release is not None and (index_deploypkg is not None or index_profile is not None)) or \
           (index_deploypkg is not None and (index_release is not None or index_profile is not None)) or \
           (index_profile is not None and (index_deploypkg is not None or index_release is not None)):
            raise errors.CliError(
                    "Exclusive options: --release or --deploy-package or --profile")

        if index_release is not None:
            opt_release = optional_param[index_release + 1]
            docker_cmd = "%s mdt status %s -r %s" % (DCK_CLI_CMD, self.resource, opt_release)
        elif index_deploypkg is not None:
            opt_deploypkg = optional_param[index_deploypkg + 1]
            docker_cmd = "%s mdt status %s -d %s" % (DCK_CLI_CMD, self.resource, opt_deploypkg)
        elif index_profile is not None:
            opt_profile = optional_param[index_profile + 1]
            docker_cmd = "%s mdt status %s -p %s" % (DCK_CLI_CMD, self.resource, opt_profile)
        else:
            docker_cmd = "%s mdt status %s " % (DCK_CLI_CMD, self.resource)

        if index_file is not None:
            opt_file = optional_param[index_file + 1]
            docker_cmd += " -o %s" % opt_status_file

        # Run mdt cli command
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)

        if index_file is not None:
            # copy file
            docker_cmd = "sudo docker cp mdt-cli:%s %s" % ( opt_status_file, opt_file)
            exec_cmd_shell(docker_cmd)

            # Clean file
            docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                DCK_CLI_CMD, opt_status_file)
            exec_cmd_shell(docker_cmd)

        # Remove arguments who are processed
        if index_release is not None:
            index_release = get_index_of_arg(optional_param, '-r', '--release')
            del optional_param[index_release + 1]
            del optional_param[index_release]
        if index_deploypkg is not None:
            index_deploypkg = get_index_of_arg(optional_param, '-d', '--deploy-package')
            del optional_param[index_deploypkg + 1]
            del optional_param[index_deploypkg]
        if index_profile is not None:
            index_profile = get_index_of_arg(optional_param, '-p', '--profile')
            del optional_param[index_profile + 1]
            del optional_param[index_profile]
        if index_file is not None:
            index_file = get_index_of_arg(optional_param, '-o', '--output')
            del optional_param[index_file + 1]
            del optional_param[index_file]

    def rollback(self, optional_param=None):
        """
        A command for doing a rollback on all the (upgraded)
        profiles or only a specific profile.
        :param optional_param:
        :return:
        """
        index = get_index_of_arg(optional_param, '-p', '--profile')

        docker_cmd = "%s mdt rollback %s " % (DCK_CLI_CMD, self.resource)
        if index is not None:
            # case option profile exists
            # Run mdt cli command
            profile = optional_param[index + 1]
            docker_cmd = "%s -p %s" % (docker_cmd, profile)
        # ouput of docker cmd is redirected on stderr
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)

        # Remove arguments who are processed
        if index is not None:
            index = get_index_of_arg(optional_param, '-p', '--profile')
            del optional_param[index + 1]
            del optional_param[index]


class Profile(object):
    """ Profile class
    """

    def __init__(self):
        self.resource = type(self).__name__.lower()

    def add(self, optional_param=None):
        """
        Add profile on node
        :param optional_param:
        :return:
        """

        index_node = get_index_of_arg(optional_param, '-n', '--node')
        if index_node is None:
            raise errors.CliError("-n option must be used")
        node_name = optional_param[index_node + 1]

        index_profile = get_index_of_arg(optional_param, '-p', '--profile')
        if index_profile is None:
            raise errors.CliError("-p option must be used")
        profile_name = optional_param[index_profile + 1]

        # Run mdt cli command
        docker_cmd = "%s mdt add %s -n %s -p %s" % (
            DCK_CLI_CMD, self.resource, node_name, profile_name)
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)

        # Remove arguments who are processed
        if index_node is not None:
            index_node = get_index_of_arg(optional_param, '-n', '--node')
            del optional_param[index_node + 1]
            del optional_param[index_node]
        if index_profile is not None:
            index_profile = get_index_of_arg(optional_param, '-p', '--profile')
            del optional_param[index_profile + 1]
            del optional_param[index_profile]

    def delete(self, optional_param=None):
        """
        Delete profile on node
        :param optional_param:
        :return:
        """

        index_node = get_index_of_arg(optional_param, '-n', '--node')
        if index_node is None:
            raise errors.CliError("-n option must be used")
        node_name = optional_param[index_node + 1]

        index_profile = get_index_of_arg(optional_param, '-p', '--profile')
        # if index_profile is None:
        #     raise errors.CliError("-p option must be used")

        docker_cmd = "%s mdt delete %s -n %s" % (
            DCK_CLI_CMD, self.resource, node_name)
        if index_profile is not None:
            profile_name = optional_param[index_profile + 1]
            docker_cmd = "%s -p %s" % (docker_cmd, profile_name)
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)

        # Remove arguments who are processed
        if index_node is not None:
            index_node = get_index_of_arg(optional_param, '-n', '--node')
            del optional_param[index_node + 1]
            del optional_param[index_node]
        if index_profile is not None:
            index_profile = get_index_of_arg(optional_param, '-p', '--profile')
            del optional_param[index_profile + 1]
            del optional_param[index_profile]

## 45690 - LRO
# Add chart commands for cli with chartmuseum


class Chart(object):
    """ Chart class

    """

    def __init__(self):
        self.resource = type(self).__name__.lower()

    def load(self, optional_param=None):
        """
        load chart(s)
        1)
        mdt load chart -d <dir>
        2)
        mdt load chart -f <file>
        2a: <file>=<chart>.tgz can be 1 chart only exported by helm
        2b: <file>=<mdt-chart>.tgz starting with mdt-chart is an archive of charts exported by helm

        ouput of docker cmd is redirected on stderr: (_, out, _) = exec_cmd_shell(docker_cmd)
        :param optional_param:
        :return:
        """
        # Check If external Helm repository is used, in that case, no load
        # enable
        helm_repo_path = get_helm_charts_server() + '/' + get_helm_charts_reponame()
        if helm_repo_path[0:4] == "http":
            raise errors.LoadError(
                "Load is forbidden for external Helm repository")

        index_file = get_index_of_arg(optional_param, '-f', '--file')
        index_dir = get_index_of_arg(optional_param, '-d', '--directory')
        # Error cases
        if index_file is not None and index_dir is not None:
            # 2 options are not concomitant
            raise errors.CliError(
                "mdt-cli command load chart failed: options -d and -f can't be used simultaneously")
        if index_file is None and index_dir is None:
            # at least one option must be present
            raise errors.CliError(
                "mdt-cli command load chart failed: option -d or -f is mandatory")

        if index_file is not None:
            try:
                opt_file = optional_param[index_file + 1]
                if os.path.isfile(opt_file):
                    # file name
                    file_name = os.path.basename(opt_file)
                    # Load file to container
                    docker_cmd = "sudo docker cp %s mdt-cli:/tmp/%s" % (
                        opt_file, file_name)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.debug("%s", out)
                    # Load chart file, cli checks if chart file or archive of
                    # charts
                    docker_cmd = "%s mdt load %s -f /tmp/%s " % (
                        DCK_CLI_CMD, self.resource, file_name)
                    if optional_param.count('--force') > 0:
                        index_force = get_index_of_arg(
                            optional_param, None, '--force')
                        del optional_param[index_force]
                        # Load chart file, cli checks if chart file or archive
                        # of charts
                        docker_cmd = "%s --force" % (docker_cmd)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.info("%s", out.replace(
                        '/tmp/' + file_name, opt_file))
                    index_file = get_index_of_arg(
                        optional_param, '-f', '--file')
                    del optional_param[index_file + 1]
                else:
                    # Case option file does not exist
                    LOGGER.error("load %s failed: file %s not found",
                                 self.resource, opt_file)
                    raise errors.FileNotFound(
                        "chart or chart archive not found")
            finally:
                # Clean folder
                try:
                    docker_cmd = "%s python3.6 -c \"import os; os.remove('/tmp/%s')\"" % (
                        DCK_CLI_CMD, file_name)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.debug("%s", out)
                except (SystemExit, NameError) as msg:
                    # file not found in container. No need to send a message
                    # for this
                    pass
            # Clean argument
            del optional_param[index_file]

        # Case Directory option
        if index_dir is not None:
            try:
                opt_dir = optional_param[index_dir + 1]
                opt_cont_dir = "/tmp/cfg_dir"
                if os.path.isdir(opt_dir):
                    # Load only mdt archive of charts package
                    if not os.path.exists('/tmp/chart_tmp'):
                        os.makedirs('/tmp/chart_tmp')
                    for cfile in glob.glob(opt_dir + "/mdt-charts_*.tgz"):
                        shutil.copy(cfile, '/tmp/chart_tmp/')
                    # Test if empty folder
                    if len(os.listdir('/tmp/chart_tmp/')) == 0:
                        LOGGER.info("No MDT chart package archive to load")
                        del optional_param[index_dir + 1]
                        del optional_param[index_dir]
                        return
                    # Load folder to container
                    docker_cmd = "sudo docker cp /tmp/chart_tmp/. mdt-cli:%s" % (
                        opt_cont_dir)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.debug("%s", out)
                    # Remove folder #bugfix 61824
                    if os.path.exists('/tmp/chart_tmp'):
                        shutil.rmtree('/tmp/chart_tmp/')
                    # Load chart folder, cli checks if chart file or archive of
                    # charts
                    docker_cmd = "%s mdt load %s -d %s " % (
                        DCK_CLI_CMD, self.resource, opt_cont_dir)
                    if optional_param.count('--force') > 0:
                        index_force = get_index_of_arg(
                            optional_param, None, '--force')
                        del optional_param[index_force]
                        docker_cmd = "%s --force" % (docker_cmd)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.info("%s", out.replace(opt_cont_dir, opt_dir))
                    index_file = get_index_of_arg(
                        optional_param, '-f', '--file')
                    del optional_param[index_dir + 1]
                else:
                    LOGGER.error(
                        "load %s failed: folder %s is not directory", self.resource, opt_dir)
                    raise errors.DirectoryNotFound(
                        "chart or chart archive not found")
            finally:
                # Clean folder
                try:
                    docker_cmd = "%s python3.6 -c \"import shutil; shutil.rmtree('%s')\"" % (
                        DCK_CLI_CMD, opt_cont_dir)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.debug("%s", out)
                except (SystemExit, NameError) as msg:
                    # file not found in container. No need to send a message
                    # for this
                    pass
            # Clean argument
            del optional_param[index_dir]

    def get(self, optional_param=None):
        """
        Get list of all the uploaded charts in the Helm repository
        mdt get chart [-o <output_file>]

        ouput of docker cmd is redirected on stderr: (_, out, _) = exec_cmd_shell(docker_cmd)
        :param optional_param:
        :return:
        """
        # Check If external Helm repository is used, in that case, no get
        # enable
        helm_repo_path = get_helm_charts_server() + '/' + get_helm_charts_reponame()
        if helm_repo_path[0:4] == "http":
            raise errors.LoadError(
                "Use Helm command to list charts of your external Helm repository")

        index = get_index_of_arg(optional_param, '-o', '--output')
        if index is not None:
            opt_cont_file = "/tmp/file.json"
            try:
                opt_file = optional_param[index + 1]
                # Get chart
                docker_cmd = "%s mdt get %s -o %s" % (
                    DCK_CLI_CMD, self.resource, opt_cont_file)
                (_, out, _) = exec_cmd_shell(docker_cmd)
                LOGGER.debug("%s", out)
                # Get file from container
                docker_cmd = "sudo docker cp mdt-cli:%s %s" % (
                    opt_cont_file, opt_file)
                (_, out, _) = exec_cmd_shell(docker_cmd)
                LOGGER.debug("%s", out)
                del optional_param[index + 1]
            finally:
                # Clean folder
                try:
                    docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                        DCK_CLI_CMD, opt_cont_file)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.debug("%s", out)
                except SystemExit as msg:
                    if "FileNotFoundError" in msg:
                        pass
                # Clean Argument
                del optional_param[index]
        else:

            # Get chart on console
            docker_cmd = "%s mdt get %s " % (DCK_CLI_CMD, self.resource)
            (_, out, _) = exec_cmd_shell(docker_cmd)
            LOGGER.info("%s", out)

    def delete(self, optional_param=None):
        """
        Delete chart(s)
        1)
        mdt delete chart [-n <name>:<version>]
        2)
        mdt delete chart [-f <chart>.tgz]

        ouput of docker cmd is redirected on stderr: (_, out, _) = exec_cmd_shell(docker_cmd)
        :param optional_param:
        :return:
        """
        # Check If external Helm repository is used, in that case, no delete
        # enable
        helm_repo_path = get_helm_charts_server() + '/' + get_helm_charts_reponame()
        if helm_repo_path[0:4] == "http":
            raise errors.LoadError(
                "Load is forbidden for external Helm repository")

        index_chart = get_index_of_arg(optional_param, '-n', '--name')
        index_file = get_index_of_arg(optional_param, '-f', '--file')
        # Error cases
        if index_chart is not None and index_file is not None:
            # 2 options are not concomitant
            raise errors.CliError(
                "mdt-cli command delete chart failed: options -n and -f can't be used simultaneously")
        if index_chart is None and index_file is None:
            # at least one option must be present
            raise errors.CliError(
                "mdt-cli command delete chart failed: option -n or -f is mandatory")
        # Case mdt delete chart [-n <name>:<version>]
        if index_chart is not None:
            chart = optional_param[index_chart + 1]
            # Delete chart file, cli checks if chart file or archive of
            # charts
            docker_cmd = "%s mdt delete %s -n %s " % (
                DCK_CLI_CMD, self.resource, chart)
            (_, out, _) = exec_cmd_shell(docker_cmd)
            LOGGER.info("%s", out)
            # Clean argument
            del optional_param[index_chart + 1]
            del optional_param[index_chart]

        #  Case mdt delete chart [-f <chart>.tgz]
        if index_file is not None:
            try:
                opt_file = optional_param[index_file + 1]
                if os.path.isfile(opt_file):
                    # file name
                    file_name = os.path.basename(opt_file)
                    # Load file to container
                    docker_cmd = "sudo docker cp %s mdt-cli:/tmp/%s" % (
                        opt_file, file_name)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.debug("%s", out)
                    # Delete chart file, cli checks if chart file or archive of
                    # charts
                    docker_cmd = "%s mdt delete %s -f /tmp/%s " % (
                        DCK_CLI_CMD, self.resource, file_name)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.info("%s", out.replace(
                        '/tmp/' + file_name, opt_file))
                    # Clean argument
                    del optional_param[index_file + 1]
                else:
                    # Case option file does not exist
                    LOGGER.error("load %s failed: file %s not found",
                                 self.resource, opt_file)
                    raise errors.FileNotFound(
                        "chart or chart archive not found")
            finally:
                # Clean folder
                try:
                    # Clean folder
                    docker_cmd = "%s python3.6 -c \"import os; os.remove('/tmp/%s')\"" % (
                        DCK_CLI_CMD, file_name)
                    (_, out, _) = exec_cmd_shell(docker_cmd)
                    LOGGER.debug("%s", out)
                except (SystemExit, NameError) as msg:
                    # file not found in container. No need to send a message
                    # for this
                    pass
            # Clean argument
            del optional_param[index_file]


class Context(object):
    """
    Label class
    """

    def __init__(self):
        self.resource = type(self).__name__.lower()

    def get(self, optional_param=None):
        """
        Get all the labels from all the nodes or from one node by defining
        the node name with the option -n. Labels are displayed in yaml format.
        If option -o is set, then the result will also be saved into the file.
        :param optional_param:
        :return:
        """
        index_file = get_index_of_arg(optional_param, '-o', '--output')
        opt_cont_file = "/tmp/context.yml"
        if index_file is not None:
            opt_file = optional_param[index_file + 1]
            # Get file from container
            docker_cmd = "%s mdt get %s -o %s" % (
                DCK_CLI_CMD, self.resource, opt_cont_file)
            exec_cmd_shell(docker_cmd)
            # copy file
            docker_cmd = "sudo docker cp mdt-cli:%s %s" % (
                opt_cont_file, opt_file)
            exec_cmd_shell(docker_cmd)
            # Clean file
            docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                DCK_CLI_CMD, opt_cont_file)
            exec_cmd_shell(docker_cmd)
        else:
            # Run mdt cli command
            docker_cmd = "%s mdt get %s" % (DCK_CLI_CMD, self.resource)
            # ouput of docker cmd is redirected on stderr
            (_, out, _) = exec_cmd_shell(docker_cmd)
            LOGGER.info("%s", out)

        # Remove arguments who are processed
        if index_file is not None:
            index_file = get_index_of_arg(optional_param, '-o', '--output')
            del optional_param[index_file + 1]
            del optional_param[index_file]


class Database(object):
    """
     Database class
    """

    def __init__(self):
        self.resource = type(self).__name__.lower()

    def backup(self, optional_param=None):
        """
        Get all the data stored in databases and save them into the specified
        database backup file. The output archive is created in the path
        given in command line with option -o.
        :param optional_param:
        :return:
        """
        index = get_index_of_arg(optional_param, '-o', '--output')
        if index is None:
            raise errors.CliError("-o option must be used")
        opt_file = optional_param[index + 1]
        opt_cont_file = "/tmp/file.tgz"
        # Run mdt cli command
        docker_cmd = "%s mdt backup %s -o %s" % (
            DCK_CLI_CMD, self.resource, opt_cont_file)

        exec_cmd_shell(docker_cmd)
        # Get file from container
        docker_cmd = "sudo docker cp mdt-cli:%s %s" % (
            opt_cont_file, opt_file)
        exec_cmd_shell(docker_cmd)
        # Clean folder
        docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
            DCK_CLI_CMD, opt_cont_file)
        exec_cmd_shell(docker_cmd)

        if index is not None:
            index = get_index_of_arg(optional_param, '-o', '--output')
            del optional_param[index + 1]
            del optional_param[index]

    def restore(self, optional_param=None):
        """
        Clear the databases and restore the data from the provided database
        backup file.
        Use the option --force to force the loading.
        :param optional_param:
        :return:
        """
        # Case Load with file parameter
        index = get_index_of_arg(optional_param, '-f', '--file')
        if index is None:
            raise errors.CliError("-f option must be used")
            # case option file exists
        try:
            opt_file = optional_param[index + 1]
            opt_cont_file = "/tmp/file.tgz"
            if os.path.isfile(opt_file):
                # Load file to container
                docker_cmd = "sudo docker cp %s mdt-cli:%s" % (
                    opt_file, opt_cont_file)
                exec_cmd_shell(docker_cmd)
                # Run mdt cli command
                docker_cmd = "%s mdt restore %s -f %s" % (
                    DCK_CLI_CMD, self.resource, opt_cont_file)
                if optional_param.count('--force') > 0:
                    index_force = get_index_of_arg(
                        optional_param, None, '--force')
                    del optional_param[index_force]
                    docker_cmd = "%s --force" % (docker_cmd)
                else:
                    if confirm("Are you sure you want to restore the database? The current configurations will be lost."):
                        docker_cmd = "%s --force" % (docker_cmd)
                (_, out, _) = exec_cmd_shell(docker_cmd)
                LOGGER.info("%s", out)
            else:
                # case option file does not exist
                LOGGER.error("load %s failed: file %s not found",
                             self.resource, opt_file)
                raise errors.FileNotFound("config bundle not found")
            index = get_index_of_arg(optional_param, '-f', '--file')
            del optional_param[index + 1]
            del optional_param[index]

        finally:
            # Clean folder
            try:
                docker_cmd = "%s python3.6 -c \"import os; os.remove('%s')\"" % (
                    DCK_CLI_CMD, opt_cont_file)
                exec_cmd_shell(docker_cmd)
            except (SystemExit, NameError) as msg:
                # file not found in container. No need to send a message
                # for this
                pass


def call_mdt_cli(operation, resource, args=[]):
    """
    Run docker or kubectl command to call mdt-cli
    :param operation: [ load, get ]
    :param resource: [ cfg, label ]
    :return:
    """
    try:
        LOGGER.info("calling %s operation for %s resource",
                    operation, resource)
        class_object = globals()[resource.title()]()
        getattr(class_object, operation)(args)
        LOGGER.info("done")
    except NameError as msg:
        raise errors.CliError("mdt-cli resource does not exist: %s" % msg)
    except AttributeError as msg:
        raise errors.CliError("mdt-cli operation is not allowed: %s" % msg)
    except (ValueError, OSError, SystemExit) as msg:
        raise errors.CliError("mdt-cli %s %s command failed: %s" %
                              (operation, resource, msg))
    except IndexError:
        docker_cmd = "%s mdt %s %s --help " % (
            DCK_CLI_CMD, operation, resource)
        (_, out, _) = exec_cmd_shell(docker_cmd)
        LOGGER.info("%s", out)
        raise errors.CliError(
            "mdt-cli %s %s failed: missing parameter value" % (operation, resource))
    except Exception as msg:
        raise errors.CliError("%s" % msg)
