#!/usr/bin/env python
#
# Copyright (c) 2013 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Jackie Li
# Created: Dec 30, 2013
# Description: A class for diff Content Class

import xml.etree.ElementTree as ET

from FileAssembler import ContentClassHandler

class ContentClassDiffer(object):
    uriIndent = "/"
    #===========================================================================
    # Compare src exported content class file against the targeted Content Class
    #===========================================================================
    @staticmethod   
    def compareContentClasses(srcContentClassFile):
        """
        contentClass
            |-attributes[]
            |-name
            |-description
            |-usage
            |-metadata
            |    |-attributes[]
            |    |-asset
            |        |-attributes[]
            |        |-fields[]
            |            |-attributes[]
            |            |-options[]
            |                |-option
            |                    |-displayName
            |                    |-value
            |-partners
                |-partner
                    |-name
        """       
        targetContentClassFile = '/tmp/contentClassTarget.xml'
        try:        
            ContentClassHandler.exportById(targetContentClassFile, 1, "admin", "admin")
        except Exception:
            print "Retry compare content class..."
            ContentClassHandler.exportById(targetContentClassFile, 1, "admin", "admin1234")
        result = dict()
        ContentClassDiffer.diffContentClassFiles(srcContentClassFile, targetContentClassFile, result)
       
    
    #===========================================================================
    # Diff two content class files
    #===========================================================================    
    @staticmethod  
    def diffContentClassFiles(srcContentClassFile, targetContentClassFile, result):
        different = False
        if result == None:
            result = dict()
        srcContentClass = ET.parse(srcContentClassFile)
        srcRoot = srcContentClass.getroot()
        
        targetContentClass = ET.parse(targetContentClassFile)
        targetRoot = targetContentClass.getroot()
        
        numberOfSpace = 36
        space = " " 
        indent = space * numberOfSpace
        print "##############################################################################################################"
        print "#                                       Content Class Diff Report                                            #"
        print "##############################################################################################################"
        print "Source Content Class File: %s" % srcContentClassFile
        print "Target Content Class File: %s" % targetContentClassFile
        print "--------------------------------------------------------------------------------------------------------------"
        print "Tag Uri"
        print "%sDescription"%indent
        parentUri = ""
        
        ContentClassDiffer.diffContentClassElement(srcRoot, targetRoot, parentUri, result) 
        for uri, description in result.items():
            print "%s" % uri
            print "%s%s" %(indent, description.encode("utf-8"))
        print "--------------------------------------------------------------------------------------------------------------"
        if result:
            different = True
        return different
    
    #===========================================================================
    # Diff two elements recursively
    #===========================================================================    
    @staticmethod  
    def diffContentClassElement(srcElement, targetElement, parentUri, result):
        """compare the element recursively 
        @param srcElement: The source element to compare
        @param targetElement: The target element to compare
        @param result: Where the compare result will be saved
        @param parentUri: The Uri of the parent of current compared element

        element
            |-tag
            |-attrib
            |-children
            |-text
        """
        tab = "    " 
        indent = tab * 9
        currentUri = None

        if srcElement.tag != targetElement.tag:
            print "SoftwareError:%ssrcElement: %s and targetElement: %s are not comparable" %(indent, parentUri+"/"+srcElement.tag,  parentUri+"/"+targetElement.tag)

        if srcElement.get("path") != None:
            if srcElement.tag == "field":
                currentUri = parentUri + ContentClassDiffer.uriIndent + srcElement.get("path")[1:]
            else: 
                currentUri = parentUri + ContentClassDiffer.uriIndent + srcElement.get("path")
        else:
            currentUri = parentUri + ContentClassDiffer.uriIndent + srcElement.tag     
        
        ContentClassDiffer.diffElementAttributes(srcElement, targetElement, result, currentUri)       
        
        if len(srcElement.getchildren())==0 and len(targetElement.getchildren())==0:  
            if srcElement.text !=  targetElement.text:
                "The case values are the general tag, i.e. <value>example</value>, <displayName>Producer</displayName>"
                result.update({currentUri+ContentClassDiffer.uriIndent+"Value Modified":"Value is different: src("+str(srcElement.text)+") vs target("+str(targetElement.text)+")"})
        else:            
            ContentClassDiffer.diffElementChildren(srcElement, targetElement, result, currentUri)
            
    #===========================================================================
    # Diff children elements
    #===========================================================================         
    @staticmethod
    def diffElementChildren(srcElement, targetElement, result, currentUri):
        """compare the children element  
        @param srcElement: The source element to compare
        @param targetElement: The target element to compare
        @param result: Where the compare result will be saved
        @param currentUri: The Uri of current compared element
        """
        
        srcElementChildren = dict()
        ContentClassDiffer.getChildrenElementMap(srcElement,srcElementChildren, currentUri)
            
        targetElementChildren = dict()  
        ContentClassDiffer.getChildrenElementMap(targetElement,targetElementChildren, currentUri)
        
        srcElementChildrenKeys = srcElementChildren.keys()
        targetElementChildrenKeys = targetElementChildren.keys()
        
        new_children, deleted_children, share_children = ContentClassDiffer.getDiffItems(srcElementChildrenKeys, targetElementChildrenKeys)
      
        ContentClassDiffer.updateResult(new_children, currentUri, "New Children Found", result)
        ContentClassDiffer.updateResult(deleted_children, currentUri, "Children Not Found", result)
                        
        for item in share_children:
            if item.endswith("/options"):
                ContentClassDiffer.diffOptionsTag(srcElementChildren.get(item), targetElementChildren.get(item), currentUri, result)
            elif item.endswith("/partners"):
                ContentClassDiffer.diffPartnersTag(srcElementChildren.get(item), targetElementChildren.get(item), currentUri, result)
            else:   
                ContentClassDiffer.diffContentClassElement(srcElementChildren.get(item), targetElementChildren.get(item), currentUri, result)
                
    @staticmethod
    def getChildrenElementMap(element,elementChildren, currentUri):
        """Get a map of <Uri, Child Element>
        @param parentElement: The element whose children will be elaborated into a map
        @param elementChildren:  The dict where the elaborated <Uri, Children Element> map will be saved
        @param currentUri: The Uri of current element
        """
        if elementChildren == None:
            elementChildren = dict()  
        for ch in element.getchildren():
            if ch.get("path") != None:
                if ch.tag == "field":
                    chUri = currentUri + ContentClassDiffer.uriIndent + ch.get("path")[1:]
                else:
                    chUri = currentUri + ContentClassDiffer.uriIndent + ch.get("path")
            else:
                if ch.tag == "options":
                    chUri = currentUri + ContentClassDiffer.uriIndent + ch.tag
                    "display:value"    
                    optionDict = dict()                 
                    ContentClassDiffer.getOptionsMap(optionDict, ch)
                    elementChildren.update({chUri:optionDict}) 
                    continue  
                if ch.tag == "partners":
                    chUri = currentUri + ContentClassDiffer.uriIndent + ch.tag
                    "name"    
                    partnerList = list()                 
                    ContentClassDiffer.getPartnerList(partnerList, ch)
                    elementChildren.update({chUri:partnerList}) 
                    continue  
                else:
                    chUri = currentUri + ContentClassDiffer.uriIndent + ch.tag                
            elementChildren.update({chUri:ch})  
        pass    
       
    @staticmethod
    def diffElementAttributes(srcElement, targetElement, result, currentUri):
        """Diff the attributes of the elements identified by the currentUri between soruce and target 
        @param srcElement: The source element to compare
        @param targetElement: The target element to compare
        @param result: Where the compare result will be saved
        @param currentUri: The Uri of current compared element
        """
        srcAttributes = srcElement.keys(); 
        targetAttributes = targetElement.keys();
        new_attributes, deleted_attributes, share_attributes = ContentClassDiffer.getDiffItems(srcAttributes, targetAttributes)
                
        ContentClassDiffer.updateResult(new_attributes, currentUri, "New Attributes Found", result)
        ContentClassDiffer.updateResult(deleted_attributes, currentUri, "Attributes Not Found", result)
                
        for item in share_attributes:
            if srcElement.get(item) != targetElement.get(item):
                result.update({currentUri+ContentClassDiffer.uriIndent+item:"attribute value is different: src("+str(srcElement.get(item))+ ") vs target("+str(targetElement.get(item))+")"})
                
    @staticmethod
    def getOptionsMap(optionsDict, element):
        """ Get the map of <displayName, value> of options
        @param optionDict: Where the map <displayName, value> will be saved
        @param element: The options element whose option will be retrieved 
        """
        "for each option in options"
        for ch in element.getchildren():
            "for each displayName:value pair"
            displayName = ""
            value = ""
            for grantCh in ch.getchildren():
                if grantCh.tag == "displayName":
                    displayName = grantCh.text
                elif grantCh.tag == "value":
                    value = grantCh.text                    
            optionsDict.update({displayName:value})
        
    @staticmethod
    def diffOptionsTag(srcOptionDict, targetOptionDict, currentUri, result):
        """ Diff two options map <displayName, value>
        @param srcOptionDict: The source options map to compare
        @param targetOptionDict: The target options map to compare
        @param currentUri: The Uri used to save in the result
        @param result: The result where will be saved
        """
        srcDisplyNames = srcOptionDict.keys()
        targetDisplayNames = targetOptionDict.keys()
        new_options, deleted_options, share_options = ContentClassDiffer.getDiffItems(srcDisplyNames, targetDisplayNames)

        ContentClassDiffer.updateResult(new_options, currentUri, "New Options Found", result)
        ContentClassDiffer.updateResult(deleted_options, currentUri, "Options Not Found", result)
                
        for item in share_options:
            if srcOptionDict.get(item) != targetOptionDict.get(item):
                result.update({currentUri+ContentClassDiffer.uriIndent+item:"option value is different: src("+str(srcOptionDict.get(item))+ ") vs target("+str(targetOptionDict.get(item))+")"})

    @staticmethod
    def getPartnerList(partnerList, element):
        """ Get the lsit of <partner name> of partners
        @param partnerList: Where the partner name list will be saved
        @param element: The partners element whose partner names will be retrieved 
        """
        "for each partner in partners"
        for ch in element.getchildren():
            "for each partner"
            partnerName = ""
            for grantCh in ch.getchildren():
                if grantCh.tag == "name":
                    partnerName = grantCh.text              
            partnerList.append(partnerName)
        pass
    
    @staticmethod
    def diffPartnersTag(srcPartnerList, targetPartnerList, currentUri, result):
        """ Diff two partner name list
        @param srcPartnerList: The source partner name list to compare
        @param targetPartnerList: The target partner name list to compare
        @param currentUri: The Uri used to save in the result
        @param result: The result where will be saved
        """
        new_partners, deleted_partners, share_partner = ContentClassDiffer.getDiffItems(srcPartnerList, targetPartnerList)        
        ContentClassDiffer.updateResult(new_partners, currentUri, "New Partners Found", result)
        ContentClassDiffer.updateResult(deleted_partners, currentUri, "Partners Not Found", result)
        pass
    
    @staticmethod
    def getDiffItems(srcItems, targetItems):
        """Get three diff item set new, deleted, shared items
        @param srcItems: The source items to compare
        @param targetItems: The target items to compare 
        """
        new_items = [item for item in targetItems if item not in srcItems]
        deleted_items = [item for item in srcItems if item not in targetItems]
        share_items = [item for item in srcItems if item not in deleted_items]
        return (new_items, deleted_items, share_items)
    
    @staticmethod
    def updateResult(items, currentUri, indicatorMsg, result):
        """This method will be called to save the new or deleted items into the result
        @param items:  The new or deleted children items to be save into the diff result
        @param currentUri: Identify the element whose children items will be save into result
        @param indicatorMsg: Indicates New/Deleted items (Option, Attributes or general children items)
        @param result: The result where will be saved
        """        
        if items:
            descriptionStr = None
            for item in items:
                descriptionStr = item if descriptionStr is None else descriptionStr +"," + item
            result.update({currentUri+ContentClassDiffer.uriIndent+indicatorMsg:descriptionStr})
        pass
