import os

class InstallComponent(object):
    KEYWORD_DELETE = "DELETE"
    KEYWORD_UPGRADE = "UPGRADE"
    KEYWORD_INSTALL = "INSTALL"
    ROOT_NAME = "ROOT"
    BASE_STUFF = "BASE"
    
    def __init__(self, componentName):
        self.componentName = componentName
        self.subComponents = []
        self.parentComponent = None
        self.keyWord = ""
        self.isFinalLeaf = False
        
    def hasSubComponent(self):
        return len(self.subComponents) > 0

    def hasSubComponentExceptBase(self):
        if self.getInstallComponentByName(self.BASE_STUFF):
            return len(self.subComponents) > 1
        else:
            return self.hasSubComponent()

    def addSubComponent(self, subComponent):
        self.subComponents.append(subComponent)
        subComponent.parentComponent = self
        return self
    
    def getSubComponents(self):
        return self.subComponents
        
    def appendBaseComponent(self):
        if not self.getInstallComponentByName(self.BASE_STUFF):
            self.addSubComponent(InstallComponent(self.BASE_STUFF))
        return self
        
    def getInstallComponentByName(self, name):
        for subComponent in self.getSubComponents():
            if subComponent.componentName == name:
                return subComponent
        return None    
        
    def printMe(self):
        print self.componentName + ": " + self.getConfigLocation()
        if self.subComponents:
            for subComponent in self.subComponents:
                subComponent.printMe()
    
    def setAsDelete(self):
        self.keyWord = InstallComponent.KEYWORD_DELETE
        
    def setAsUpgrade(self):
        self.keyWord = InstallComponent.KEYWORD_UPGRADE
        
    def setAsInstall(self):
        self.keyWord = InstallComponent.KEYWORD_INSTALL
    
    def isUpgrade(self):
        return self.keyWord == InstallComponent.KEYWORD_UPGRADE
    
    def isDelete(self):
        return self.keyWord == InstallComponent.KEYWORD_DELETE
    
    def isInstall(self):
        return self.keyWord == InstallComponent.KEYWORD_INSTALL
    
    def setAsFinalLeave(self):
        self.isFinalLeaf = True
    
    def isRoot(self):
        return self.componentName == self.ROOT_NAME
    
    def isBase(self):
        return self.componentName == self.BASE_STUFF
    
    def getConfigLocation(self):
        if self.parentComponent and self.parentComponent.componentName != self.ROOT_NAME:
            return self.parentComponent.getConfigLocation() + self.componentName + os.sep
        else: 
            return self.componentName + os.sep
    
    def substract(self, anotherComponent):
        return self.substractRecursively(anotherComponent).removeNonLeafComponentIfNoSubcomponents()
    
    def substractRecursively(self, anotherComponent):
        # use reserved index for loop in order to do correct removal
        if self.getSubComponents():
            for i in reversed(range(0, len(self.getSubComponents()))):
                subComponent = self.getSubComponents()[i]
                anotherSubComponent = anotherComponent.getInstallComponentByName(subComponent.componentName)
                if anotherSubComponent:
                    if subComponent.hasSubComponent():
                        subComponent.substractRecursively(anotherSubComponent)
                    else:
                        self.getSubComponents().remove(subComponent)
        return self
    
    def removeNonLeafComponentIfNoSubcomponents(self):
        if self.hasSubComponent():
            for i in reversed(range(0, len(self.getSubComponents()))):
                subComponent = self.getSubComponents()[i]
                subComponent.removeNonLeafComponentIfNoSubcomponents()
                if not subComponent.hasSubComponent() and not subComponent.isFinalLeaf:
                    self.getSubComponents().remove(subComponent)
        return self
    
    def merge(self, anotherComponent):
        for subComponent in anotherComponent.getSubComponents():
            if not self.getInstallComponentByName(subComponent.componentName):
                self.addSubComponent(subComponent.copy())
            else:
                self.getInstallComponentByName(subComponent.componentName).merge(subComponent)
        return self
    
    def intersect(self, anotherComponent):
        if self.hasSubComponent():
            for i in reversed(range(0, len(self.getSubComponents()))):
                subComponent = self.getSubComponents()[i]
                anotherSubComponent = anotherComponent.getInstallComponentByName(subComponent.componentName)
                if not anotherSubComponent:
                    self.getSubComponents().remove(subComponent)
                else:
                    subComponent.intersect(anotherSubComponent)
        return self
    
    def copy(self):
        copiedComponent = InstallComponent(self.componentName)
        copiedComponent.keyWord = self.keyWord
        copiedComponent.isFinalLeaf = self.isFinalLeaf
        for subComponent in self.getSubComponents():
            copiedComponent.addSubComponent(subComponent.copy())
        return copiedComponent
    
    def resolveLeaves(self, fullInsatllComponent=None):
        '''
            Resolve all final leaf nodes by the full components tree, if the fullInstallComponent is None, it indicates the object itself is a fullInstallComponent, the leaf nodes of it can be found out. 
        '''
        if not self.hasSubComponent():
            if not fullInsatllComponent or fullInsatllComponent.isFinalLeaf:
                self.setAsFinalLeave()
        else:
            for subComponent in self.getSubComponents():
                if fullInsatllComponent:
                    subComponent.resolveLeaves(fullInsatllComponent.getInstallComponentByName(subComponent.componentName))
                else: 
                    subComponent.resolveLeaves(None)
        return self
    
    def toJsonObject(self):
        jsonObj = {}
        jsonObj["componentName"] = self.componentName
        if self.subComponents:
            jsonObj["subComponents"] = []
            for subComponent in self.subComponents:
                jsonObj["subComponents"].append(subComponent.toJsonObject())
        return jsonObj
    
    @staticmethod
    def fromJsonObject(jsonObj):
        if not jsonObj:
            return None
        component = InstallComponent(jsonObj["componentName"])
        if jsonObj.has_key("subComponents") and jsonObj["subComponents"]:
            for subComponent in jsonObj["subComponents"]:
                component.addSubComponent(InstallComponent.fromJsonObject(subComponent))
        return component
    
    @staticmethod
    def createRootInstallComponentForList(componentList):
        rootInstallComponent = InstallComponent(InstallComponent.ROOT_NAME)
        if componentList:
            for component in componentList:
                rootInstallComponent.addSubComponent(component)
        return rootInstallComponent
    
    @staticmethod
    def createEmptyInstallComponent():
        return InstallComponent.createRootInstallComponentForList(None)
