import os
import copy
from InstallComponent import InstallComponent 
from PrepackInstalledDecider import PrepackInstalledDecider
from InstallComponentBuilder import InstallComponentBuilder
from InstallHelper import InstallHelper
from ComponentSelectionConfig import ComponentSelectionConfig
from Common import Common

class InstallComponentCollector(object):
    COMPONENT_SELECTION_CONFIG = "component-selections.json"
    UPGRADE_COMPONENTS_FILE_NAME = "upgrade-components.json"

    def __init__(self):
        self.helper = InstallHelper()
        self.decider = None
        if os.path.exists(self.COMPONENT_SELECTION_CONFIG):
            self.fullComponentSelectionConfig = ComponentSelectionConfig.fromJson(InstallHelper.loadFromJson(self.COMPONENT_SELECTION_CONFIG))
            self.fullInstallComponents = self.fullComponentSelectionConfig.constructInstallComponents().resolveLeaves()
    
    def getInstalledComponents(self):
        installedComponents = InstallComponent.createEmptyInstallComponent()
        
        if self.helper.hasInstalledComponents():
            installedComponents = InstallComponent.fromJsonObject(self.helper.loadInstalledComponents())
        else:
            decider = self.getPrepackInstalledDecider()
            legacyPrepackInstalled = decider.isAnyPrepackInstalled()
            if legacyPrepackInstalled:
                installedComponents = decider.getInstalledComponentsByPrepackName(decider.getInstalledPrepackName())
            #Save the installedComponents for legal prepack once the prepack is decided.
            self.helper.saveNeccessaryInfoOnceLegacyPrepackDecided(installedComponents.toJsonObject(),legacyPrepackInstalled)
        return installedComponents
    
    def getPrepackInstalledDecider(self):
        if not self.decider:
            self.decider = PrepackInstalledDecider()
        return self.decider
    
    #===========================================================================
    # getComponentsToDelete
    #===========================================================================
    def getComponentsToDelete(self, installedComponents):
        componentsToDelete = InstallComponent.createEmptyInstallComponent()
        if installedComponents.hasSubComponentExceptBase():
            installedDownsteams = installedComponents.copy().resolveLeaves().substract(self.fullInstallComponents)
            installedComponents = installedComponents.copy().resolveLeaves().substract(installedDownsteams)
            self.printInstalledComponents(installedComponents)
            installedcomponentSelection = ComponentSelectionConfig.constructFromInstallComponents(installedComponents, self.fullComponentSelectionConfig)
            InstallHelper.printSeperateLine()
            self.printSelectionInstructions("You can remove the installed components if they are no longer used.")
            toContinue = False
            while not toContinue:
                componentList = self.promptForSelection(installedcomponentSelection.rootSelection, installedcomponentSelection, "Select %s to remove", False)[0]
                componentsToDelete = InstallComponent.createRootInstallComponentForList(componentList)
                InstallHelper.printSeperateLine()
                print "Please confirm your selection of components to be removed: "
                self.printInstallComponents(componentsToDelete)
                toContinue = self.helper.promptYesOrNo("Type 'yes' to continue, type 'no' to reselect components to remove. Your input==> ")
        return componentsToDelete
            
        
    #===============================================================================
    # getComponentsToInstall 
    #===============================================================================
    def getComponentsToInstall(self, installedComponents):
        componentsToInstall = InstallComponent.createEmptyInstallComponent()
        availableInstallComponents = self.fullInstallComponents.copy().substract(installedComponents)
        if availableInstallComponents.hasSubComponentExceptBase():
            availableComponentSelection = ComponentSelectionConfig.constructFromInstallComponents(availableInstallComponents, self.fullComponentSelectionConfig)
            InstallHelper.printSeperateLine()
            self.printSelectionInstructions("Please select available components to install.")
            toContinue = False
            while not toContinue:
                selectedComponents = self.promptForSelection(availableComponentSelection.rootSelection, availableComponentSelection,"Select %s to install")[0]
                componentsToInstall = InstallComponent.createRootInstallComponentForList(selectedComponents)
                InstallHelper.printSeperateLine()
                print "Please confirm your selection of components to be installed: "
                self.printInstallComponents(componentsToInstall)
                toContinue = self.helper.promptYesOrNo("Type 'yes' to continue, type 'no' to reselect components to install. Your input==> ")
        else:
            InstallHelper.printSeperateLine()
            self.printSelectionInstructions("No other components available to install.")  
        return componentsToInstall
    
    #===============================================================================
    # processToCollectAllSelectionsWithLicense
    #===============================================================================
    def processToCollectAllSelections(self, preConfiguredFile=None, allowSelect=True, licensed_downstreams_paths=[]):
        installedComponents = self.getInstalledComponents()
        licensedComponents = InstallComponentBuilder().buildInstallComponents(licensed_downstreams_paths)
        componentsToDelete = InstallComponent.createEmptyInstallComponent()
        componentsToInstall = InstallComponent.createEmptyInstallComponent()
        installedDownsteams = InstallComponent.createEmptyInstallComponent()
        if (installedComponents.getInstallComponentByName('DOWNSTREAMS')):
            installedDownsteams.addSubComponent(installedComponents.getInstallComponentByName('DOWNSTREAMS'))
            
        reselect = False
        if self.helper.hasTempSelectedComponents() and not Common.isPatchMode():
            tempSelectedComponents = self.helper.loadTempSelectedComponents()
            #filter down streams
            todeletes = tempSelectedComponents["TO_DELETE"].get("subComponents")
            if todeletes:
                todeletes = filter(lambda x: x.get("componentName") != "DOWNSTREAMS", todeletes)
                tempSelectedComponents["TO_DELETE"]["subComponents"] = todeletes
            
            toinstalls = tempSelectedComponents["TO_INSTALL"].get("subComponents")
            if toinstalls:
                toinstalls = filter(lambda x: x.get("componentName") != "DOWNSTREAMS", toinstalls)
                tempSelectedComponents["TO_INSTALL"]["subComponents"] = toinstalls
                
            componentsToDelete = InstallComponent.fromJsonObject(tempSelectedComponents["TO_DELETE"])
            componentsToInstall = InstallComponent.fromJsonObject(tempSelectedComponents["TO_INSTALL"])
            InstallHelper.printSeperateLine()
            
            #merge down streams again
            downsteamsToDelete = installedDownsteams.copy().resolveLeaves().substract(licensedComponents)
            componentsToDelete.merge(downsteamsToDelete)
            downsteamToInstall = licensedComponents.copy().resolveLeaves().substract(installedDownsteams)
            componentsToInstall.merge(downsteamToInstall)
            
            print "The last installation failed with unexpected errors, however your selection have been saved, your last selections:"
            self.printAllUserSelections(componentsToDelete, componentsToInstall, installedComponents)
            allowSelect = not self.helper.promptYesOrNo("Type 'yes' to continue with last selections, type 'no' to re-select components.\nPlease noted: To re-select components, system will roll back the partially installed components of your last selections before installing your new selections. Your input ==> ")
            reselect = allowSelect
        elif preConfiguredFile:
            preConfiguredSelection = InstallHelper.loadFromJson(preConfiguredFile)
            #filter down streams
            if preConfiguredSelection.get("TO_INSTALL"):
                to_installs = filter(lambda x: not x.startswith('DOWNSTREAMS') , preConfiguredSelection.get("TO_INSTALL"))
                preConfiguredSelection["TO_INSTALL"] = to_installs
            if preConfiguredSelection.get("TO_DELETE"):
                to_installs = filter(lambda x: not x.startswith('DOWNSTREAMS') , preConfiguredSelection.get("TO_DELETE"))
                preConfiguredSelection["TO_DELETE"] = to_installs
            componentsToDelete, componentsToInstall = self.loadSelectionFromPreConfiguration(preConfiguredSelection, installedComponents)
        
        if not preConfiguredFile and allowSelect:
            componentsToDelete = self.getComponentsToDelete(installedComponents)
            componentsToInstall = self.getComponentsToInstall(installedComponents)
                
        downsteamsToDelete = installedDownsteams.copy().resolveLeaves().substract(licensedComponents)
        downstreams = downsteamsToDelete.getInstallComponentByName('DOWNSTREAMS')
        if downstreams and downstreams.hasSubComponent():
            print ''        
            print 'Below downsteams are not licensed, System will remove the unlicensed downstremas automatically'
            print ''
            self.printInstallComponents(downsteamsToDelete)
            print ''
            proceed = self.helper.promptYesOrNo("Type 'yes' to continue, type 'no' to skip the installation. Your input==> ")
            if not proceed:
                return False
        componentsToDelete.merge(downsteamsToDelete)
        downsteamToInstall = licensedComponents.copy().resolveLeaves().substract(installedDownsteams)
        componentsToInstall.merge(downsteamToInstall)
                
        if allowSelect:
            InstallHelper.printSeperateLine()
            print "You have selected below components to remove or install: "
            self.printAllUserSelections(componentsToDelete, componentsToInstall, installedComponents)
            proceed = self.helper.promptYesOrNo("Type 'yes' to continue, type 'no' to skip the installation. Your input==> ")
            if not proceed:
                return False
        
        if not installedComponents.hasSubComponent():
            componentsToInstall.appendBaseComponent()

        return (self.fullInstallComponents, installedComponents, componentsToDelete, componentsToInstall, reselect)
    
    def saveUserSelectionAsTemp(self, componentsToDelete, componentsToInstall):
        '''
            Save current user selection as temp, in case there is failure during the deployment, user would be allowed to rerun with the same selections again.
        '''
        saveSelections = {}
        saveSelections["TO_INSTALL"] = componentsToInstall.toJsonObject()
        saveSelections["TO_DELETE"] = componentsToDelete.toJsonObject()
        self.helper.saveUserSelectionAsTemp(saveSelections)
    
    def saveCurrentInstalledComponents(self, installedComponents, componentsToDelete, componentsToInstall):
        '''
            Save the current installed components after the deployment done.
        '''
        currentInstalledComponents = installedComponents.copy().substract(componentsToDelete).merge(componentsToInstall)
        self.helper.saveCurrentInstalledComponents(currentInstalledComponents.toJsonObject())
    
    def loadSelectionFromPreConfiguration(self,preConfiguredSelection,installedComponents):
        '''
            support two modes, DELTA mode should have install and delete items defined, FULL mode can have only install items defined, the items is all the user expect the system to have.
        '''
        
        builder = InstallComponentBuilder()
        componentsToDelete = InstallComponent.createEmptyInstallComponent()
        componentsToInstall = InstallComponent.createEmptyInstallComponent()
        mode = preConfiguredSelection["mode"]
        if mode == "DELTA":
            # Have to decide which components indeed need to be installed or deleted based on current installed components  
            if preConfiguredSelection.has_key("TO_INSTALL"):
                componentsToInstall = builder.buildInstallComponents(preConfiguredSelection["TO_INSTALL"]).resolveLeaves(self.fullInstallComponents).substract(installedComponents)
            if preConfiguredSelection.has_key("TO_DELETE"):
                componentsToDelete = builder.buildInstallComponents(preConfiguredSelection["TO_DELETE"]).resolveLeaves(self.fullInstallComponents).intersect(installedComponents).removeNonLeafComponentIfNoSubcomponents()
        else:
            fullCurrentComponents = builder.buildInstallComponents(preConfiguredSelection["TO_INSTALL"])
            fullCurrentComponents.appendBaseComponent()
            componentsToInstall = fullCurrentComponents.copy().resolveLeaves(self.fullInstallComponents).substract(installedComponents)
            componentsToDelete = installedComponents.copy().resolveLeaves(self.fullInstallComponents).substract(fullCurrentComponents)
        return (componentsToDelete, componentsToInstall)

    def printSelectionInstructions(self, primaryMessage):
        InstallHelper.printHighlightMessage([
            primaryMessage
#             "To select the components, under each menu you have to select all items you want." #better instructions here?
        ])

    def promptForSelection(self, selectionItems, componentSelectionConfig, selectMessage, includeBack = False, filterSelectionByDependency = True):
        '''
            prompt for selection recursively via the selection config.
            @param selectionItems            the    current selectionItems to handle, this is an array containing item names.
            @param componentSelectionConfig        the configuration that would be used to prompt selection
        '''
        installComponents = []
        allAvailalbeComponents = None
        for selectionItem in selectionItems:
            selectionConfig = componentSelectionConfig.getSelectionConfig(selectionItem)
            if selectionConfig:
                if filterSelectionByDependency:
                    availableComponents, autoSelectComponents = self.getAvailableComponentsAndAutoSelectComponents(componentSelectionConfig, [selectionItem])
                    if not allAvailalbeComponents:
                        allAvailalbeComponents = InstallComponent.createEmptyInstallComponent()
                    allAvailalbeComponents.merge(availableComponents)
                    
                installComponent = InstallComponent(selectionItem)
                if selectionConfig.selections:
                    selections = None
                    while selections is None:
                        userInput = self.constructPromptMessageAndGetInput(selectionConfig, componentSelectionConfig, selectMessage, includeBack)
                        if self.trim(userInput) == str(len(selectionConfig.selections) +1):
                            return None, componentSelectionConfig
                        selectedItems = self.collectSelectItems(userInput, selectionConfig)
                        selections, componentSelectionConfig = self.promptForSelection(selectedItems, componentSelectionConfig, selectMessage, True, filterSelectionByDependency)
                    for childComponent in selections:
                        installComponent.addSubComponent(childComponent)
                    if installComponent.hasSubComponent():
                        installComponents.append(installComponent)
                else:
                    installComponents.append(installComponent)
        return installComponents, componentSelectionConfig

    def constructPromptMessageAndGetInput(self, selectionConfig, componentSelectionConfig, selectMessage, includeBack):
        '''
            construct the prompt message with selections and get the value from user userInput.
            @param selectionConfig:    the current item to prompt for selection.
            @param componentSelectionConfig:    this full configuration that would be used to prompt selection.
            @param selectMessage            the customized message to prompt for selection.  
        '''
        
        message = selectMessage % selectionConfig.displayName + ":\n"
        singleSelectioinIndexes = []
        i = 1;
        for selection in selectionConfig.selections:
            subSelectionConfig = componentSelectionConfig.getSelectionConfig(selection)
            if subSelectionConfig:
                if subSelectionConfig.singleSelection:
                    singleSelectioinIndexes.append(i)
                selection = subSelectionConfig.displayName
            message = message + str(i) + ". " + selection + "; "
            i=i+1
        if includeBack:
            message = message + str(i) + ". " + self.fullComponentSelectionConfig.getBackDisplayName() + "; "
            singleSelectioinIndexes.append(i)
        InstallHelper.printShortSeperateLine()
        print message
        promptMessage = "Enter the number of the item you want to select, and then press ENTER. For multiple items, separate them by spaces. For example: 1 3. If none of the components are needed, press ENTER directly. Your input ===>  "
        userInput = None
        while userInput is None or not self.validateInput(userInput, len(selectionConfig.selections), singleSelectioinIndexes):
            userInput = self.helper.getRawInput(promptMessage)
        return userInput

    def validateInput(self,userInput, maxIndex, singleSelectioinIndexes):
        singleSelectioins = []
        otherSelected = False
        for selectIndex in userInput.split(" "):
            selectIndex = self.trim(selectIndex)
            if selectIndex:
                index = 0
                try:
                    index = int(selectIndex)
                except ValueError:
                    print "[ERROR] Invalid index number: " +  selectIndex
                    return False
                if index in singleSelectioinIndexes:
                    singleSelectioins.append(index)
                elif index > maxIndex + 1 or index <= 0:
                    print "[ERROR] Invalid index number: " +  selectIndex
                    return False
                else:
                    otherSelected = True
        if singleSelectioins and otherSelected:
            print "[ERROR] You can not select other items when item " + " or ".join([str(a) for a in singleSelectioins]) + " is selected. "
            return False
        return True

    def collectSelectItems(self, userInput, selectionConfig):
        selectedItems = []
        selectedIndexs = []
        for selectIndex in userInput.split(" "):
            selectIndex = self.trim(selectIndex) 
            if not selectIndex:
                continue
            
            if selectIndex not in selectedIndexs:
                selectedItems.append(selectionConfig.selections[int(selectIndex)-1])
                selectedIndexs.append(selectIndex)
        return selectedItems

    def getAvailableComponentsAndAutoSelectComponents(self, componentSelectionConfig, selectedItems):
        allAutoSelectComponents = self.filterByCurerntAvailableSelection(self.fullComponentSelectionConfig.collectAutoSelectComponentsForSelectedItems(selectedItems), componentSelectionConfig)
        availableInstallComponentsExcludeAutoSelected = self.filterByCurerntAvailableSelection(self.fullComponentSelectionConfig.collectAvailableComponentsExcludeAutoSelectComponents(selectedItems), componentSelectionConfig)
        return availableInstallComponentsExcludeAutoSelected, allAutoSelectComponents

    def filterByCurerntAvailableSelection(self, components, currentSelectionConfig):
        currentAvailableComponents = currentSelectionConfig.constructInstallComponents()
        components.resolveLeaves(self.fullInstallComponents).intersect(currentAvailableComponents).removeNonLeafComponentIfNoSubcomponents()
        return components

    def resolveAvailalbeComponentsWithDependencies(self, installedComponents, availableInstallComponents):
        self.resolveSingleSelection(installedComponents, availableInstallComponents)
        self.resolveAvaliableComponents(installedComponents, availableInstallComponents)
        return availableInstallComponents
    
    def resolveSingleSelection(self, installedComponents, availableInstallComponents):  
        for installedComponent in installedComponents.getSubComponents():
            selectionConfig = self.fullComponentSelectionConfig.getSelectionConfig(installedComponent.componentName)
            if selectionConfig:
                if selectionConfig.singleSelection:
                    if availableInstallComponents:
                        availableInstallComponents.subComponents = []
                else:
                    if availableInstallComponents:
                        self.removeSingleSelectionComponents(availableInstallComponents)
                self.resolveSingleSelection(installedComponent, (availableInstallComponents.getInstallComponentByName(installedComponent.componentName) if availableInstallComponents else None))
        if availableInstallComponents:
            availableInstallComponents.removeNonLeafComponentIfNoSubcomponents()
    
    def removeSingleSelectionComponents(self, availableInstallComponents):
        for i in reversed(range(0, len(availableInstallComponents.getSubComponents()))):
            subComponent = availableInstallComponents.getSubComponents()[i]
            selectionConfig = self.fullComponentSelectionConfig.getSelectionConfig(subComponent.componentName)
            if selectionConfig.singleSelection: 
                availableInstallComponents.getSubComponents().remove(subComponent)
                        
    def resolveAvaliableComponents(self, installedComponents, availableInstallComponents):
        availableComponentsForInstalledComponents = None
        for installedComponent in installedComponents.getSubComponents():
            selectionConfig = self.fullComponentSelectionConfig.getSelectionConfig(installedComponent.componentName)
            if selectionConfig:
                if selectionConfig.availableComponents or selectionConfig.unAvailableComponents:
                    if not availableComponentsForInstalledComponents:
                        availableComponentsForInstalledComponents = InstallComponent.createEmptyInstallComponent()
                    availableComponentsForInstalledComponents.merge(self.fullComponentSelectionConfig._collectAvailableComponents(selectionConfig))    
                availableInstallComponents = self.resolveAvaliableComponents(installedComponent, availableInstallComponents)
        if availableComponentsForInstalledComponents:
            return availableInstallComponents.resolveLeaves(self.fullInstallComponents).intersect(availableComponentsForInstalledComponents).removeNonLeafComponentIfNoSubcomponents()
        else:
            return availableInstallComponents

    def trim(self, string):
        return string.strip('\n\t ')

    def printInstalledComponents(self, installedComponents):
        '''
        print the installed components
        @param installedComponents:        installedComponents json file. 
        '''
        InstallHelper.printSeperateLine()
        print 'Below Components has been installed on current system: '
        self.printInstallComponents(installedComponents)

    def printAllUserSelections(self,componentsToDelete, componentsToInstall, installedComponents):
        InstallHelper.printHighlightMessage("Components to delete: ")
        self.printInstallComponents(componentsToDelete)
        print ""
        InstallHelper.printHighlightMessage("Components to install: ")
        self.printInstallComponents(componentsToInstall)
        print ""
        InstallHelper.printHighlightMessage("Components to keep")
        self.printInstallComponents(installedComponents.copy().resolveLeaves(self.fullInstallComponents).substract(componentsToDelete))
        print ""
        InstallHelper.printHighlightMessage("After the installation, system will have below components:")
        self.printInstallComponents(installedComponents.copy().resolveLeaves(self.fullInstallComponents).substract(componentsToDelete).merge(componentsToInstall))
        print ""
        InstallHelper.printSeperateLine()
    
    def printInstallComponents(self, components, intent=""):
        if components.isRoot() and not components.hasSubComponent():
            print "    No Component is selected" 
        if not components.isRoot() and not components.isBase():
            intent = intent + '--|-'
            print intent + self.fullComponentSelectionConfig.getSelectionConfig(components.componentName).displayName
        for subComponent in components.getSubComponents():
            self.printInstallComponents(subComponent, intent)
    
    def collectAllUpgradeComponents(self, pathPrefix, upgradePath):
        '''
            decide all upgrade components in current upgrade folder. either by looping the folder structure or have an config file for it.
        '''
        upgradeComponentsFilePath = ComponentPathBuilder.getComponentsPathPrefix(pathPrefix, upgradePath) + self.UPGRADE_COMPONENTS_FILE_NAME
        if os.path.exists(upgradeComponentsFilePath):
            return self.collectAllUpgradeComponentsViaConfigFile(upgradeComponentsFilePath)
        else:
            return self.collectAllUpgradeComponentsViaDirectories(pathPrefix, upgradePath)
    
    def collectAllUpgradeComponentsViaConfigFile(self, upgradeComponentsFilePath):
        upgradeComponents = InstallHelper.loadFromJson(upgradeComponentsFilePath)
        return InstallComponentBuilder().buildInstallComponents(upgradeComponents)
    
    def collectAllUpgradeComponentsViaDirectories(self, pathPrefix, upgradePath):
        availableUpgradeComponents = self.collectAllUpgradeComponentsViaDirectoryRecursively(self.getInstalledComponents(), pathPrefix, upgradePath)
        if not availableUpgradeComponents:
            availableUpgradeComponents = InstallComponent.createEmptyInstallComponent()
        return availableUpgradeComponents
        
    def collectAllUpgradeComponentsViaDirectoryRecursively(self, fullInstallComponents, pathPrefix, upgradePath):
        availableUpgradeComponent = InstallComponent(fullInstallComponents.componentName)
        if fullInstallComponents.hasSubComponent():
            for subInstallComponent in fullInstallComponents.getSubComponents():
                availableSubUpgradeComponent = self.collectAllUpgradeComponentsViaDirectoryRecursively(subInstallComponent, pathPrefix, upgradePath)
                if availableSubUpgradeComponent:
                    availableUpgradeComponent.addSubComponent(availableSubUpgradeComponent)
        if not availableUpgradeComponent.hasSubComponent() and not self.checkComponentExists(pathPrefix, upgradePath, fullInstallComponents):
            return None
        else:
            return availableUpgradeComponent
            
    def checkComponentExists(self, pathPrefix, upgradePath, component):
        return os.path.exists(ComponentPathBuilder.getComponentsPathPrefix(pathPrefix, upgradePath) + component.getConfigLocation() + "config.json")

class ComponentPathBuilder(object):
    COMPONENTS_FOLDER = "components"
    @staticmethod
    def getComponentsPathPrefix(pathPrefix, upgradePath):
        if pathPrefix:
            pathPrefix = pathPrefix + os.sep
        else:
            pathPrefix = ""
        if upgradePath:
            pathPrefix = pathPrefix + upgradePath + os.sep
        return pathPrefix + ComponentPathBuilder.COMPONENTS_FOLDER + os.sep
    
if __name__ == "__main__":
    collector = InstallComponentCollector()
    collector.printInstallComponents(collector.getInstalledComponents())
    
