#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Jerish Lin
# Created: Nov 4, 2013
# Description:
# Helper Class to check versions, etc.

import os
import time
import json
import re
import shutil
from PrepackInstalledDecider import PrepackInstalledDecider

class InstallHelper(object):
    PREPACK_DIR = "/opt/tandbergtv/cms/prepack"
    BACKUP_DIR = PREPACK_DIR + os.sep + "backup-restore"
    BACKUP_ABANDON_DIR = BACKUP_DIR + os.sep + "abandoned"
    ROLLBACK_DIR = BACKUP_DIR + os.sep + "rollback"
    
    PREPACK_VERSION_FILE = PREPACK_DIR + os.sep + "prepackVersion.dat"
    PATCH_VERSION_FILE = PREPACK_DIR + os.sep + "patchVersion.dat"
    BACKUP_VERSION_FILE = BACKUP_DIR + os.sep + "backupVersion.dat"
    LAST_HANDLED_BACKUP_VERSION_FILE = BACKUP_DIR + os.sep + "lastHandledBackupVersion.dat"
    INSTALL_ACTION_LOG = PREPACK_DIR + os.sep + "installAction.log"
    INSTALL_STATUS_FILE = "installStatus.log"
    
    SELECTED_COMPONENTS_TEMP = PREPACK_DIR + os.sep + "selected-component-temp.json"
    INSTALLED_COMPONENTS_FILE_NAME = "installedcomponent.json"
    INSTALLED_COMPONENTS = PREPACK_DIR + os.sep + INSTALLED_COMPONENTS_FILE_NAME
    INSTALLATION_TEMP = PREPACK_DIR + os.sep + "installation_temp"
    COMPONENTS_TO_HANDLE_COMBINED_CONFIG_FILE_NAME = "combined-config.json"
    CMS_FILE_BACKUP_DIR = "cmsFiles"
    PREPACK_INFO_BACKUP_DIR = "prepackInfo"
    FLAG_EVER_IMP_FILE_PATH = PREPACK_DIR + os.sep +'content_class_ever_imported.dat'
    
    ROLLBACK_JSON = "rollback.json"
    
    DEPLOY = "Deploy"
    BACKUP = "Backup"
    ROLLBACK = "Rollback"
    START = "Start"
    FINISH = "Finish"
    
    LEGACY_PREPACK_DUMMY_VERSION = "0.5"
    
    def __init__(self):
        self.decider = None
        self.isLegacyPrepackInstalled = None
    
    def __overWriteFile(self, text, filePath):
        d = os.path.dirname(filePath)
        if not os.path.isdir(d):
            os.makedirs(d)
        with open(filePath, "w") as f:
            f.write(text)
            
    def __appendToFileAsNewLine(self, text, filePath):
        d = os.path.dirname(filePath)
        if not os.path.isdir(d):
            os.makedirs(d)
        newline = ''
        if os.path.exists(filePath) and open(filePath, 'r').read():
            newline = '\n'
        with open(filePath, "a") as f:
            f.write(newline + text)
    
    def __grepFile(self, words, filePath):
        foundRecords = []
        if os.path.exists(filePath):
            allRecords = open(filePath, 'r').readlines()
            for record in allRecords:
                found = True       
                for word in words:
                    if record.find("[" + word + "]") == -1:
                        found = False
                        break
                if found:
                    foundRecords.append(record)
        return foundRecords
    
    def __removeLinesWithWords(self, words, filePath):
        if os.path.exists(filePath):
            allRecords = open(filePath, 'r').readlines()
            f = open(filePath, 'w')
            for record in allRecords:
                found = True       
                for word in words:
                    if record.find("[" + word + "]") == -1:
                        found = False
                        break
                if not found:
                    f.write(record)

    def __readLastLine(self, filePath):
        if os.path.exists(filePath):
            alllines = open(filePath, 'r').readlines()
            if len(alllines) > 0:
                return alllines[len(alllines) - 1].replace('\n', '')
        return None
    
    def __removeLastLine(self, filePath):
        if os.path.exists(filePath):
            alllines = open(filePath, 'r').readlines()
            f = open(filePath, 'wb')
            for i in range(0, len(alllines) - 1):
                if i == len(alllines) - 2:
                    f.write(alllines[i].replace('\n', ''))
                else:    
                    f.write(alllines[i])
    
    def getPrepackInstalledDecider(self):
        if not self.decider:
            self.decider = PrepackInstalledDecider()
        return self.decider
    
    def saveNewPrepackVersion(self, version):
        if self.isVersionPatchFormat(version):
            if version != self.__readLastLine(self.PATCH_VERSION_FILE):
                self.__appendToFileAsNewLine(version, self.PATCH_VERSION_FILE)
        elif self.isVersionUpgradeFormat(version):
            self.__overWriteFile(self.getUpgradeFromTo(version)[1], self.PREPACK_VERSION_FILE)
        else:
            self.__overWriteFile(version, self.PREPACK_VERSION_FILE)
    
    def saveNewBackupVersion(self, backupVersion):
        self.__overWriteFile(backupVersion, self.BACKUP_VERSION_FILE)
        
    def saveLastHandledBackupVersion(self, backupVersion):
        if backupVersion != self.__readLastLine(self.LAST_HANDLED_BACKUP_VERSION_FILE):
            self.__appendToFileAsNewLine(backupVersion, self.LAST_HANDLED_BACKUP_VERSION_FILE)
    
    def getLastHandledBackupVersion(self):
        return self.__readLastLine(self.LAST_HANDLED_BACKUP_VERSION_FILE)
    
    def removeLastHandledBackupVersion(self):
        self.__removeLastLine(self.LAST_HANDLED_BACKUP_VERSION_FILE)
    
    def __getPrepackVersion(self, prepackVersionFile):
        prePackVersion = self.__readLastLine(prepackVersionFile)
        if (not prePackVersion or float(prePackVersion) < 3.0) and not self.hasInstalledComponents() and self.getIsAnyLegecyPrepackInstalled():
            return self.LEGACY_PREPACK_DUMMY_VERSION
        return prePackVersion
    
    def getCurrentPrepackVersion(self):
        return self.__getPrepackVersion(self.PREPACK_VERSION_FILE)
    
    def getBackupPrepackVersion(self, backupVersion):
        backupDir = self.getPrepackInfoBackupDir(backupVersion)
        return self.__getPrepackVersion(backupDir + self.PREPACK_VERSION_FILE)
    
    def getIsAnyLegecyPrepackInstalled(self):
        if self.isLegacyPrepackInstalled is None:
            self.isLegacyPrepackInstalled = self.getPrepackInstalledDecider().isAnyPrepackInstalled()
        return self.isLegacyPrepackInstalled
   
    def __getPatchesForVersion(self, patchVersionFile, prepackVersion):
        matchedPatches = []
        if prepackVersion:
            if os.path.exists(patchVersionFile):
                allPatches = open(patchVersionFile, 'r').readlines()
                allPatches.reverse()
                for patchVersion in allPatches:
                    if self.getVersionMain(patchVersion) == prepackVersion:
                        matchedPatches.append(patchVersion.replace('\n', ''))
        matchedPatches.reverse()
        return matchedPatches
    
    def getPatchesForCurrentPrepackVersion(self):
        return self.__getPatchesForVersion(self.PATCH_VERSION_FILE, self.getCurrentPrepackVersion())
    
    def getPatchesForBackupPrepackVersion(self, backupVersion):
        backupDir = self.getPrepackInfoBackupDir(backupVersion)
        return self.__getPatchesForVersion(backupDir + os.sep + self.PATCH_VERSION_FILE, self.getBackupPrepackVersion(backupVersion))
    
    def getLatestBackupVersion(self):
        return self.__readLastLine(self.BACKUP_VERSION_FILE)
    
    def rollBackToLastVersion(self, backupVersion):
        self.moveFolderAsHistory(backupVersion, self.ROLLBACK_DIR)
        self.removeLastHandledBackupVersion()
    
    def abandonCurrentBackup(self, backupVersion):
        self.moveFolderAsHistory(backupVersion, self.BACKUP_ABANDON_DIR)
        self.removeLastHandledBackupVersion()
        
    def getBackupDir(self, backupVersion):
        return self.BACKUP_DIR + os.sep + backupVersion
    
    def getFileBackupDir(self, backupVersion):
        return self.getBackupDir(backupVersion) + os.sep + self.CMS_FILE_BACKUP_DIR
    
    def getPrepackInfoBackupDir(self, backupVersion):
        return self.getBackupDir(backupVersion) + os.sep + self.PREPACK_INFO_BACKUP_DIR
    
    def moveFolderAsHistory(self, folder, targetDir):
        currentBackupDir = self.getBackupDir(folder)
        if not os.path.exists(currentBackupDir):
            return
        currenttime = time.strftime('%Y%m%d%H%M%S')
        if not os.path.exists(targetDir):
            os.makedirs(targetDir)
        shutil.move(currentBackupDir, targetDir + os.sep + folder + "_" + currenttime)
    
    def hasInstalledComponents(self):
        return os.path.exists(self.INSTALLED_COMPONENTS)
    
    def loadInstalledComponents(self):
        return self.loadFromJson(self.INSTALLED_COMPONENTS)

    def loadBackupInstalledComponents(self, backupVersion):
        backupDir = self.getPrepackInfoBackupDir(backupVersion)
        return self.loadFromJson(backupDir + self.INSTALLED_COMPONENTS)

    def saveUserSelectionAsTemp(self, userSelections):
        self.writeJsonToFile(userSelections, self.SELECTED_COMPONENTS_TEMP)
    
    def saveCurrentInstalledComponents(self, currentInstalledComponents):
        self.writeJsonToFile(currentInstalledComponents, self.INSTALLED_COMPONENTS)
        if os.path.exists(self.SELECTED_COMPONENTS_TEMP):
            os.remove(self.SELECTED_COMPONENTS_TEMP)
    
    def hasTempSelectedComponents(self):
        return os.path.exists(self.SELECTED_COMPONENTS_TEMP)
    
    def loadTempSelectedComponents(self):
        return self.loadFromJson(self.SELECTED_COMPONENTS_TEMP)
    
    def getVersionMain(self, version):
        if version:
            return re.match("\d+\.\d+", version).group(0)  
    
    def isVersionUpgradeFormat(self, version):
        return re.match("\d+\.\d+-\d+\.\d+$", version) is not None
    
    def isVersionPatchFormat(self, version):
        return re.match("\d+\.\d+\.(\d|\.)*$", version) is not None
    
    def isVersionFreshInstallFormat(self, version):
        return re.match("\d+\.\d+$", version) is not None
    
    def isCurrentVersionFromUpgrade(self):
        if os.path.exists(InstallHelper.BACKUP_DIR):
            for dirName in os.listdir(os.path.abspath(InstallHelper.BACKUP_DIR)):
                index = dirName.find("@")
                if index > -1:
                    if self.isVersionUpgradeFormat(dirName[0:index]):
                        return True
        return False
    
    def getUpgradeFromTo(self, version):
        index = version.find("-")
        if index == -1:
            return (version, version)
        else:
            return (version[0:index], version[index + 1:len(version)])
        
    def compareVersion(self, version1, version2):
        currentVersionFloat = float(self.getVersionMain(version1))
        configVersionFloat = float(self.getVersionMain(version2))
        if currentVersionFloat == configVersionFloat:
            return 0
        if currentVersionFloat > configVersionFloat:
            return -1
        return 1    
    
    def getBackupDeployStatusForVersion(self, backupVersion):
        if not backupVersion:
            return (False, False, False, False)
        installStatusFile = self.getInstallStatusFile(backupVersion)
        if not os.path.exists(installStatusFile):
            return (False, False, False, False)
        backupStarted = len(self.__grepFile([self.BACKUP, self.START], installStatusFile)) > 0
        backupFinished = len(self.__grepFile([self.BACKUP, self.FINISH], installStatusFile)) > 0
        deployStarted = len(self.__grepFile([self.DEPLOY, self.START], installStatusFile)) > 0
        deployFinished = len(self.__grepFile([self.DEPLOY, self.FINISH], installStatusFile)) > 0
        return (backupStarted, backupFinished, deployStarted, deployFinished)
            
    def solveBackupVersion(self, installerVersion):
        if self.isInstallerRerun(installerVersion):
            return installerVersion + "_rerun"
        return installerVersion
    
    def isInstallerRerun(self, installerVersion):
        return not self.isVersionPatchFormat(installerVersion) and self.getFinalInstallerVersion(installerVersion) == self.getCurrentPrepackVersion()
    
    def isDifferentInstaller(self, backupVersion, installerVersion):
        index = backupVersion.find("@")
        return backupVersion[0:index] != installerVersion
    
    def getFinalInstallerVersion(self, installerVersion):
        if self.isVersionUpgradeFormat(installerVersion):
            return self.getUpgradeFromTo(installerVersion)[1]
        return installerVersion
    
    def appenTimeStampToBackupVersion(self, backupVersion):
        return backupVersion + "@" + time.strftime('%Y-%m-%d-%H-%M-%S')
    
    def saveCombinedConfigObject(self, configObject):
        self.writeJsonToFile(configObject, self.INSTALLATION_TEMP + os.sep + self.COMPONENTS_TO_HANDLE_COMBINED_CONFIG_FILE_NAME + "_" + time.strftime('%Y-%m-%d-%H-%M-%S'))
    
    def getPrepackVersionInfoFiles(self):
        return [self.BACKUP_VERSION_FILE, self.PREPACK_VERSION_FILE, self.PATCH_VERSION_FILE, self.INSTALLED_COMPONENTS]
    
    def logActions(self, step, version, action):
        self.__appendToFileAsNewLine(time.strftime('%Y-%m-%d-%H-%M-%S') + ": [" + step + "] at Version [" + version + "] [" + action + "]", self.INSTALL_ACTION_LOG)
        self.logInstallStatus(step, version, action)
    
    def logInstallStatus(self, step, version, action): 
        installStatusFile = self.getInstallStatusFile(version)
        InstallHelper.mkdirForFile(installStatusFile)
        if not len(self.__grepFile([step, action], installStatusFile)):
            self.__appendToFileAsNewLine("[" + step + "] [" + action + "]", installStatusFile)

    def getInstallStatusFile(self, backupVersion):
        return self.getBackupDir(backupVersion) + os.sep + self.INSTALL_STATUS_FILE

    def getRawInput(self, message):
        try:
            return raw_input(message)
        except (IOError, EOFError):
            return self.getRawInput("")

    def promptYesOrNo(self, message):
        userInput = ''
        while userInput != 'yes' and userInput != 'no':
            userInput = self.getRawInput(message)
        return userInput == 'yes'
    
    def contentClassEverImported(self):
        return os.path.exists(self.FLAG_EVER_IMP_FILE_PATH)
    
    def writeContentClassImportedFlagFile(self):
        if not os.path.exists(self.FLAG_EVER_IMP_FILE_PATH):
            InstallHelper.mkdirForFile(self.FLAG_EVER_IMP_FILE_PATH)
            with open(self.FLAG_EVER_IMP_FILE_PATH, "w") as text_file:
                text_file.write("Please DON'T remove this file once CMS installation is done! \n")
    
    def saveNeccessaryInfoOnceLegacyPrepackDecided(self, installedComponents, legacyPrepackInstalled):
        self.saveCurrentInstalledComponents(installedComponents)
        if legacyPrepackInstalled:
            self.writeContentClassImportedFlagFile()
        self.saveNewPrepackVersion(self.LEGACY_PREPACK_DUMMY_VERSION if legacyPrepackInstalled else "")
    
    @staticmethod
    def loadFromJson(jsonFile):
        if not os.path.exists(jsonFile):
            return None
        json_data = open(jsonFile)
        jsonObject = json.load(json_data)
        json_data.close()
        return jsonObject  
    
    @staticmethod
    def writeJsonToFile(jsonObject, jsonFile):
        InstallHelper.mkdirForFile(jsonFile)
        with open(jsonFile, 'w') as f:
            json.dump(jsonObject, f, ensure_ascii=False, indent=4)
            
    @staticmethod
    def mkdirForFile(path):
        dirPath = os.path.dirname(path)
        if not os.path.isdir(dirPath):
            os.makedirs(dirPath)
            
    @staticmethod
    def printHighlightMessage(message):
        holderLength = 0
        if type(list()) == type(message):
            for m in message:
                if len(m) > holderLength:
                    holderLength = len(m)
            print "#"*(holderLength+4)
            for m in message:
                print "# " + m + " "*(holderLength-len(m)+1) + "#"
            print "#"*(holderLength+4)
        else:
            print "#"*(len(message)+4)
            print "# " + message +" #"
            print "#"*(len(message)+4) 
    
    @staticmethod
    def printSeperateLine():
        print "*"*150 
        print ""
     
    @staticmethod   
    def printShortSeperateLine():
        print "-"*100
        print ""