import unittest 
import os
from InstallTestBase import InstallTestBase
from InstallHelper import InstallHelper

class InstallHelperTest(InstallTestBase):
    TEST_TEMP_DIR = "test_temp"
    
    def setUp(self):
        InstallTestBase.setUp(self)
        self.helper = InstallHelper()
    
    def testSaveNewPrepackVersionForInstallVersion(self):
        patchVersion = "1.0"
        self.helper.saveNewPrepackVersion(patchVersion)
        lines = self._readLines(InstallHelper.PREPACK_VERSION_FILE)
        self.assertEqual(1, len(lines))
        self.assertEqual("1.0", lines[0])

    def testSaveNewPrepackVersionForUpgrade(self):
        self._writeToFile("1.0", InstallHelper.PREPACK_VERSION_FILE)
        patchVersion = "1.0-2.0"
        self.helper.saveNewPrepackVersion(patchVersion)
        lines = self._readLines(InstallHelper.PREPACK_VERSION_FILE)
        self.assertEqual(1, len(lines))
        self.assertEqual("2.0", lines[0])
        patchLines = self._readLines(InstallHelper.PATCH_VERSION_FILE)
        self.assertTrue(not patchLines)
    
    def testSaveNewPrepackVersionForPatchVersion(self):
        self._writeToFile("1.0", InstallHelper.PREPACK_VERSION_FILE)
        
        self.helper.saveNewPrepackVersion("1.0.001")
        patchLines = self._readLines(InstallHelper.PATCH_VERSION_FILE)
        self.assertEqual(1, len(patchLines))
        self.assertEqual("1.0.001", patchLines[0])
        
        self.helper.saveNewPrepackVersion("1.0.001")
        patchLines = self._readLines(InstallHelper.PATCH_VERSION_FILE)
        self.assertEqual(1, len(patchLines))
        self.assertEqual("1.0.001", patchLines[0])
        
        self.helper.saveNewPrepackVersion("1.0.002")
        patchLines = self._readLines(InstallHelper.PATCH_VERSION_FILE)
        self.assertEqual(2, len(patchLines))
        self.assertEqual("1.0.001", patchLines[0])
        self.assertEqual("1.0.002", patchLines[1])
    
    def testSaveLastHandledBackupVersion(self):
        self.helper.saveLastHandledBackupVersion("2.0@201311111111")
        lines = self._readLines(InstallHelper.LAST_HANDLED_BACKUP_VERSION_FILE)
        self.assertEqual(1, len(lines))
        self.assertEqual("2.0@201311111111", lines[0])
        
        self.helper.saveLastHandledBackupVersion("2.0.001@201311111112")
        lines = self._readLines(InstallHelper.LAST_HANDLED_BACKUP_VERSION_FILE)
        self.assertEqual(2, len(lines))
        self.assertEqual("2.0@201311111111", lines[0])
        self.assertEqual("2.0.001@201311111112", lines[1])
    
    def testGetLastHandledBackupVersion(self):
        self._writeToFile("2.0@201311111111\n2.0.001@201311111112", InstallHelper.LAST_HANDLED_BACKUP_VERSION_FILE)
        self.assertEqual("2.0.001@201311111112", self.helper.getLastHandledBackupVersion())
    
    def testRemoveLastHandledBackupVersion(self):
        self._writeToFile("2.0@201311111111\n2.0.001@201311111112", InstallHelper.LAST_HANDLED_BACKUP_VERSION_FILE)
        self.helper.removeLastHandledBackupVersion()
        lines = self._readLines(InstallHelper.LAST_HANDLED_BACKUP_VERSION_FILE)
        self.assertEqual(1, len(lines))
        self.assertEqual("2.0@201311111111", lines[0])
    
    def testGetCurrentPrepackVersion(self):
        self._writeToFile("3.0", InstallHelper.PREPACK_VERSION_FILE)
        self.assertEqual("3.0", self.helper.getCurrentPrepackVersion())
    
    def testGetPatchesForCurrentPrepackVersion(self):
        self._writeToFile("3.0", InstallHelper.PREPACK_VERSION_FILE)
        self._writeToFile("1.0.001\n3.0.001\n3.0.002", InstallHelper.PATCH_VERSION_FILE)
        patches = self.helper.getPatchesForCurrentPrepackVersion()
        self.assertEqual(2, len(patches))
        self.assertEqual("3.0.001", patches[0])
        self.assertEqual("3.0.002", patches[1])
    
    def testGetVersionMain(self):
        self.assertEqual("2.0", self.helper.getVersionMain("2.0.001"))
        self.assertEqual("2.10", self.helper.getVersionMain("2.10.001.001"))
        self.assertEqual("2.10", self.helper.getVersionMain("2.10.001"))
    
    def testIsVersionUpgradeFormat(self):
        self.assertTrue(self.helper.isVersionUpgradeFormat("1.0-1.1"))
        self.assertFalse(self.helper.isVersionUpgradeFormat("1.0"))
        self.assertFalse(self.helper.isVersionUpgradeFormat("1.0.001"))
    
    def testIsVersionPatchFormat(self):
        self.assertFalse(self.helper.isVersionPatchFormat("1.0-1.1"))
        self.assertFalse(self.helper.isVersionPatchFormat("1.0"))
        self.assertTrue(self.helper.isVersionPatchFormat("1.0.001"))
    
    def testIsCurrentVersionFromUpgrade(self):
        pass
#         self.assertTrue(self.helper.isCurrentVersionFromUpgrade())
    
    def testGetUpgradeFromTo(self):
        self.assertEqual("1.0", self.helper.getUpgradeFromTo("1.0-2.0")[0])
        self.assertEqual("2.0", self.helper.getUpgradeFromTo("1.0-2.0")[1])
    
    def testGetBackupDeployStatusForVersion(self):
        backupVersion = "2.0@201311111111"
        backupDir = self.helper.getBackupDir(backupVersion)
        if not os.path.exists(backupDir):
            os.makedirs(backupDir)
        self._writeToFile("[Backup] [Start]\n[Backup] [Finish]", backupDir + os.sep + InstallHelper.INSTALL_STATUS_FILE)
        self.assertTrue(self.helper.getBackupDeployStatusForVersion(backupVersion)[0])
        self.assertTrue(self.helper.getBackupDeployStatusForVersion(backupVersion)[1])
        self.assertFalse(self.helper.getBackupDeployStatusForVersion(backupVersion)[2])
        self.assertFalse(self.helper.getBackupDeployStatusForVersion(backupVersion)[3])
    
    def testSolveBackupVersion(self):
        self._writeToFile("3.1", InstallHelper.PREPACK_VERSION_FILE)
        self.assertEqual("3.0.001", self.helper.solveBackupVersion("3.0.001"))
        self.assertEqual("3.0-4.0", self.helper.solveBackupVersion("3.0-4.0"))
        self.assertEqual("3.1_rerun", self.helper.solveBackupVersion("3.1"))
    
    def testMoveFolderAsHistory(self):
        backupVersion = "2.0@201311111111"
        backupDir = self.helper.getBackupDir(backupVersion)
        if not os.path.exists(backupDir):
            os.makedirs(backupDir)
        self._writeToFile("[Backup] [Start]\n[Backup] [Finish]", backupDir + os.sep + InstallHelper.INSTALL_STATUS_FILE)
        self.assertTrue(not os.path.exists(InstallHelper.BACKUP_ABANDON_DIR))
        self.helper.moveFolderAsHistory(backupVersion, InstallHelper.BACKUP_ABANDON_DIR)
        self.assertTrue(not os.path.exists(backupDir))
        self.assertTrue(os.path.exists(InstallHelper.BACKUP_ABANDON_DIR))
    
if __name__ == "__main__":
    unittest.main()     