import unittest 
import os
import time
import sys
from distutils.dir_util import remove_tree
from distutils.file_util import copy_file
from InstallHelper import InstallHelper
from install import Install
from backup import Backup
from InstallComponentCollector import InstallComponentCollector
from PrepackInstalledDecider import PrepackInstalledDecider
import random
from UnittestBase import UnittestBase

class InstallTestBase(UnittestBase):
    sequence = 0
    inputsBySequence = []
    TEST_TEMP_DIR = "/tmp/test_temp"
    
    def setUp(self):
        UnittestBase.setUp(self)
        if not os.path.exists(self.TEST_TEMP_DIR):
            os.makedirs(self.TEST_TEMP_DIR)
            os.makedirs(self.TEST_TEMP_DIR + os.sep + "log")
        cwd = os.path.dirname(os.path.realpath(__file__))
        if not os.path.exists(cwd + '/rpms'):
            os.mkdir(cwd + '/rpms')
        if not os.path.exists(cwd + '/rpms/prepack-license-manager-4.0.000.001.jar'):
            jar = '/var/lib/jenkins/.m2/repository/com/ericsson/cms/prepack-license-manager/4.0.000.001/prepack-license-manager-4.0.000.001.jar'
            distfile = cwd + '/rpms'
            copy_file(jar, distfile)
        self._resetPrepackDir(self.TEST_TEMP_DIR)
        self._resetVariables()
        self._initComponentSelection()
    
    def tearDown(self):
        if os.path.exists(self.TEST_TEMP_DIR):
            remove_tree(self.TEST_TEMP_DIR)
        unittest.TestCase.tearDown(self)
    
    def _constructSysInfoFile(self, version):
        configObject = {
            "name":"CMS AIO PrePack",
            "version":version,
            "cluster_service_nodes": [
                {"ip":"127.0.0.1", "user":"cms", "password":"cmspwd", "rootPasswd":"rootpwd"}
            ]
        }
        sysInfoPath = self.TEST_TEMP_DIR + os.sep + "sysinfo.json"
        InstallHelper.writeJsonToFile(configObject, sysInfoPath)
        return sysInfoPath
    
    def _defineUpgradePaths(self, installer, paths = []):
        installer.config["upgradePath"] = paths
    
    def _initUpgradeComponents(self, version, availableUpgrades):
        availableUpgradeConfigPath = self.TEST_TEMP_DIR + os.sep + version + os.sep + "components" + os.sep + "upgrade-components.json"
        InstallHelper.writeJsonToFile(availableUpgrades, availableUpgradeConfigPath)
        Install.COMPONENTS_PATH_PREFIX = self.TEST_TEMP_DIR
    
    def _initComponentSelection(self):
        selections = {
            "selectionItems": [
                "DOWNSTREAMS",
                "DEVICES"
            ],
            
            "DEVICES":{
                "displayName":"Devices",
                "selections" : [
                    "QC",
                    "TRANSCODE",
                    "ENCRYPTION",
                    "COMBINER"
                ]
            },
        
            "QC":{
                "displayName":"QC",
                "selections":[
                    "BATON",
                    "CERIFY",
                    "VERIFIER"
                ]
            },
            
            "TRANSCODE":{
                "displayName":"Transcoder",
                "selections":[
                    "ENVIVIO",
                    "ELEMENTAL",
                    "TITAN",
                    "RHOZET",
                    "DIGITALRAPID"
                ]
            },
            
            "ENCRYPTION":{
                "displayName":"Encryptor",
                "selections":[
                    "VERIMATRIX"
                ]
            },
            
            "COMBINER":{
                "displayName":"Combiner"
            },
            
            "BATON":{
                "displayName":"Baton"
            },
            
            "CERIFY":{
                "displayName":"Cerify"
            },
            
            "VERIFIER":{
                "displayName":"Verifier"
            },
        
            "ENVIVIO":{
                "displayName":"Envivio"
            },
            
            "ELEMENTAL":{
                "displayName":"Elemental"
            },
            
            "TITAN":{
                "displayName":"Titan"
            },
            
            "RHOZET":{
                "displayName":"Rhozet"
            },
            
            "DIGITALRAPID":{
                "displayName":"DigitalRapid"
            },
            
            "VERIMATRIX":{
                "displayName":"Verimatrix"
            },
            
            "DOWNSTREAMS":{
                "displayName":"Downstreams",
                "selections" : [
                    "AMS",
                    "MEDIAROOM",
                    "OTT",
                    "MSMW_MDMS",
                    "MSMW_OS",
                    "XPMP",
                    "FABRIX"
                ]
            },
            
            "AMS":{
                "displayName":"AMS",
                "unAvailableComponents":[
                    "DEVICES/COMBINER"
                ],
            },
            
            "MEDIAROOM":{
                "displayName":"Mediaroom"
            },
            
            "OTT":{
                "displayName":"OTT"
            },
            
            "MSMW_MDMS":{
                "displayName":"Msmw with MDMS"
            },
            
            "MSMW_OS":{
                "displayName":"Msmw with Openstream"
            },
            
            "XPMP":{
                "displayName":"XPMP",
                "availableComponents":[
                    "DEVICES/QC/VERIFIER",
                    "DEVICES/COMBINER",
                    "DEVICES/QC/BATON",
                ],
                "autoSelectComponents":[
                    "DEVICES/QC/VERIFIER",
                    "DEVICES/COMBINER"
                ],
                "singleSelection": True
            },
            
            "FABRIX":{
                "displayName":"FABRIX"
            },
                      
            "BACK":{
                "displayName":"Back To Last"
            }
        }
        selectionFilePath = self.TEST_TEMP_DIR + os.sep + "component-selections.json"
        InstallHelper.writeJsonToFile(selections, selectionFilePath)
        InstallComponentCollector.COMPONENT_SELECTION_CONFIG = selectionFilePath
    
    def _createInstaller(self, mode, version, backupStart = True, backupFinish = True, deployStart = True, deployFinish = True):
        sysInfoFile = self._constructSysInfoFile(version)
        installer = InstallMock(sysInfoFile, mode, backupStart, backupFinish, deployStart, deployFinish)
        self._mockPrepackDecider(installer)
        self._mockRawInputForInstaller(installer)
        return installer
    
    def _mockPrepackDecider(self, installer):
        mockDecider = PrepackInstalledDeciderMock(False, "")
        installer.helper.decider = mockDecider
        try:
            installer.installComponentCollector.decider = mockDecider
        except:
            pass
        
    def _mockRawInputForInstaller(self, installer):
        self._mockRawInputForHelper(installer)
        self._mockRawInputForHelper(installer.installComponentCollector)
    
    def _mockRawInputForHelper(self, object):
        object.helper.getRawInput = rawInputMock
        
    def _resetInputs(self,inputs):
        InstallTestBase.sequence = 0
        InstallTestBase.inputsBySequence = inputs

    def _doInstall(self, installer, preConfiguredFile=None, outputToFile = False):
        logFile = None
        logFileName = ""
        if outputToFile:
            logFileName = self._getInstallLogFileName()
            logFile = InstallOut(logFileName, "a")
        installer.install(preConfiguredFile)
        self.assertEqual(InstallTestBase.sequence, len(InstallTestBase.inputsBySequence),"Expected user input not match the really input.")
        if logFile:
            logFile._restore_standard_output_()
        return logFileName
    
    def _readLines(self, filePath):
        if os.path.exists(filePath):
            alllines = open(filePath, 'r').readlines()
            for i in range(0, len(alllines)):
                alllines[i] = alllines[i].replace('\n', '')
            return alllines
        return None
    
    def _writeToFile(self, text, filePath):
        d = os.path.dirname(filePath)
        if not os.path.isdir(d):
            os.makedirs(d)
        with open(filePath, "w") as f:
            f.write(text)
    
    def _resetPrepackDir(self, directory):
        InstallHelper.PREPACK_DIR = directory
    
    def _resetVariables(self):
        InstallHelper.BACKUP_DIR = InstallHelper.PREPACK_DIR + os.sep + "backup-restore"
        InstallHelper.BACKUP_ABANDON_DIR = InstallHelper.BACKUP_DIR + os.sep + "abandoned"
        InstallHelper.ROLLBACK_DIR = InstallHelper.BACKUP_DIR + os.sep + "rollback"
        
        InstallHelper.PREPACK_VERSION_FILE = InstallHelper.PREPACK_DIR + os.sep + "prepackVersion.dat"
        InstallHelper.PATCH_VERSION_FILE = InstallHelper.PREPACK_DIR + os.sep + "patchVersion.dat"
        InstallHelper.BACKUP_VERSION_FILE = InstallHelper.BACKUP_DIR + os.sep + "backupVersion.dat"
        InstallHelper.LAST_HANDLED_BACKUP_VERSION_FILE = InstallHelper.BACKUP_DIR + os.sep + "lastHandledBackupVersion.dat"
        InstallHelper.INSTALL_ACTION_LOG = InstallHelper.PREPACK_DIR + os.sep + "installAction.log"
        
        InstallHelper.SELECTED_COMPONENTS_TEMP = InstallHelper.PREPACK_DIR + os.sep + "selected-component-temp.json"
        InstallHelper.INSTALLED_COMPONENTS = InstallHelper.PREPACK_DIR + os.sep + "installedcomponent.json"
        InstallHelper.INSTALLATION_TEMP = InstallHelper.PREPACK_DIR + os.sep + "installation_temp"
        
    def _getInstallLogFileName(self):
        logFilePath = self.TEST_TEMP_DIR + os.sep + "log" + os.sep + "install.log." + time.strftime('%Y%m%d%H%M%S') + str(random.getrandbits(30))
        InstallHelper.mkdirForFile(logFilePath)
        return logFilePath
    
    def _verifyLogFromLogFile(self, logFile, logs):
        foundRecords = []
        if os.path.exists(logFile):
            allRecords = open(logFile, 'r').readlines()
            for log in logs:
                found = False
                for record in allRecords:
                    if log == record.strip('\n\t '):
                        found = True
                self.assertTrue(found, "Log is not found: " + log)
        return foundRecords
    
    def _getBackupFolders(self):
        backupDirs = []
        if os.path.exists(InstallHelper.BACKUP_DIR):
            for dirName in os.listdir(os.path.abspath(InstallHelper.BACKUP_DIR)):
                index = dirName.find("@")
                if index > -1:
                    backupDirs.append(dirName[0:index])
        return backupDirs
        
    def _getBackupFolderForVersion(self, version):
        return [backupDir for backupDir in self._getBackupFolders() if backupDir == version]
    
    def _getBackupAbandonFolders(self):
        backupDirs = []
        if os.path.exists(InstallHelper.BACKUP_ABANDON_DIR):
            for dirName in os.listdir(os.path.abspath(InstallHelper.BACKUP_ABANDON_DIR)):
                index = dirName.find("@")
                if index > -1:
                    backupDirs.append(dirName[0:index])
        return backupDirs
        
    def _getBackupAbandonFolderForVersion(self, version):
        return [backupDir for backupDir in self._getBackupAbandonFolders() if backupDir == version]
    
    def _getRollbackFolders(self):
        backupDirs = []
        if os.path.exists(InstallHelper.ROLLBACK_DIR):
            for dirName in os.listdir(os.path.abspath(InstallHelper.ROLLBACK_DIR)):
                index = dirName.find("@")
                if index > -1:
                    backupDirs.append(dirName[0:index])
        return backupDirs
        
    def _getRollbackFolderForVersion(self, version):
        return [backupDir for backupDir in self._getRollbackFolders() if backupDir == version]
    
class InstallOut(object):
    def __init__(self, name, mode, withTimeStamp=False):
        if withTimeStamp:
            name = name + "." + time.strftime('%Y%m%d%H%M%S')
        self.file = name
        self.stdout = sys.stdout
        sys.stdout = self
    
    def _restore_standard_output_(self):
        sys.stdout = self.stdout
    
    def write(self, data):
        with open(self.file, "a") as f:
            f.write(data)
        self.stdout.write(data)
    
def rawInputMock(prompt):
    print prompt
    returnInput = InstallTestBase.inputsBySequence[InstallTestBase.sequence]
    InstallTestBase.sequence = InstallTestBase.sequence + 1
    return returnInput

class InstallMock(Install):
    def __init__(self, sysInfoFile, mode, backupStart = True, backupFinish = True, deployStart = True, deployFinish = True):
        Install.__init__(self, sysInfoFile, mode)
        self.backupStart = backupStart
        self.backupFinish = backupFinish
        self.deployStart = deployStart
        self.deployFinish = deployFinish
    
    def rollbackFirst(self):
        self.helper.rollBackToLastVersion(self.helper.getLatestBackupVersion())
        return True
    
    def initPolicy(self):
        pass
        
    def backup(self, inputfileOrObject):
        if self.backupStart:
            return Install.backup(self, inputfileOrObject)
        else:
            return False
    
    def doBackup(self, inputfileOrObject):
        if self.backupFinish:
            bak = Backup(inputfileOrObject)
            bak.copyInfoToRollBackJson()
            bak.backupPrepackVersionInfo()
            bak.generateRollBackJson()
            return True
        else:
            return False
    
    def deploy(self, inputfileOrObject):
        if self.deployStart:
            return Install.deploy(self, inputfileOrObject)
        else:
            return False
        
    def doDeploy(self, inputfileOrObject):
        if self.deployFinish:
            return True
        else:
            return False

class PrepackInstalledDeciderMock(PrepackInstalledDecider):
    
    def __init__(self, prepackInstalled, installedPrepackName):
        self.prepackInstalled = prepackInstalled
        self.installedPrepackName = installedPrepackName
        PrepackInstalledDecider.__init__(self)
     
    def isAnyPrepackInstalled(self):
        return self.prepackInstalled
    
    def getInstalledPrepackName(self):
        return self.installedPrepackName