#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Jerish Lin
# Created: Oct 17, 2013
# Description:
# Generic PrePackage Backup Script
#

from BaseClass import BaseClass
from Common import Common
from FileAssembler import ContentClassHandler
from dbutils import Dao, DateTimeEncoder
from distutils.dir_util import copy_tree
from distutils.file_util import copy_file
import DistTemplateConfiguration
import FTPConfiguration
import Partners
import ReportCategoryConfiguration
import ReportConfiguration
import WatchFolderConfiguration
import distutils
import shutil
import fnmatch
import json
import os
import tarfile
import time
from CMSVersionDeploymentPolicy import CMSVersionDeploymentPolicy
from Alerts import AlertPolicyForCMS4X, AlertPolicyForCMS5X
from SSHCommander import SSHCommander
from EnhancementRuleset import EnhancementRuleset

class Backup(Common):
    
    def __init__(self, inputfileOrObject):
        Common.__init__(self)
        self.backupItems = {}
        if type(str()) == type(inputfileOrObject):
            self.loadJsonConfig(inputfileOrObject, True)
        else:
            self.config = inputfileOrObject
        self.initParameters()
        self.__conn = CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().getDatabaseConnection()
        self.__cursor = self.__conn.cursor()

    def backupItemsGeneric(self, itemType):
        jsonKey = self.getItemJsonKey(itemType)
        configItems = self.getConfig(jsonKey)
        if not configItems:
            return    
        
        backupItems = []        
        handleItemObj = self.getHandleItemObj(itemType)
        keyArray = self.getItemKeyArray(itemType)
        uniqueKeyArray = self.getItemUniqueKeyArray(itemType)
        backupDict = {}
        
        childItemKeys = self.findItemKeysByParent(itemType)
        for configItem in configItems:
            backupDict = self.generateBackUpItem(keyArray, handleItemObj.backup(self.getUniqueKeyTuple(configItem,uniqueKeyArray)+(self.getIsDelete(configItem),)))
            if childItemKeys:
                for childItemKey in childItemKeys:
                    childItemKeyType, childItemKeyValues = childItemKey
                    childJsonKey = childItemKeyValues["jsonKey"]
                    childKeyArray = childItemKeyValues["keys"]
                    childUniqueKeyArray = childItemKeyValues["uniqueKeys"]
                    childHandleItemObj = self.getHandleItemObj(childItemKeyType)
                    
                    if(self.getIsDelete(configItem)):
                        backupDict[childJsonKey] = childHandleItemObj.backUpAll(self.getUniqueKeyTuple(configItem,uniqueKeyArray))
                    else:
                        childBackupList = []
                        if configItem.has_key(childJsonKey) and configItem[childJsonKey]:
                            for childConfigItem in configItem[childJsonKey]:
                                childBackupList.append(self.generateBackUpItem(childKeyArray,childHandleItemObj.backup(self.getUniqueKeyTuple(childConfigItem,childUniqueKeyArray)+self.getUniqueKeyTuple(configItem,uniqueKeyArray)+(self.getIsDelete(childConfigItem),))))
                            backupDict[childJsonKey] = childBackupList
            backupItems.append(backupDict)
        self.backupItems[jsonKey] = backupItems
                    
    def getUniqueKeyTuple(self, configItem, uniqueKeys):
        tempArray = []
        for uniqueKey in uniqueKeys:
            tempArray.append(configItem[uniqueKey])
        return tuple(tempArray)
    
    def mergeTuple(self, *tuples):
        mergedList = []        
        for t in tuples:
            mergedList = mergedList + list(t)
        return tuple(mergedList)

    def backupPartners(self, partners_):
        if partners_ is None or not len(partners_):
            return
 
        keyArray = self.getItemKeyArray("PARTNER")
        partners = Partners.Partners()
        items = []
        for partner in partners_:
            for provider in partner["PROVIDERID"]:
                partnerItem = self.generateBackUpItem(keyArray, partners.backup((provider,)+(self.getIsDelete(partner),)))
                if type(partnerItem["PROVIDERID"]) is not list:
                    partnerItem["PROVIDERID"] = [partnerItem["PROVIDERID"]]
                items.append(partnerItem)
                 
        self.backupItems["partners"] = items

    def generateBackUpItem(self, keyArray, item):
        backupItemObj = {}
        isDelete, valueArray = item
        for i in range(0, len(valueArray)):
            if valueArray[i] is None:
                backupItemObj[keyArray[i]] = ""
            else:
                backupItemObj[keyArray[i]] = valueArray[i]
        if(isDelete):
            backupItemObj["IS_DELETE"] = "true"
        return backupItemObj
    
    def backupContentClasses(self, username, password):
        print 'backup content classes'
        xml_path = self.getBackupDir() + os.sep + 'current_conent_class.xml'
        ContentClassHandler.exportById(xml_path, 1, username, password)
        self.backupItems["contentClassBackupPaths"] = [xml_path]
    
    def backupRuleSets(self):
        print 'backup rules'
        outputPath = self.getBackupDir() + os.sep + "Exported_Full_RuleSet.xml"
        self.exportRules('-101', outputPath, self.config["cms"]["user"], self.config["cms"]["password"])
        
        self.updateProfileName(outputPath)
        
        self.backupItems["ruleSetsBackupPath"] = outputPath
        ruleSetStatus = Dao.export(self.__cursor, "select RULESETID,RULENAME,ENABLED from TRE_RULESET")
        self.backupItems["ruleSetStatus"] = ruleSetStatus
        
    def updateProfileName(self, rulesetPath):
        """
        Description: Update the ME profile id with the actual profile name 
        """
        enhancement = EnhancementRuleset()
        enhancement.replaceWithProfileName(rulesetPath)
            
    def backupReports(self, categoryList):
        cc = ReportCategoryConfiguration.ReportCategoryConfiguration()
        self.backupFileOrDir(CMSVersionDeploymentPolicy.getReportDeploymentPath())
        
        """"cms version < 3.1"""
        if categoryList:
            keyArray = ['NAME', 'DESC', 'PATH']
            catalogListValue = cc.backup(categoryList)
            catalogList = []
            
            for catalog in catalogListValue:
                catalogList.append(self.generateBackUpItem(keyArray, catalog))
            self.backupItems["jreports"] = catalogList
    
    def backupDirectoriesToCreate(self, dirs):
        if dirs is None or not len(dirs):
            return
        for directory in dirs:
            self.getNotExistDirectories(directory, self.getBackupItems("directoriesToRemove"))
                
    def getNotExistDirectories(self, dirPath, paths):
        if not os.path.isdir(dirPath):
            paths.append(dirPath)
            parent = os.path.abspath(os.path.join(dirPath, os.pardir))
            self.getNotExistDirectories(parent, paths)
    
    def backupAllRemoteFiles(self):
        pass
    
    def backupAllConfigurationFiles(self):
        print 'backup All Configuration Files'
        self.backupFileOrDir("/etc/hosts")
        self.backupFileOrDir(CMSVersionDeploymentPolicy.getReportConfigurationFilePath())
        self.backupFileOrDir(CMSVersionDeploymentPolicy.getJreportCatalogDataSourceFilePath())
        self.backupFileOrDir(WatchFolderConfiguration.WatchFolderConfiguration().filePath)
        self.backupFileOrDir(DistTemplateConfiguration.DistTemplateConfiguration().filePath)
        self.backupFileOrDir(FTPConfiguration.FTPConfiguration().filePath)
        if self.getConfig("configFileDirs"):
            for configDir in self.getConfig("configFileDirs"):
                self.backupViaDirectory(configDir, "/", [], [])
        if self.getConfig("configFileToDeleteDirs"):
            for configDir in self.getConfig("configFileToDeleteDirs"):
                self.backupViaDirectory(configDir, "/", [], [])
                
    def backupViaDirectoryWithDefaultDirExcluded(self, directory, targetDir, excludefilters, copyFolderFilters):
        defaultExcludeFilters = self.BACKUP_FULL_DIRS + [fil + "/*" for fil in self.BACKUP_FULL_DIRS]
        self.backupViaDirectory(directory, targetDir, defaultExcludeFilters + excludefilters, copyFolderFilters)
                        
    def backupViaDirectory(self, parentDirectory, targetDir, excludefilters, copyFolderFilters):
        pathFilters = []
        for path, dirNames, fileNames in os.walk(os.path.abspath(parentDirectory)):
            if self.matchFile(self.getPathUnderSystem(parentDirectory, targetDir, path), pathFilters + excludefilters):
                continue
            for dirName in dirNames:
                actualSrcDir = self.getPath(parentDirectory, targetDir, path, dirName)
                if self.matchFile(actualSrcDir, copyFolderFilters):
                    self.backupFileOrDir(actualSrcDir)
                    pathFilters.append(actualSrcDir)
                    pathFilters.append(actualSrcDir + "/*")
            for fileName in fileNames:
                actualSrcFile = self.getPath(parentDirectory, targetDir, path, fileName)
                if self.matchFile(actualSrcFile, excludefilters):
                    continue
                self.backupFileOrDir(actualSrcFile)
    
    def _backupFileOrDir(self, path, fileOrDirToDeleteConfigName, targetBackupDir):
        if os.path.exists(path):
            if os.path.isdir(path):
                try:
                    copy_tree(path, targetBackupDir + path, preserve_symlinks=1)
                except Exception, e:
                    #Ignore the error, gennerally, it's due to file already exist
                    pass
                self.getBackupItems(fileOrDirToDeleteConfigName).append(path)
            else:
                f = targetBackupDir + os.sep + path
                d = os.path.dirname(f)
                if not os.path.isdir(d):
                    os.makedirs(d)
                try:
                    if os.path.islink(path):
                        linkto = os.readlink(path)
                        os.symlink(linkto, f)
                    else:
                        copy_file(path, f)
                except Exception, e:
                    #Ignore the error, gennerally, it's due to file already exist
                    pass
        else:
            self.getBackupItems(fileOrDirToDeleteConfigName).append(path)
    
    def backupFileOrDir(self, path):
        self._backupFileOrDir(path, "fileOrDirToDelete", self.getFileBackupDir())
    
    def backupRemoteFileOrDir(self, rmtPath, rmtNodeConfigName):
        '''Backup a remote file or folder to local. Add an entry into rollback.json.'''
        nodes = self.getConfig(rmtNodeConfigName)
        if not nodes:
            return
        print('backup remote file [%s] of [%s]' % (rmtPath, nodes[0]['ip']))
        
        # create necessary folders
        bak_path = self.getFileBackupDir() + rmtPath
        d = os.path.dirname(bak_path)
        if not os.path.exists(d):
            os.makedirs(d)
        
        shared_file = Common.COMMON_SHARED_FOLDER + rmtPath
        d = os.path.dirname(shared_file)
        if not os.path.exists(d):
            os.makedirs(d)
        
        # only backup one of the remote nodes for now.
        client = None
        try:
            client = SSHCommander(nodes[0]['ip'], nodes[0]['user'], nodes[0]['password'])
            cmdstr = 'su root -c "\cp %s %s"' % (rmtPath, shared_file)
            status, output = client.execute_interactive(cmdstr, nodes[0]['rootPasswd'])
            print(output)
            if status != 0:
                print "failed to backup remote file on " + nodes[0]['ip']
                print "Alert configuration on CS node may do not exist"
            
            # use 'move' to reduce waste temporary files
            else:
                shutil.move(shared_file, bak_path)
                self.getBackupItems("remoteFilesToRestore").append({"remotePath": rmtPath, 'remoteNodeConfigName': rmtNodeConfigName})
        except Exception:
            raise   # NOP
        finally:
            if client:
                client.logout()

    def backupPrepackVersionInfo(self):
        for prepackVersionInfoFile in self.helper.getPrepackVersionInfoFiles():
            self._backupFileOrDir(prepackVersionInfoFile, "prepackVersionInfoFileToDelete", self.getPrepackInfoBackupDir())
                
    def getBackupItems(self, itemName):
        if not self.backupItems.has_key(itemName):
            self.backupItems[itemName] = []
        return self.backupItems[itemName]
    
    def createDirectory(self, dir):
        if not os.path.exists(dir):
            os.makedirs(dir)
    
    def matchFile(self, file, filters):
        for pattern in filters:
            if fnmatch.fnmatch(file, pattern):
                return True
        return False
    
    def getPath(self, directory, targetDir, path, file):
        fullpath = os.path.join(path, file)
        return self.getPathUnderSystem(directory, targetDir, fullpath) 
    
    def getPathUnderSystem(self, directory, targetDir, fullpath):
        # Handle the origin path
        if((fullpath + "/").index(directory) + len(directory) == len(fullpath) + 1):
            return targetDir
        return targetDir + fullpath[fullpath.index(directory) + len(directory):]
    
    def putToRollBackJson(self, configName):
        self.backupItems[configName] = self.getConfig(configName)
        
    def generateRollBackJson(self):
        print 'current backup dir: ' + self.getBackupDir()
        file = self.getBackupDir() + os.sep + self.ROLLBACK_JSON
        with open(file, 'w') as f:
            json.dump(self.backupItems, f, cls=DateTimeEncoder, ensure_ascii=False, indent=4)
        self.replaceStringInFile(file, "null", "\"\"")
            
    def replaceStringInFile(self, filePath, target, src):
        """
        Input:
            filePath: the file to be modified
            target: target string in the file will be replaced
            src:  source string to replace target string
        """
        tmpFile = filePath + ".bkp"
        with open(tmpFile, 'w') as fileOut:
            with open(filePath) as fileIn:
                for line in fileIn:
                    fileOut.write(line.replace(target, src))
        os.remove(filePath)
        os.rename(tmpFile, filePath)                
        
            
    def backupAllCurrentTemplates(self, templates):
        print "backup all templates"
        if templates:
            extractDir = time.strftime('%Y%m%d%H%M%S') + "_template_files"
            self.__extractRpms(self.getConfig("templateRpms"), extractDir)
            self.backupViaDirectoryWithDefaultDirExcluded(os.getcwd() + os.sep + extractDir + '/', '/', ["/opt/tandbergtv/cms/workflow/templates/*.par"], ["/opt/tandbergtv/cms/workflow/plugins/subsystems/*", "/opt/tandbergtv/cms/workflow/plugins/groups/*"])
            distutils.dir_util.remove_tree(os.getcwd() + os.sep + extractDir)
        templateVersions = Dao.export(self.__cursor, "select ID_,NAME_,PROCESSDEFINITIONTYPEID from JBPM_PROCESSDEFINITION")
        self.backupItems["templateVersions"] = templateVersions
        
        
    def backupResourceTypes(self, resourceTypes):
        print "backup resource types"
        if resourceTypes:
            extractDir = time.strftime('%Y%m%d%H%M%S') + "_resources_type_files"
            self.__extractRpms(self.getConfig("resourceTypeRpms"), extractDir)
            self.backupViaDirectoryWithDefaultDirExcluded(os.getcwd() + os.sep + extractDir + '/', '/', ["/opt/tandbergtv/cms/workflow/templates/*.par"], ["/opt/tandbergtv/cms/workflow/plugins/subsystems/*", "/opt/tandbergtv/cms/workflow/plugins/groups/*"])
            distutils.dir_util.remove_tree(os.getcwd() + os.sep + extractDir)
        resourceTypes = Dao.export(self.__cursor, "select RESOURCETYPEID,NAME,RESOURCECONNECTIONTYPEID from TTV_RESOURCETYPE")
        self.backupItems["resourceTypes"] = resourceTypes
    
    def backupResourceGroups(self):
        print "backup resource groups"
        resourceGroups = Dao.export(self.__cursor, "SELECT RESOURCEGROUPID,NAME FROM TTV_RESOURCEGROUP")
        self.backupItems["resourceGroups"] = resourceGroups
    
    def backupAlertsConfigFile(self):
        print "backup alerts"
        cmsVersion = CMSVersionDeploymentPolicy.getCMSVersion()
        if cmsVersion >=4.0 and cmsVersion < 5.0:
            self.backupRemoteFileOrDir(AlertPolicyForCMS4X.ALERT_CONF_FILE, 'cluster_service_nodes')
        else:
            self.backupFileOrDir(AlertPolicyForCMS5X.ALERT_CONF_FILE)
    
    def backupForRpms(self, rpms):
        print "backup for rpms"
        if rpms:
            extractDir = time.strftime('%Y%m%d%H%M%S') + "_rmps_files"
            self.__extractRpms(rpms, extractDir)
            self.backupViaDirectoryWithDefaultDirExcluded(os.getcwd() + os.sep + extractDir + '/', '/', [], [])
            distutils.dir_util.remove_tree(os.getcwd() + os.sep + extractDir)
        print "backup for rpms done"
        
    def backupFullDirs(self):
        print 'backup full directories'
        for path in Common.BACKUP_FULL_DIRS:
            
            #ignore none exist dir
            try:
                copy_tree(path, self.getFileBackupDir() + path, preserve_symlinks=1)
                self.getBackupItems("fileOrDirToDelete").append(path)
            except:
                pass
    
    def __extractRpms(self, rpms, targetDir):
        os.mkdir(targetDir)
        os.chdir(targetDir)
        for rpm in rpms:
            extractRPM = 'rpm2cpio ../rpms/%s' % rpm + ' | cpio -idm --quiet'    
            os.system(extractRPM)
        os.chdir("..")
        
    def backupQueriesData(self):
        queriesData = self.getConfig("queriesData")
        if queriesData:
            for r in queriesData:
                r['data'] = Dao.export(self.__cursor, r['BACKUP_SQL'])
            self.backupItems["queriesData"] = queriesData

    def copyInfoToRollBackJson(self):
        self.putToRollBackJson("name")
        self.putToRollBackJson("version")
        self.putToRollBackJson("backupVersion")
        self.putToRollBackJson("cms")
        self.putToRollBackJson("oracle")
        self.putToRollBackJson("postgresql")
        self.putToRollBackJson("ignoreFileSycn")
        self.putToRollBackJson("rollBackQueries")
        self.putToRollBackJson("rollbackPostScripts")
        if CMSVersionDeploymentPolicy.getCMSVersion() >= 4.0:
            self.putToRollBackJson("cluster_service_nodes")
            self.putToRollBackJson("cluster_app_nodes")

    def backup(self):
        self.output("Backing up for " + self.name)
        self.output(self.backupVersion)
        self.output('\n');
        
        self.copyInfoToRollBackJson()
        self.backupQueriesData()
        self.backupFullDirs()
         
        self.backupAllCurrentTemplates(self.getConfig("templateRpms"))
        self.backupResourceTypes(self.getConfig("resourceTypeRpms"))
        self.backupForRpms(self.getConfig("otherRpms"))
        self.backupForRpms(self.getConfig("commonRpms"))
        self.backupForRpms(self.getConfig("rpmsToDelete"))
        self.backupResourceGroups()
    
        #Backup partner is special, so not apply the generic method.
        self.backupPartners(self.getConfig("partners"))

        if CMSVersionDeploymentPolicy.getCMSVersion() < 4.0:
            ITEMS_BACK_UP = ["SITE","SELECOTRKEY","RESOURCE","CUSTOMFIELDGROUP","MESUBSPROFILE","ALERTNAME"]
        else:
            ITEMS_BACK_UP = ["SITE","SELECOTRKEY","RESOURCE","CUSTOMFIELDGROUP","MESUBSPROFILE"]
            self.backupAlertsConfigFile()
        for eachItemType in ITEMS_BACK_UP:
            self.backupItemsGeneric(eachItemType)

        if CMSVersionDeploymentPolicy.getCMSVersion() < 3.1:
            self.backupReports(self.getConfig("jreports"))
        else:
            self.backupReports(None)
        
        # backup directoriesToCreate, if the directory no exists, rolling back should remove the directory.
        self.backupDirectoriesToCreate(self.getConfig("directoriesToCreate"))
        self.backupPrepackVersionInfo()
        self.backupAllConfigurationFiles()
        
        if not ContentClassHandler.isServiceUp():
            self.output('Required services have not started yet, exit backup.')
            return False
        self.backupContentClasses(self.config["cms"]["user"], self.config["cms"]["password"])
        self.backupRuleSets()
        
        self.generateRollBackJson()
        #self.tarBackupDir()
        
        # Finally, let the user know they're done
        self.output('Backup complete.')
        return True
        
    def tarBackupDir(self):
        """tar and compress a single directory"""
        tarf = tarfile.open(self.getBackupDir() + '.tar.gz', 'w:gz')
        tarf.add(self.getBackupDir(), arcname=self.backupVersion)
        tarf.close()   
         
if __name__ == "__main__":
    backup = Backup("/opt/tandbergtv/cms/prepack/installation_temp/combined-config.json_2015-05-06-01-49-18")
    backup.backup()        
