#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Bill Hood
# Author: Jerish Lin
# Created: Dec 11, 2012
# Description:
# Generic PrePackage Deployment Script
#

from FileAssembler import ContentClassHandler
from xml.etree.ElementTree import ElementTree
from BaseClass import BaseClass
from Common import Common
import DistTemplateConfiguration
import FTPConfiguration
import WatchFolderConfiguration
import DatabaseSchemaExecutor

import fnmatch
import hashlib
import os
import shutil
import socket
import sys
import time
from filesync import FileSync
from CMSVersionDeploymentPolicy import CMSVersionDeploymentPolicy

class Deploy(Common):
    def __init__(self, inputfileOrObject):
        Common.__init__(self)
        if type(str()) == type(inputfileOrObject):
            self.loadJsonConfig(inputfileOrObject, True)
        else:
            self.config = inputfileOrObject
        self.initParameters()
       
    def checkTargetTitle(self):
        workflow = "/opt/tandbergtv/cms/workflow"
        rejected = workflow + "/rejected"
        templates = workflow + "/templates"
        dirlist = os.listdir(rejected)
        
        tt = filter(lambda t : "Target Title" in t, dirlist)
        if not len(tt):
            return
        else:
            os.rename(rejected + "/" + tt[0], templates + "/" + tt[0])

    def deleteConfigFiles(self, configFileToRemoveDirectories):
        if configFileToRemoveDirectories:
            for configFileDir in configFileToRemoveDirectories:
                self.deleteConfigFilesViaDirectory(configFileDir, "/")
    
    def deleteConfigFilesViaDirectory(self, directory, targetDir):
        for path, dirs, files in os.walk(os.path.abspath(directory)):
            for filename in files:
                targetFilePath = self.getTargetFilePath(directory, targetDir, os.path.join(path, filename))
                if os.path.exists(targetFilePath):
                    os.remove(targetFilePath)

    def copyAllResouceFiles(self, configFileDirectories, params):
        if configFileDirectories:
            for configFileDir in configFileDirectories:
                self.copyResourceFilesAndReplaceParams(configFileDir, "/", params)

    def copyResourceFilesAndReplaceParams(self, directory, targetDir, params):
        self.output("Copying all file from resource directory and replace with parameters.")
        for path, dirs, files in os.walk(os.path.abspath(directory)):
            for dir in dirs:
                dirpath = os.path.join(path, dir)
                saveDirpath = self.getTargetFilePath(directory, targetDir,dirpath)
                if not os.path.exists(saveDirpath):
                    os.mkdir(saveDirpath)
                    if saveDirpath.find("/opt/") == 0:
                        self.chown('nobody', 'nobody', saveDirpath, True)
            for filename in fnmatch.filter(files, "*"):
                if not filename.startswith('.svn'):
                    filepath = os.path.join(path, filename)
                    with open(filepath) as f:
                        s = f.read()
                    if not '\0' in s:
                        if params is not None and len(params):
                            for param in params:
                                s = s.replace("${"+param["NAME"]+"}",param["VALUE"])
                    finalFilepath = self.getTargetFilePath(directory, targetDir, filepath)
                    tempSaveFilepth = finalFilepath + ".tmp"
                    with open(tempSaveFilepth, "w") as f:
                        f.write(s)
                        #Only change the owner for those files under /opt
                        if tempSaveFilepth.find("/opt/") == 0:
                            self.chown('nobody', 'nobody', tempSaveFilepth, True)
                    if not os.path.exists(finalFilepath):
                        shutil.move(tempSaveFilepth, finalFilepath);
                        self.output("File not exists, create file: " + finalFilepath)
                    else:
                        oldFilemd5 = hashlib.md5(open(finalFilepath).read()).hexdigest()
                        newFilemd5 = hashlib.md5(open(tempSaveFilepth).read()).hexdigest()
                        if oldFilemd5 != newFilemd5:
                            self.backupFile(finalFilepath)
                            shutil.move(tempSaveFilepth, finalFilepath);
                            self.output("File changed, backup and over write file: " + finalFilepath)
                        else:
                            os.remove(tempSaveFilepth)
    
    def getTargetFilePath(self, directory, targetDir, filepath): 
        return targetDir + filepath[filepath.index(directory)+len(directory):]                   

    def configureResourceHosts(self, hosts):
        if hosts is None or not len(hosts):
                return

        toBeDeletedHosts = []
        toBeAddedHosts = []
        for host in hosts:
            if self.getIsDelete(host):
                toBeDeletedHosts.append(host)
            else:
                toBeAddedHosts.append(host)

        print "The following hosts will be deleted: "
        print toBeDeletedHosts

        print "The following hosts will be added or updated: "
        print toBeAddedHosts

        fn = "/etc/hosts"
        existingHosts = file(fn, "r").readlines()
        newHostFile = file(fn, "w")

        # add existing
        for host in existingHosts:
                # empty line
                if not host.strip():
                        newHostFile.write(host)
                        continue

                delete = False
                update = False
                for toBeDeletedHost in toBeDeletedHosts:
                        # found
                        for hostPart in host.rsplit():
                            if hostPart==toBeDeletedHost["NAME"]:
                                delete = True
                                break
                for toBeAddedHost in toBeAddedHosts:
                        # found
                        for hostPart in host.rsplit():
                            if hostPart==toBeAddedHost["NAME"]:
                                update = True
                                break
                if not delete and not update:
                        print 'add host(%s) to %s' % (host[0:-1], fn)
                        newHostFile.write(host)

        # add new hosts or update existing hosts
        for toBeAddedHost in toBeAddedHosts:
                newHostFile.write("\n%s\t%s" % (toBeAddedHost["VALUE"], toBeAddedHost["NAME"]))

        newHostFile.close()

        
    def configureWatchfolders(self, watchfolders):
        print BaseClass.isOverride
        if watchfolders is None:
            return
        wfc = WatchFolderConfiguration.WatchFolderConfiguration()
        self.backupFile(wfc.filePath)
        for wf in watchfolders:
            if not wf.has_key("commandParameter"):
                commandParameter = None
            else:
                commandParameter = wf["commandParameter"]     
    
            if not wf.has_key("messageParameter"):
                messageParameter = None
            else:
                messageParameter = wf["messageParameter"]     
    
            if not wf.has_key("routingDir"):
                routingDir = ""
            else:
                routingDir = wf["routingDir"] 
                
            if not wf.has_key("failureDir"):
                failureDir = ""
            else:
                failureDir = wf["failureDir"] 
                
            if not wf.has_key("looseFileNamePattern"):
                looseFileNamePattern = ""
            else:
                looseFileNamePattern = wf["looseFileNamePattern"] 
                
            wfc.handle(wf["path"],wf["filter"],str(wf["frequency"]),wf["processClass"],wf["messageUID"],commandParameter,messageParameter,routingDir,failureDir, looseFileNamePattern, wf["events"],str(wf["threads"]),self.getIsDelete(wf))
    
    def configureDistTemplates(self, templates):
        if templates is None:
            return
        
        dt = DistTemplateConfiguration.DistTemplateConfiguration()
        self.backupFile(dt.filePath)
        for template in templates:
            dt.handle(template["NAME"], template["GROUPS"],self.getIsDelete(template))
    
    def configureFTPConfig(self, ftpConfig):
        if ftpConfig is None:
            return
        
        fc = FTPConfiguration.FTPConfiguration()
        self.backupFile(fc.filePath)
        for f in ftpConfig:
            fc.handle(f["HOST"], f["USER"], f["PASSWORD"],self.getIsDelete(f))
    
    def setupSFTPUpload(self, config):
        setupSFTP = self.getConfig("setupSFTP")
        if setupSFTP is not None and setupSFTP.lower() == 'true':
            print os.popen("sh scripts/SetupSFTPUpload.sh", "r").read()
    
    def doDataPatch(self, sqlFiles):
        if sqlFiles:
            if CMSVersionDeploymentPolicy.getCMSVersion() < 4.0:
                # construct a string like 'wfs/Wf$1234@dbserver:1521/ttv'
                dbSettings = self.config["oracle"]["user"] + "/" \
                        + self.config["oracle"]["password"].replace("$", "\$") + "@" \
                        + self.config["oracle"]["host"] + ":" \
                        + self.config["oracle"]["port"] + "/" \
                        + self.config["oracle"]["sid"]
                print os.popen('sh scripts/DataPatch.sh ' + dbSettings + ' "' + ' '.join(sqlFiles) + '"', "r").read()
            else:
                connection = CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().getDatabaseConnection()
                for sf in sqlFiles:
                    print 'Starting to execute datapath: ' + sf
                    with open(sf, 'r') as f:
                        connection.cursor().execute( f.read() )
                connection.commit()

    def configureBatonReportSymbolicLink(self):
        tree = ElementTree()
        tree.parse('/opt/tandbergtv/cms/workflow/preferences/batonqc.xml')
        entries = tree.getroot().getiterator("entry")
        reportEnable = ""
        symbolicLinkName = ""
        for e in entries:
            if e.attrib["key"] == "report.enable":
                reportEnable = e.text
        for e in entries:
            if e.attrib["key"] == "report.symbolic.link.name":
                symbolicLinkName = e.text
        if reportEnable == "true" and symbolicLinkName != "":
            mountPrefix = self.getMountPrefix()
            if not os.path.exists(mountPrefix + "/"+ symbolicLinkName):
                os.mkdir(mountPrefix + "/"+ symbolicLinkName)
                self.chown('nobody', 'nobody', mountPrefix + "/"+ symbolicLinkName, True)
            linkFilePath = "/opt/tandbergtv/watchpoint/tomcat/webapps/baton-reports/" + symbolicLinkName
            self.createSymlink(mountPrefix + "/"+ symbolicLinkName, linkFilePath)
            os.popen("chown %s%s:%s %s" % ("-h ", "nobody", "nobody", linkFilePath))
    
    def configureVerifierReport(self):
        tree = ElementTree()
        tree.parse('/opt/tandbergtv/cms/workflow/preferences/verifier.xml')
        entries = tree.getroot().getiterator("entry")
        reportEnable = ""
        symbolicLinkName = "verifier-reports"
        for e in entries:
            if e.attrib["key"] == "report.enable":
                reportEnable = e.text

        if reportEnable == "true" and symbolicLinkName != "":
            mountPrefix = self.getMountPrefix()
            if not os.path.exists(mountPrefix + "/"+ symbolicLinkName):
                os.mkdir(mountPrefix + "/"+ symbolicLinkName)
                self.chown('nobody', 'nobody', mountPrefix + "/"+ symbolicLinkName, True)
            linkFilePath = "/opt/tandbergtv/watchpoint/tomcat/webapps/verifier-reports"
            self.createSymlink(mountPrefix + "/"+ symbolicLinkName, linkFilePath)
            os.popen("chown %s%s:%s %s" % ("-h ", "nobody", "nobody", linkFilePath))

            
    def getMountPrefix(self):
        tree = ElementTree()
        tree.parse('/opt/tandbergtv/cms/workflow/preferences/system.xml')
        entries = tree.getroot().getiterator("entry")
        for e in entries:
            if e.attrib["key"] == "mount.prefix":
                return e.text
            
    def installRpm(self, rpmName):
        RPMCMDLINE = "/bin/rpm -Uvh --ignoreos --nodeps --force --replacepkgs " + "rpms/%s" % rpmName
        outputstr = 'command=' + RPMCMDLINE
        self.output('RPM installing.')
        self.output(outputstr)
        os.system(RPMCMDLINE)
                
    def initialSetUp(self):
        self.chown('root', 'root', '/etc/vsftpd', True)
        self.restartLocalServices(["vsftpd"])
        # Adding ads ftpuser
        self.output('Configure ftpusers ads and iptv')
        os.popen('useradd -d /content -m ads -p \$1\$Z6Giw8dM\$inoJyvC6w/7q27ukZabZB.')
        os.popen('useradd -d /content/ftp -m iptv; (echo iptv123; echo iptv123) | passwd iptv')
    
        # Adding required Ingest directories
        self.output('Verify and correct /content required directories')
        contentDirs = [
        '/content/adi',
        '/content/unmapped',
        '/content/mapped',
        '/content/derived',
        '/content/uploaded',
        '/content/txns_temp',
        '/home/cms/adi'
        ]
        map(self.makeDir, contentDirs)
            
        self.output('Creating FTP directories')

        #chown(user, group, path, recursive)
        self.chown("nobody", "nobody", "/home/cms/adi", True)
        
        self.output('Creating Content Ingest directories')
        dataDirs = [
        '/content/data',
        '/content/data/temp',
        '/content/data/dist',
        '/content/data/failed',
        '/content/data/ingest'
        ]
        map(self.makeDir, dataDirs)        
        for directory in dataDirs:
            self.chown('nobody', 'nobody', directory, True)        
        
        #self.checkTargetTitle()
        self.setupSFTPUpload(self.config)
    
    def waitForAllTemplatesIngested(self):
        retry_times = 0
        count = len(os.listdir('/opt/tandbergtv/cms/workflow/templates'))
        if count > 0:
            print count, "templates are not ready, waiting..."
        while True:
            cnt_templates = len(os.listdir('/opt/tandbergtv/cms/workflow/templates'))
            if cnt_templates == 0:
                print "All templates are ready now"
                break
            else:
                print 'Ingested templates:', count - cnt_templates, '/' , count
            retry_times = retry_times + 1
            time.sleep(5)
            if retry_times > 360: # 5 mins
                raise RuntimeError('Templates are not ready!!! please check the path "/opt/tandbergtv/cms/workflow/templates"')
    
    def importContentClass(self):
        username = self.config["cms"]["user"]
        password = self.config["cms"]["password"]
        mer_ct_cls_list = []
        del_ct_cls_list = []
        
        if self.getConfig("contentClasses") is not None:
            mer_ct_cls_list = self.getConfig("contentClasses")
        if self.getConfig("contentClassesToDelete") is not None:
            del_ct_cls_list = self.getConfig("contentClassesToDelete")

        if len(mer_ct_cls_list) or len(del_ct_cls_list):
            contentClassesList={}
            for mer_ct_cls_item in mer_ct_cls_list:
                if mer_ct_cls_item["NAME"] not in contentClassesList.keys():
                    contentClassesList.update({mer_ct_cls_item["NAME"]:mer_ct_cls_item["DESCRIPTION"]})
            for del_ct_cls_item in del_ct_cls_list:
                if del_ct_cls_item["NAME"] not in contentClassesList.keys():
                    contentClassesList.update({del_ct_cls_item["NAME"]:del_ct_cls_item["DESCRIPTION"]})       
            for ct_cls_item in contentClassesList.keys():
                mer_ct_cls=[]
                if self.helper.contentClassEverImported():
                    ct_cls_id=ContentClassHandler.getContentClassIdByName(ct_cls_item)
                    if ct_cls_id>0:
                        cur_cls_path = 'current_content_class_'+str(ct_cls_item)+time.strftime('%Y%m%d%H%M%S') + '.xml'
                        ContentClassHandler.exportById(cur_cls_path, ct_cls_id, username, password)
                        # The merge is base on the export's if ever import content class success
                        mer_ct_cls = [cur_cls_path]                 
                for mer_ct_cls_item in mer_ct_cls_list:
                    if ct_cls_item==mer_ct_cls_item["NAME"]:
                        mer_ct_cls=mer_ct_cls+[c['FILENAME'] for c in mer_ct_cls_item["CONTENTCLASSFILES"]]
                del_ct_cls=[]
                for del_ct_cls_item in del_ct_cls_list:
                    if ct_cls_item==del_ct_cls_item["NAME"]:
                        del_ct_cls=del_ct_cls+[c['FILENAME'] for c in del_ct_cls_item["CONTENTCLASSFILES"]]
                xmlPath = 'final_merged_content_class_'+ct_cls_item+'_'+time.strftime('%Y%m%d%H%M%S') + '.xml'
                print 'merge content class for '+ct_cls_item+': ', mer_ct_cls, del_ct_cls
                ContentClassHandler.mergeContentClasses(mer_ct_cls, del_ct_cls, xmlPath)
                ContentClassHandler.addHeadOfContentClass(ct_cls_item,contentClassesList[ct_cls_item],xmlPath)
                ContentClassHandler.importContentClass(xmlPath, username, password)
                self.helper.writeContentClassImportedFlagFile()
        else:
            print 'Content class no change.'
        print 'Import content class completed!'
    def updateDatabaseSchema(self):
        if CMSVersionDeploymentPolicy.getCMSVersion() == 3.1:
           dse = DatabaseSchemaExecutor.DatabaseSchemaExecutor()
           dse.execute()
           print 'Database schema update done.'
        
    def deploy(self):
        if not Common.isFreshInstall():
            BaseClass.setOverride()
        
        self.output("deploy for " + self.name)
        self.output(self.installerVersion)
        if Common.installMode is not None:
            self.output("Mode: " + Common.installMode)
        self.output('\n');
        
        self.stopServiceInClusterNodes(["cms", "workflow", "tomcat"])
        self.stopLocalServices(["cms", "workflow", "tomcat"])
        
        self.configureResourceHosts(self.getConfig("resourceHosts"))
 
        dnshostnames = self.getConfig("hostnames")
        failure = 0
 
        if dnshostnames is not None and len(dnshostnames):
            for hostname in dnshostnames:
                try:
                    self.output(hostname + " = " + socket.gethostbyname(hostname))
                except socket.gaierror:
                    self.output("'" + hostname + "' is not a resolvable hostname, please fix...")
                    failure = 1
            if failure == 1:
                self.output("exiting...")
                sys.exit(1)
 
#         # Update database schema
#         self.updateDatabaseSchema()
#         print 'Wait for database startup...'
#         time.sleep(15)
#         DatabaseConnectionProvider.getProvider().cleanCurrentConnection()
        
        # execute Scripts before installation
        self.executeScripts(self.getConfig("preInstallationScripts"))
        
        #backup related adapter properties files
        adapterProps = self.getConfig("AdapterProperties")
        if adapterProps:
            for f in adapterProps:
                self.backupFile(f)
        # Generate the full command line for installing all the RPMs
        #1. resource type "resourceTypeRpms"
        #2. resource group "resourceGroupRpms"
        #3. workflow    "templateRpms"
        #4. others        "otherRpms"
        resourceTypeRpms=self.getConfig("resourceTypeRpms")
        if resourceTypeRpms is not None and len(resourceTypeRpms):
            for f in resourceTypeRpms:
                self.installRpm(f)
        
        resourceGroupRpms=self.getConfig("resourceGroupRpms")
        if resourceGroupRpms is not None and len(resourceGroupRpms):
            for f in resourceGroupRpms:
                self.installRpm(f)
                
        templateRpms=self.getConfig("templateRpms")
        if templateRpms is not None and len(templateRpms):
            for f in templateRpms:
                self.installRpm(f)
            
        otherRpms=self.getConfig("otherRpms")
        if otherRpms is not None and len(otherRpms):
            for f in otherRpms:
                self.installRpm(f)
                
        commonRpms = self.getConfig("commonRpms")
        if commonRpms:
            for f in commonRpms:
                self.installRpm(f)
        
        ###########################################################################################
        # If it is fresh installatioin, setup it initially.
        if not Common.isUpgradeMode():
            self.initialSetUp()
        ###########################################################################################
        
        self.output("Creating required directories")
        directoriesToCreate=self.getConfig("directoriesToCreate")
        if directoriesToCreate is not None and len(directoriesToCreate):
            for directory in directoriesToCreate:
                self.makeDir(directory)
                self.chown('nobody', 'nobody', directory, True)
 
        self.restartLocalServices(["cms", "workflow"])
        
        # Wait for CMS to ingest adaptors and templates
        self.waitForAllTemplatesIngested()
        self.waitForOtherServersAvailable()
        
        
        # Disable or delete something
        self.unstallRpms(self.getConfig("rpmsToDelete"))
        self.disableProcessDefinitions(self.getConfig("templateToDelete"))
        self.deleteRuleSets(self.getConfig("ruleSetsToDelete"))
        
        
        self.insertSelectorKeys(self.getConfig("selectorKeys"))
        self.insertCustomFieldsGroups(self.getConfig("customFields"))
        self.insertResources(self.getConfig("resources"))
        self.insertSites(self.getConfig("sites"))
        self.insertAlerts(self.getConfig("alerts"))

        self.handleMeSubsProfiles(self.getConfig("meprofiles"))
                
        self.restartAlerts()   
        
        self.deleteConfigFiles(self.getConfig("configFileToDeleteDirs"))
        self.copyAllResouceFiles(self.getConfig("configFileDirs"), self.getConfig("params"))
        self.configureDistTemplates(self.getConfig("distTemplates"))
        self.configureWatchfolders(self.getConfig("watchfolders"))
        self.configureFTPConfig(self.getConfig("ftpConfig"))
        
        self.configureDatabase(self.getConfig("queries"))
 
        CMSVersionDeploymentPolicy.installAndConfigureReports(self.config)
        
        self.importContentClass()
        self.insertPartners(self.getConfig("partners"))
        self.importRules(self.getConfig("ruleSets"),self.config["cms"]["user"], self.config["cms"]["password"])
        
        self.sortRuleSet()
        
        #may need to add a config item to indicate whether baton report is required.
        if self.getConfig("batonReportEnable"):
            self.configureBatonReportSymbolicLink()
            
        if self.getConfig("verifierReportEnable"):
            self.configureVerifierReport()
  
        self.executeScripts(self.getConfig("postInstallationScripts"))
        self.doDataPatch(self.getConfig("dataPatchSqlFiles"))

        #sync files to slave nodes
        if (self.getConfig('cluster_app_nodes') and (not self.getConfig('ignoreFileSycn') or 
         (self.getConfig('ignoreFileSycn').upper() != 'Y' and
          self.getConfig('ignoreFileSycn').upper() != 'YES'))):
            print self.getConfig('cluster_app_nodes')
            sync_successed = FileSync(self.getConfig('cluster_app_nodes')).dosync()
            if not sync_successed:
                return False
        
        ###########################################################################################
        # vsftpd server should be restart after the anonymous account be added. otherwise,
        # openstream can not login to cms server to do package provision.
        # Notes: anonymous user is added in this step: 
        #    self.copyAllResouceFiles(self.getConfig("configFileDirs"), self.getConfig("params"))
        ###########################################################################################
        self.restartLocalServices(["vsftpd", "tomcat", "cms", "workflow"])
        self.restartServiceInClusterNodes(["vsftpd", "tomcat", "cms", "workflow"])
        
        # Finally, let the user know they're done
        self.output('Deployment complete.')
        return True
    
def main():
#    deploy = Deploy("/opt/tandbergtv/cms/prepack/installation_temp/combined-config.json_2015-05-06-01-49-18")
    deploy = Deploy("/root/prepack-aio-master-SNAPSHOT.1698/scripts/file.json")
#    deploy = Deploy("/root/prepack-aio-master-SNAPSHOT.1698/scripts/removeSD.json")
    deploy = Deploy("/root/prepack-aio-master-SNAPSHOT.1698/scripts/deployMERuleset.json")
    deploy.initPolicy();
    BaseClass.setOverride()
    deploy.deploy()
    
if __name__ == "__main__":
    main()
