#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Jerish Lin
# Created: Dec 11, 2012
# Description:
# Generic PrePackage Deployment Script
#

from InstallComponentCollector import InstallComponentCollector
from InstallConfigObjectConstructor import InstallConfigObjectConstructor
from InstallHelper import InstallHelper
from backup import Backup
from deploy import Deploy
from rollback import Rollback
from Common import Common
import Tee
import getopt
import os
import sys
import traceback
from CMSVersionDeploymentPolicy import CMSVersionDeploymentPolicy

class Install(Common):
    
    def __init__(self, systemInfoFile, mode):
        Common.__init__(self)
        self.loadJsonConfig(systemInfoFile, True)
        self.config["mode"] = mode
        self.initPolicy()
        self.initParameters()
        self.installComponentCollector = InstallComponentCollector()
       
    #===========================================================================
    # Run the license manager to get the license info.
    # The license info is output 3 lines. 
    # Line 1 is the license state, line 2 is the error message, the line 3 is the licensed features
    # result sample:
    # LIC_OK
    # 
    # DEVICES/TRANSCODE/ENVIVIO,DOWNSTREAM/OTT,DEVICES/TRANSCODE/ELEMENTAL,DOWNSTREAM/MEDIAROOM  
    def getLicenseInfo(self):
        import subprocess
        p = subprocess.Popen('java -jar ./rpms/prepack-license-manager*.jar', shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        lines = p.stdout.readlines() 
        if len(lines) == 3:
            return (lines[0].replace('\n',''), lines[1].replace('\n',''), lines[2].replace('\n','')); 
        return (False,False,False)
            
    def install(self, preConfiguredFile):
        state, errorMessge, components = self.getLicenseInfo()
        if state == 'LIC_NOTFOUND':
            print errorMessge
            return False
        elif state == 'LIC_CAPACITY_EXCEEDED':
            print errorMessge
            return False
        elif state == 'LIC_INVALID':
            print errorMessge
            return False
        elif state == False:
            print 'Failed to check license!'
            return False
        
        if not self.determineUpgradePath():
            return False
        
        if not self.validateVersionNumber():
            return False
        
        if not self.checkInstallerValidToRun():
            return False
        
        if not self.checkRequiredConfig():
            print '\nRequired configuration is not correct. Please fix it first.\n'
            return False
        
        #Start to collect User selections       
        componentPaths = components.split(',')
        result = self.installComponentCollector.processToCollectAllSelections(preConfiguredFile,not Common.isPatchMode(), componentPaths)
        if not result:
            return False
        fullInstallComponents, installedComponents, componentsToDelete, componentsToInstall, reselect = result
        availableUpgradeComponents = None
        if not Common.isFreshInstall():
            availableUpgradeComponents = self.installComponentCollector.collectAllUpgradeComponents(self.COMPONENTS_PATH_PREFIX, self.installerVersion)
        
        configObjectConstructor = InstallConfigObjectConstructor(fullInstallComponents, installedComponents, componentsToDelete, componentsToInstall, availableUpgradeComponents)
        isInstallNewOrRemove, isUpgradeOnly, isInvolveNothing = configObjectConstructor.getInstallStatus()
        if isInvolveNothing:
            self.output("There is no operations selected to handle. System would skip the installation.")
            return False
        #Start to construct the final config object    
        configObject = configObjectConstructor.constructFinalJsonObj(self.COMPONENTS_PATH_PREFIX, self.installerVersion)
        configObject = self.mergeConfig(configObject, self.config)
        self.helper.saveCombinedConfigObject(configObject)

        deployStarted, proceedBackup, self.backupVersion = self.decideBackupOrNotAndSolveBackupVersion(isInstallNewOrRemove)
        
        if reselect and deployStarted:
            #if user reselected and last deploy has been started, roll back to the last snapshot first
            self.output("System is going to roll back to last snapshot before installing the components you selected again.")
            rollbackResult = self.rollbackFirst()
            if not rollbackResult:
                self.output("System is encountering problem when rolling back to last snapshot, please contact the system administrator.")
                return False
        #Save the User Selection Only after roll back is done
        self.installComponentCollector.saveUserSelectionAsTemp(componentsToDelete, componentsToInstall)
                
        configObject["backupVersion"] = self.backupVersion
        if proceedBackup:
            if not self.backup(configObject):
                return False

        if not self.deploy(configObject):
            return False
        
        #Save the installed Components at the end.
        self.installComponentCollector.saveCurrentInstalledComponents(installedComponents, componentsToDelete, componentsToInstall)    
        self.output('Installation complete.')
        return True;
    
    def validateVersionNumber(self):
        if Common.isFreshInstall():
            if self.helper.isVersionUpgradeFormat(self.installerVersion):
                self.output("Fresh installation should have a version number without '-' in it, Please check the system info configuration file." )
                return False
        if Common.isUpgradeMode():
            if not self.helper.isVersionUpgradeFormat(self.installerVersion):
                self.output("Upgrade installation should have a version number with '-' in it indicating from a earlier version to the latest version, Please check the system info configuration file.")
                return False
        if Common.isPatchMode():
            if not self.helper.isVersionPatchFormat(self.installerVersion):
                self.output("Error: Patch installation should have a version number like '3.1.001', Please check the system info configuration file.")
                return False
        return True
    
    def checkInstallerValidToRun(self):
        currentInstalledPrepackVersion = self.helper.getCurrentPrepackVersion()
        patches = self.helper.getPatchesForCurrentPrepackVersion()
            
        if Common.isFreshInstall():
            if currentInstalledPrepackVersion is not None:
                compareResult = self.helper.compareVersion(currentInstalledPrepackVersion, self.installerVersion)
                if compareResult == 0:
                    if not self.helper.promptYesOrNo("Version " + currentInstalledPrepackVersion + " is already installed, Do you still want to rerun this? type 'yes' to continue rerun; type 'no' to skip.======> "):
                        return False
                if compareResult == 1:  
                    self.output("There is an earlier version " + currentInstalledPrepackVersion + " installed, Please run the upgrade installer instead.")
                    return False
                if compareResult == -1:
                    self.output("The current version is already up to " + currentInstalledPrepackVersion + ", it is not allowed to run an earlier version.")
                    return False
        if Common.isUpgradeMode():
            upgradePaths = self.getConfig("upgradePath");
            
            if not currentInstalledPrepackVersion:
                self.output("There is not any previous prepack version installed, Please check if the correct installer is used. Skipping...")
                return False
            else:
                fromPrepackVersion, toPrepackVersion = self.helper.getUpgradeFromTo(self.installerVersion)
                if self.helper.compareVersion(currentInstalledPrepackVersion, toPrepackVersion) == 0:
                    if patches:
                        self.output("Version " + currentInstalledPrepackVersion + " is already installed, But patches with version " + ','.join(patches) +" has been applied, Please run the rollback to get it back to version " + currentInstalledPrepackVersion + " instead of running this installer.")
                        return False
                    else:
                        if not self.helper.isCurrentVersionFromUpgrade():
                            self.output("Version " + currentInstalledPrepackVersion + " is installed, It is not required to run the upgrade")
                            return False
                        else:
                            self.output("Current Prepack has already been upgraded to version " + currentInstalledPrepackVersion + ", it is not required to run the upgrade.");
                            return False
                else:
                    if self.helper.compareVersion(currentInstalledPrepackVersion, fromPrepackVersion) != 0:
                        self.output("The current version is: " + currentInstalledPrepackVersion + ", but the installer is expected to be upgraded based on version: " + fromPrepackVersion + ", Please check if the correct installer is used. Skipping...")
                        return False
                
                
                if upgradePaths is None:
                    self.output("Error: configuration error! upgrade path does not defined in upgrade_sysInfo.json, contact PDU for next " \
                                "level support.");
                    return False
                
                # check if current upgrade path is a valid upgrade
                if self.installerVersion not in upgradePaths:
                    self.output("Error: " + self.installerVersion + " is not a valid upgrade path for this pre-package! " \
                                    "This installer package only supports the following upgrade path: " + str(upgradePaths))
                    return False
                
        if Common.isPatchMode():
            patchOnVersin = self.helper.getVersionMain(self.installerVersion)
            if not currentInstalledPrepackVersion:
                self.output("There is not any previous prepack version installed, but the patch is expected to be applied to version: " + patchOnVersin + ", Please check if the correct installer is used. Skipping...")
                return False
            else:
                if self.helper.compareVersion(currentInstalledPrepackVersion, patchOnVersin) != 0:
                    self.output("The current version is: " + currentInstalledPrepackVersion + ", but the patch is expected to be applied to version: " + patchOnVersin + ", Please check if the correct installer is used. Skipping...")
                    return False
                if patches:
                    if self.installerVersion in patches:
                        if patches[len(patches)-1] == self.installerVersion:
                            if not self.helper.promptYesOrNo("Patch " + self.installerVersion + " is already installed, Do you still want to rerun this? type yes to continue rerun; type no to skip.======> "):
                                return False
                        else:
                            self.output("The lastest patches applied is " + patches[len(patches)-1] + ", you are not allowed to reinstall this patch any more.")
                            return False
                        
        return True
    
    def determineUpgradePath(self):
        """
        determine current upgrade path and update the installer version 
        """
        if not Common.isUpgradeMode():
            return True
        
        currentInstalledPrepackVersion = self.helper.getCurrentPrepackVersion()
        
        if currentInstalledPrepackVersion:
            self.installerVersion = currentInstalledPrepackVersion + "-" + self.installerVersion
            self.config["version"] = self.installerVersion
            self.output("installer version is: " + self.installerVersion);
            return True
        else:
            self.output("There is not any previous prepack version installed, Please check if the correct installer is used. Skipping...")
            return False
        
    def rollbackFirst(self):
        try:
            rol = Rollback('rollback_sysInfo.json')
            return rol.rollback(True)
        except Exception:
            print traceback.format_exc()
            return False
#         return True
    
    def decideBackupOrNotAndSolveBackupVersion(self, isInstallNewOrRemove):
        backupVersion = lastHandledBackupVersion = self.helper.getLastHandledBackupVersion()
        backupStarted, backupFinished, deployStarted, deployFinished = self.helper.getBackupDeployStatusForVersion(lastHandledBackupVersion)                    
        
        proceedBackup = False
        if not backupFinished:
            proceedBackup = True
            if backupStarted:
                self.output("Backup for current version: " + self.installerVersion + " failed on last execution, Last backup will be abandoned, new backup with current status will be executed.")
                self.helper.abandonCurrentBackup(backupVersion)
        else:
            if not deployStarted:
                if self.helper.promptYesOrNo("Backup for current version: " + self.installerVersion + " is done on last execution, but nothing has been deployed since the backup, type 'yes' to backup the current status, type 'no' to keep the status of last backup.=====> "):
                    proceedBackup = True
                    self.helper.abandonCurrentBackup(backupVersion)
            if deployFinished:
                if self.helper.isInstallerRerun(self.installerVersion) and isInstallNewOrRemove:
                    self.output("System will backup this snapshot because the installer is rerun to remove existing components or install more components ")
                    proceedBackup = True
                elif self.helper.isDifferentInstaller(backupVersion, self.installerVersion):
                    proceedBackup = True
        if proceedBackup:
            backupVersion = self.helper.appenTimeStampToBackupVersion(self.helper.solveBackupVersion(self.installerVersion))
            self.helper.saveLastHandledBackupVersion(backupVersion)
        return (deployStarted, proceedBackup, backupVersion)
    
    def checkRequiredConfig(self):
        """Try to check required configuration is available."""
        if CMSVersionDeploymentPolicy.getCMSVersion() >= 4.0:
            csnodes = self.getConfig("cluster_service_nodes")
            if csnodes is None:
                print('Configuration entry of "cluster_service_nodes" cannot be empty.')
                return False
            if type(csnodes) != type([]) or len(csnodes) <= 0:
                print('Error configuration of "cluster_service_nodes".')
                return False
            for cn in csnodes:
                if "ip" not in cn or cn["ip"] is None or cn["ip"] == "":
                    print('Error configuration of "cluster_service_nodes".')
                    return False
            
            appnodes = self.getConfig("cluster_app_nodes")
            if appnodes and type(appnodes) == type([]):
                for apn in appnodes:
                    if "ip" not in apn or apn["ip"] is None or apn["ip"] == "":
                        print('Error configuration of "cluster_app_nodes".')
                        return False
        
        return True
    
    def backup(self, inputfileOrObject):
        self.helper.logActions(InstallHelper.BACKUP, self.backupVersion, InstallHelper.START)
        result = self.doBackup(inputfileOrObject)
        if result:
            self.helper.logActions(InstallHelper.BACKUP, self.backupVersion, InstallHelper.FINISH)
            self.helper.saveNewBackupVersion(self.backupVersion)
        return result
#         return True

    def doBackup(self, inputfileOrObject):
        try:
            bak = Backup(inputfileOrObject)
            return bak.backup()
        except Exception:
            print traceback.format_exc()
            return False
        

    def deploy(self, inputfileOrObject):
        self.helper.logActions(InstallHelper.DEPLOY, self.backupVersion, InstallHelper.START)
        result = self.doDeploy(inputfileOrObject)
        if result:
            self.helper.logActions(InstallHelper.DEPLOY, self.backupVersion, InstallHelper.FINISH)
            self.helper.saveNewPrepackVersion(self.installerVersion)
        return result

    def doDeploy(self, inputfileOrObject):
        try:
            dep = Deploy(inputfileOrObject)
            return dep.deploy()
        except Exception:
            print traceback.format_exc()
            return False

###########################################################################
#
# Main
#
###########################################################################
def output(outputstr):
    print outputstr
    
def main():
    preConfigFile = None
    systemInfoFile = None
    mode = 'Fresh'
    try:
        opts, args = getopt.getopt(sys.argv[1:], "h:i:p:m:v", ["help", "ifile", "preconfigFile", "mode"])
    except getopt.GetoptError as err:
        output("install.py [-p preconfigFile]" + str(err))
        sys.exit(2)
    for o, a in opts:
        if o in ("-h", "--help"):
            output('install.py [-p preconfigFile] -m <mode>')
            sys.exit(0)
        elif o in ("-p", "--preconfigFile"):
            preConfigFile = a
        elif o in ("-i", "--ifile"):
            systemInfoFile = a         
        elif o in ("-m", "--mode"):
            mode = a   
    # Check and make sure script is run by 'root'
    userlogin = os.environ.get('LOGNAME')
    if userlogin != "root":
        output('You must run this as user root.')
        output('Exiting...')
        sys.exit(1)
    try:
        ins = Install(systemInfoFile, mode)
        result = ins.install(preConfigFile)
        if not result:
            output('Installation Failed Or Skipped.')
            sys.exit(2)
    except KeyboardInterrupt:
        output('')
        output('<CTRL>-C detected...Exiting...')
        sys.exit(2)
    finally:
        CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().cleanDatabaseConnection()
    sys.exit(0)
             
if __name__ == "__main__":
     logFile = Tee.Tee("install.log", "a", True)
     main()
