from InstallTestBase import InstallTestBase
import unittest
from InstallComponent import InstallComponent

class InstallTest(InstallTestBase):
    
    def testValidateInstallerForFreshInstall(self):
        installer = self._createInstaller("", "1.0-2.0")
        self._resetInputs([])
        logFile = self._doInstall(installer)
        self._verifyLogFromLogFile(logFile, [
            "Fresh installation should have a version number without '-' in it, Please check the system info configuration file."
        ])
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertEqual(0, len(self._getBackupFolders()))
       
    def xtestValidateInstallerForUpgrade(self):  
        installer = self._createInstaller("UPGRADE", "1.0")
        self._resetInputs([])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Upgrade installation should have a version number with '-' in it indicating from a earlier version to the latest version, Please check the system info configuration file."
        ])
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertEqual(0, len(self._getBackupFolders()))
      
    def testValidateInstallerForPatch(self):  
        installer = self._createInstaller("PATCH", "1.0")
        self._resetInputs([])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Error: Patch installation should have a version number like '3.1.001', Please check the system info configuration file."
        ])
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertEqual(0, len(self._getBackupFolders()))
        
    def testFreshInstallOnScratch(self):
        installer = self._createInstaller("", "1.1")
        self._resetInputs([ "1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
    
    def testUpgradeOnScratch(self):
        installer = self._createInstaller("UPGRADE", "2.0")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "There is not any previous prepack version installed, Please check if the correct installer is used. Skipping..."
        ])
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertEqual(0, len(self._getBackupFolders()))
        
    def testPatchOnScratch(self):
        installer = self._createInstaller("PATCH", "1.0.001")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "There is not any previous prepack version installed, but the patch is expected to be applied to version: 1.0, Please check if the correct installer is used. Skipping..."
        ])
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertEqual(0, len(self._getBackupFolders()))
      
    def testFreshInstallOnExistingVersionWithVersionLess(self):
        self.testFreshInstallOnScratch()
        
        installer = self._createInstaller("", "1.0")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "The current version is already up to 1.1, it is not allowed to run an earlier version."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
    
    def testFreshInstallOnExistingVersionWithVersionLarger(self):
        self.testFreshInstallOnScratch()
        
        installer = self._createInstaller("", "2.0")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "There is an earlier version 1.1 installed, Please run the upgrade installer instead."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
    
    def xtestFreshInstallOnExistingVersionWithSameVersionButWithoutExtraComponentsSelected(self):
        self.testFreshInstallOnScratch()
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes","","","yes","","","yes","yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Version 1.1 is already installed, Do you still want to rerun this? type 'yes' to continue rerun; type 'no' to skip.======>",
            "There is no operations selected to handle. System would skip the installation."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        
    def testFreshInstallOnExistingVersionWithSameVersionButWithExtraComponentsSelected(self):
        self.testFreshInstallOnScratch()
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes","","","yes","1","1","","yes","yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Version 1.1 is already installed, Do you still want to rerun this? type 'yes' to continue rerun; type 'no' to skip.======>",
            "System will backup this snapshot because the installer is rerun to remove existing components or install more components"
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1_rerun")))
    
    def xtestUpgradeOnExistingVersionWithBothFromToNotEqual(self):
        self.testFreshInstallOnScratch()
        
        installer = self._createInstaller("UPGRADE", "2.0")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "The current version is: 1.1, but the installer is expected to be upgraded based on version: 1.0, Please check if the correct installer is used. Skipping..."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
    
    def testUpgradeOnExistingVersionWithSameFromVersionWithoutExtraComponentsSelectedAndNoAvailableUpgrade(self):
        self.testFreshInstallOnScratch()
        
        self._initUpgradeComponents("1.1-2.0", [])
        installer = self._createInstaller("UPGRADE", "2.0")
        self._defineUpgradePaths(installer, ["1.1-2.0"])
        self._resetInputs(["","","yes","","","yes","yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "There is no operations selected to handle. System would skip the installation."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
    
    def testUpgradeOnExistingVersionWithSameFromVersionWithoutExtraComponentsSelectedAndWithAvailableUpgrade(self):
        self.testFreshInstallOnScratch()
          
        self._initUpgradeComponents("1.1-2.0", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("UPGRADE", "2.0")
        self._defineUpgradePaths(installer, ['1.1-2.0'])
        self._resetInputs(["", "", "yes", "", "", "yes", "yes"])
        self._doInstall(installer)
        self.assertEqual("2.0", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1-2.0")))
    
    def testUpgradeOnExistingVersionWithSameFromVersionWithExtraComponentsSelectedAndWithOutAvailableUpgrade(self):
        self.testFreshInstallOnScratch()
        
        self._initUpgradeComponents("1.1-2.0", [""])
        installer = self._createInstaller("UPGRADE", "2.0")
        self._defineUpgradePaths(installer, ['1.1-2.0'])
        self._resetInputs(["","","yes","1","1","","yes","yes"])
        self._doInstall(installer)
        self.assertEqual("2.0", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1-2.0")))
    
    def testInvalidUpgradePath(self):
        self.testFreshInstallOnScratch()
        
        self._initUpgradeComponents("1.1-1.2", [])
        installer = self._createInstaller("UPGRADE", "1.3")
        self._defineUpgradePaths(installer, ["1.2-1.3"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Error: 1.1-1.3 is not a valid upgrade path for this pre-package! This installer package only supports the following upgrade path: ['1.2-1.3']"
        ])
    
    def testUpgradeOnExistingFreshInstallVersionWithSameToVersion(self):
        self.testFreshInstallOnScratch()
        
        self._initUpgradeComponents("1.0-1.1", [])
        installer = self._createInstaller("UPGRADE", "1.1")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Version 1.1 is installed, It is not required to run the upgrade",
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
    
    def testUpgradeOnExistingVersionWithSameToVersionWithoutExtraComponentsSelected(self):
        self.testUpgradeOnExistingVersionWithSameFromVersionWithoutExtraComponentsSelectedAndWithAvailableUpgrade()
        
        self._initUpgradeComponents("1.1-2.0", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("UPGRADE", "2.0")
        self._resetInputs([])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Current Prepack has already been upgraded to version 2.0, it is not required to run the upgrade."
        ])
        self.assertEqual("2.0", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1-2.0")))
        
    def testSelectExtraComponentsToInstallAfterUpgradeDone(self):
        self.testUpgradeOnExistingVersionWithSameFromVersionWithoutExtraComponentsSelectedAndWithAvailableUpgrade()
        
        installer = self._createInstaller("", "2.0")
        self._resetInputs(["yes","","","yes","1","1","","yes","yes"])
        logFile = self._doInstall(installer, outputToFile=True)

        self.assertEqual("2.0", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(3, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1-2.0")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("2.0_rerun")))
    
    def testPatchOnExistingVersionWithDifferentMainVersion(self):
        self.testFreshInstallOnScratch()
        
        installer = self._createInstaller("PATCH", "2.0.001")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "The current version is: 1.1, but the patch is expected to be applied to version: 2.0, Please check if the correct installer is used. Skipping..."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        
    def testPatchOnExistingVersionWithSameMainVersion(self):
        self.testFreshInstallOnScratch()
          
        self._initUpgradeComponents("1.1.001", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("PATCH", "1.1.001")
        self._doInstall(installer)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
          
        self._initUpgradeComponents("1.1.002", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("PATCH", "1.1.002")
        self._doInstall(installer)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual("1.1.002", installer.helper.getPatchesForCurrentPrepackVersion()[1])
        self.assertEqual(3, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.002")))
        
    def testFreshInstallOnExistingPatchWithVersionLess(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        installer = self._createInstaller("", "1.0")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "The current version is already up to 1.1, it is not allowed to run an earlier version."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
    
    def testFreshInstallOnExistingPatchWithVersionLarger(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        installer = self._createInstaller("", "2.0")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "There is an earlier version 1.1 installed, Please run the upgrade installer instead."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
       
    def testFreshInstallOnExistingPatchWithSameVersion(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
         
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes","","","yes","1","1","","yes","yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Version 1.1 is already installed, Do you still want to rerun this? type 'yes' to continue rerun; type 'no' to skip.======>"
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual("1.1.002", installer.helper.getPatchesForCurrentPrepackVersion()[1])
        self.assertEqual(4, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.002")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1_rerun")))

    def testUpgradeOnExistingPatchWithSameToVersion(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        installer = self._createInstaller("UPGRADE", "1.1")
        self._defineUpgradePaths(installer, ["0.5-1.1"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Version 1.1 is already installed, But patches with version 1.1.001,1.1.002 has been applied, Please run the rollback to get it back to version 1.1 instead of running this installer."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
    
    def xtestUpgradeOnExistingPatchWithBothFromToNotEqual(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        installer = self._createInstaller("UPGRADE", "1.0-2.0")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "The current version is: 1.1, but the installer is expected to be upgraded based on version: 1.0, Please check if the correct installer is used. Skipping..."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())   
    
    def testPatchOnExistingPatchWithDifferentMainVersion(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        installer = self._createInstaller("PATCH", "2.0.001")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "The current version is: 1.1, but the patch is expected to be applied to version: 2.0, Please check if the correct installer is used. Skipping..."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
    
    def testPatchOnExistingPatchWithSamePatch(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        installer = self._createInstaller("PATCH", "1.1.002")
        self._resetInputs(["yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Patch 1.1.002 is already installed, Do you still want to rerun this? type yes to continue rerun; type no to skip.======>"
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual("1.1.002", installer.helper.getPatchesForCurrentPrepackVersion()[1])
        self.assertEqual(3, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.002")))
        
    def testPatchOnExistingPatchWithSamePatchButOtherPatchesRun(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        installer = self._createInstaller("PATCH", "1.1.001")
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "The lastest patches applied is 1.1.002, you are not allowed to reinstall this patch any more."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual("1.1.002", installer.helper.getPatchesForCurrentPrepackVersion()[1])
        self.assertEqual(3, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.002")))

    def testBackupForSameVersionAgainWhenLastBackupNotStart(self):
        installer = self._createInstaller("", "1.1", backupStart= False, backupFinish=False, deployStart=False, deployFinish=False)
        self._resetInputs(["1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(0, len(self._getBackupFolders()))
        self.assertEqual(0, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes"])
        self._doInstall(installer, outputToFile=True)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
    def testBackupForSameVersionAgainWhenLastBackupStartedButNotFinished(self):
        installer = self._createInstaller("", "1.1", backupStart= True, backupFinish=False, deployStart=False, deployFinish=False)
        self._resetInputs([ "1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Backup for current version: 1.1 failed on last execution, Last backup will be abandoned, new backup with current status will be executed."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupAbandonFolders()))
        self.assertEqual(1, len(self._getBackupAbandonFolderForVersion("1.1")))
        
    def testNotBackupForSameVersionAgainWhenLastBackupFinishedButDeployNotStart(self):
        installer = self._createInstaller("", "1.1", backupStart= True, backupFinish=True, deployStart=False, deployFinish=False)
        self._resetInputs([ "1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes","no"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Backup for current version: 1.1 is done on last execution, but nothing has been deployed since the backup, type 'yes' to backup the current status, type 'no' to keep the status of last backup.=====>"
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
    def testBackupForSameVersionAgainWhenLastBackupFinishedButDeployNotStart(self):
        installer = self._createInstaller("", "1.1", backupStart= True, backupFinish=True, deployStart=False, deployFinish=False)
        self._resetInputs(["1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes","yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Backup for current version: 1.1 is done on last execution, but nothing has been deployed since the backup, type 'yes' to backup the current status, type 'no' to keep the status of last backup.=====>"
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupAbandonFolders()))
        self.assertEqual(1, len(self._getBackupAbandonFolderForVersion("1.1")))
        
    def testBackupForSameVersionAgainWhenLastBackupFinishedAndDeployStartButNotFinish(self):
        installer = self._createInstaller("", "1.1", backupStart= True, backupFinish=True, deployStart=True, deployFinish=False)
        self._resetInputs([ "1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes"])
        self._doInstall(installer, outputToFile=True)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getRollbackFolders()))
        self.assertEqual(0, len(self._getRollbackFolderForVersion("1.1")))
        
    def testNotNeedRollbackFirstWhenReselectComponentsButDeployNotStart(self):
        installer = self._createInstaller("", "1.1", backupStart= True, backupFinish=True, deployStart=False, deployFinish=False)
        self._resetInputs(["1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["no","1", "1", "1 2", "yes", "yes","no"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Please noted: To re-select components, system will roll back the partially installed components of your last selections before installing your new selections. Your input ==>"
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getRollbackFolders()))
        self.assertEqual(0, len(self._getRollbackFolderForVersion("1.1")))
        
    def testNotNeedRollbackFirstWhenReselectComponentsAndDeployStarted(self):
        installer = self._createInstaller("", "1.1", backupStart= True, backupFinish=True, deployStart=True, deployFinish=False)
        self._resetInputs(["1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertTrue(installer.helper.getCurrentPrepackVersion() is None)
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["no", "1", "1", "1 2", "yes", "yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Please noted: To re-select components, system will roll back the partially installed components of your last selections before installing your new selections. Your input ==>",
            "System is going to roll back to last snapshot before installing the components you selected again."
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getRollbackFolders()))
        self.assertEqual(1, len(self._getRollbackFolderForVersion("1.1")))
        
    def xtestXPMPAutoSelectComponentToInstall(self):
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["6", "", "yes", "yes"])
        self._doInstall(installer)
        installedComponents = InstallComponent.fromJsonObject(installer.helper.loadInstalledComponents())
        self.assertEqual(2, len(installedComponents.getInstallComponentByName("DEVICES").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("COMBINER") is None)
        self.assertEqual(1, len(installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("QC").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("QC").getInstallComponentByName("VERIFIER") is None)
        self.assertEqual(1, len(installedComponents.getInstallComponentByName("DOWNSTREAMS").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DOWNSTREAMS").getInstallComponentByName("XPMP") is None)

    def xtestXPMPAutoSelectComponentToInstallAndThenReruntoReselect(self):
        self.testXPMPAutoSelectComponentToInstall()
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes", "", "", "yes", "1", "1", "yes", "yes"])
        self._doInstall(installer)
        installedComponents = InstallComponent.fromJsonObject(installer.helper.loadInstalledComponents())
        self.assertEqual(2, len(installedComponents.getInstallComponentByName("DEVICES").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("COMBINER") is None)
        self.assertEqual(2, len(installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("QC").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("QC").getInstallComponentByName("VERIFIER") is None)
        self.assertEqual(False, installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("QC").getInstallComponentByName("BATON") is None)
        self.assertEqual(1, len(installedComponents.getInstallComponentByName("DOWNSTREAMS").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DOWNSTREAMS").getInstallComponentByName("XPMP") is None)

    def xtestSelectBackToLast(self):
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["2","6","2","1", "no", "6", "","yes","yes"])
        self._doInstall(installer)
        installedComponents = InstallComponent.fromJsonObject(installer.helper.loadInstalledComponents())
        self.assertEqual(2, len(installedComponents.getInstallComponentByName("DEVICES").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("COMBINER") is None)
        self.assertEqual(1, len(installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("QC").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DEVICES").getInstallComponentByName("QC").getInstallComponentByName("VERIFIER") is None)
        self.assertEqual(1, len(installedComponents.getInstallComponentByName("DOWNSTREAMS").getSubComponents()))
        self.assertEqual(False, installedComponents.getInstallComponentByName("DOWNSTREAMS").getInstallComponentByName("XPMP") is None)

if __name__ == "__main__":
    unittest.main()     