#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Jerish Lin
# Created: Oct 17, 2013
# Description:
# Generic PrePackage Rollback Script
#

from BaseClass import BaseClass
from Common import Common
from InstallHelper import InstallHelper
from FileAssembler import ContentClassHandler
from dbutils import Dao
import Rulesets
import Tee
import errno
import os
import sys
import optparse
import re
import shutil
import traceback
from InstallComponentCollector import InstallComponentCollector
from InstallComponent import InstallComponent
from CMSVersionDeploymentPolicy import CMSVersionDeploymentPolicy
from filesync import FileSync
from SSHCommander import SSHCommander
from EnhancementRuleset import EnhancementRuleset

class Rollback(Common):

    def __init__(self, systemInfoFile):
        Common.__init__(self)
        self.loadJsonConfig(systemInfoFile)
        self.initPolicy()
        self.__conn = CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().getDatabaseConnection()
        self.__cursor = self.__conn.cursor()
    
    def rollbackTemplate(self):
        print "Rollback template version"
        self.__cursor.execute("update JBPM_PROCESSDEFINITION set PROCESSDEFINITIONTYPEID = 1")
        Dao.update(self.__cursor, self.getConfig("templateVersions"), "JBPM_PROCESSDEFINITION","ID_")
        self.__conn.commit()
    
    def rollbackRuleSet(self,origFilePath,username, password):
        if origFilePath is None or not os.path.exists(origFilePath):
            return
        
        orgRuleSetExportPath = origFilePath+"_orig_.xml"
        shutil.copyfile(origFilePath, orgRuleSetExportPath)
        
        curRuleSetExportPath = orgRuleSetExportPath + "_export_.xml"
        print "export current ruleset to " + curRuleSetExportPath
        self.exportRules("-101",curRuleSetExportPath, username, password)
        ruleSets = Rulesets.Rulesets(self.config["cms"]["user"], self.config["cms"]["password"])
        orgRuleSetIds = ruleSets.getUuidsFromXml(orgRuleSetExportPath)
        curRuleSetIds = ruleSets.getUuidsFromXml(curRuleSetExportPath)
        toBeDeleteRuleSetIds = [id for id in curRuleSetIds if id not in orgRuleSetIds]
        print "to be delete rule set ids: %s" % toBeDeleteRuleSetIds
        ruleSets.deleteRulSetsByUuid(toBeDeleteRuleSetIds)
        
        self.updateProfileId(orgRuleSetExportPath)
        
        self.importRulesFromXml(orgRuleSetExportPath, username, password)
        
        #rollback rule set status
        Dao.update(self.__cursor, self.getConfig("ruleSetStatus"),"TRE_RULESET", "RULESETID","RULENAME")
        self.__conn.commit()
        
    def updateProfileId(self, rulesetPath):
        """
        Description: Update the ME profile id with the actual profile name 
        """
        enhancement = EnhancementRuleset()
        enhancement.replaceWithProfileId(rulesetPath)
        
    def rollbackRecourceGroup(self):
        print 'rollback resource group'
        try:
            org_recourceGroups = self.getConfig("resourceGroups")
            cur_recourceGroups = Dao.export(self.__cursor, "SELECT RESOURCEGROUPID,NAME FROM TTV_RESOURCEGROUP")
            org_rgIds = [rg["RESOURCEGROUPID"] for rg in org_recourceGroups]
            durtyRecourceGroups = [rg for rg in cur_recourceGroups if rg["RESOURCEGROUPID"] not in org_rgIds ]
            # print org_rgIds, cur_recourceGroups, durtyRecourceGroups
            for rg in durtyRecourceGroups:
                if not self.__existsRelativeRecordForResourceGroup(rg["RESOURCEGROUPID"]) and rg["NAME"] != 'Cluster':
                    print 'UPDATE jbpm_node SET resourcegroupid=NULL WHERE resourcegroupid = %s' % rg["RESOURCEGROUPID"]
                    self.__cursor.execute('UPDATE jbpm_node SET resourcegroupid=NULL WHERE resourcegroupid = %s' % rg["RESOURCEGROUPID"])
                    print 'delete resource group: ' + str(rg)
                    Dao.delete(self.__cursor, rg, 'TTV_RESOURCEGROUP', 'RESOURCEGROUPID')
            self.__conn.commit()
        except Exception, e:
            print traceback.format_exc()
            print 'FAILED TO ROLLBACK RESOURCEGROUP'
            self.__conn.rollback()

    def __existsRelativeRecordForResourceGroup(self, resourceGruopId):
        sql = 'SELECT COUNT(*) FROM JBPM_TOKEN WHERE NODE_ IN (SELECT ID_ FROM JBPM_NODE WHERE RESOURCEGROUPID = %s)' % resourceGruopId
        self.__cursor.execute(sql)
        rowCount = self.__cursor.fetchone()[0]
        if rowCount != 0:
            return True
        else:
            sql = 'SELECT COUNT(*) FROM JBPM_LOG WHERE RESOURCEGROUPID =%s' % resourceGruopId
            self.__cursor.execute(sql)
            rowCount = self.__cursor.fetchone()[0]
            if rowCount != 0:
                return True
            else:
                return False
            
    def rollbackResourceType(self):
        try:
            print 'rollback resource type'
            """
            Delete the new resource types 
            Assume the relative resources and recourceGrouopShips and recourceGroup were deleted by before rollback steps
            """
            org_recourceTypes = self.getConfig("resourceTypes")
            cur_recourceTypes = Dao.export(self.__cursor, "select RESOURCETYPEID,NAME,RESOURCECONNECTIONTYPEID from TTV_RESOURCETYPE")
            org_rtIds = [rt["RESOURCETYPEID"] for rt in org_recourceTypes]
            durtyRecourceTypes = [rt for rt in cur_recourceTypes if rt["RESOURCETYPEID"] not in org_rtIds ]
            #print org_rtIds, cur_recourceTypes, durtyRecourceTypes
            for rt in durtyRecourceTypes:
                if not self.__existsRelativeRecordForResourceType(rt["RESOURCETYPEID"]) and rt["NAME"] != 'Cluster':
                    print 'delete resource type: ' + str(rt)
                    Dao.delete(self.__cursor, rt, 'TTV_RESOURCETYPE', 'RESOURCETYPEID')
            self.__conn.commit()
            #TODO update to original data? 
        except Exception, e:
            print traceback.format_exc()
            print 'FAILED TO ROLLBACK RESOURCETYPE'
            self.__conn.rollback()
        
    def __existsRelativeRecordForResourceType(self, resourceTypeId):
        sql = 'SELECT COUNT(*) FROM TTV_RESOURCEGROUP WHERE RESOURCETYPEID = %s' % resourceTypeId
        self.__cursor.execute(sql)
        rowCount = self.__cursor.fetchone()[0]
        return rowCount != 0
        
    # For CMS above 3.0, only rollbackFiles is enough
    # For CMS <= 3.0, need to publish the catalog
    def rollBackReports(self):
        CMSVersionDeploymentPolicy.getReportInstaller(self.config).configureCategoryAndReports()

    def rollbackQueries(self):
        print 'rollback queries'
        sqls = self.getConfig("rollBackQueries")
        if sqls:
            drop_view_sqls  = [ q for q in sqls if bool(re.match('drop view', q['QUERY'], re.I))]
            drop_sequence_sqls = [ q for q in sqls if bool(re.match('drop sequence', q['QUERY'], re.I))]
            drop_table_sqls = [ q for q in sqls if bool(re.match('drop table', q['QUERY'], re.I))]
            others_qls  =     [ q for q in sqls if bool(not re.match('drop', q['QUERY'], re.I))]
            sqls =  others_qls + drop_view_sqls + drop_sequence_sqls + drop_table_sqls
            self.configureDatabase(sqls)
        
    def rollbackQueriesData(self):
        print "rollback QueriesData"
        queriesData = self.getConfig('queriesData')
        print queriesData
        if queriesData:
            for dictObj in queriesData:
                Dao.update(self.__cursor, dictObj['data'], dictObj['TABLE'], *tuple(dictObj['ID_COLUMNS']))
            self.__conn.commit()
            
    def rollbackContentClasses(self):
        print 'rollback content classes'
        contentClassPaths = self.getConfig("contentClassBackupPaths")
        if contentClassPaths:
            for xmlPath in contentClassPaths: 
                ContentClassHandler.importContentClass(xmlPath,self.config["cms"]["user"],self.config["cms"]["password"])
            
    def rollbackCreatedDirectories(self):
        directoriesToRemove = self.getConfig('directoriesToRemove')
        if directoriesToRemove:
            sorted(directoriesToRemove,reverse = True)
            for d in directoriesToRemove:
                try:
                    os.rmdir(d)
                    print d + ' directory was deleted'
                except OSError as ex:
                    if ex.errno == errno.ENOTEMPTY:
                        print d + " directory not empty"
    
    def rollbackPrepackVersionInfo(self):
        """
            roll back prepack info files
        """
        prepackInfoBackupDir = self.getPrepackInfoBackupDir()
        self._rollbackFiles("prepackVersionInfoFileToDelete", prepackInfoBackupDir)
    
    def rollbackRemoteFiles(self):
        print "Rollback remote files"
        items = self.getConfig("remoteFilesToRestore")
        if items==None:
            return
        
        cmd = None
        for i in items:
            rmtPath = i['remotePath']
            cfgName = i['remoteNodeConfigName']
            remoteNodes = self.getConfig(cfgName)
            if remoteNodes==None or len(remoteNodes)==0:
                raise Exception('Required config entry has not found: ' + cfgName)
            
            # put file to shared folder (as temporary file)
            shared_file = Common.COMMON_SHARED_FOLDER + rmtPath
            d = os.path.dirname(shared_file)
            if not os.path.exists(d):
                os.makedirs(d)
            shutil.copyfile(self.getFileBackupDir() + rmtPath, shared_file)
            
            # sync
            for rnd in remoteNodes:
                print "Start to rollback a file to remote: " + rnd['ip']
                try:
                    cmd = SSHCommander(rnd['ip'], rnd['user'], rnd['password'])
                    rollback_cmd = 'su root -c "\cp -rf %s %s"' % (shared_file, rmtPath)
                    status, output = cmd.execute_interactive(rollback_cmd, rnd['rootPasswd'])
                    print output
                    if status != 0:
                        raise Exception("Failed to rollback file [" + rmtPath + "] on " + rnd['ip'])
                except Exception:
                    raise   # NOP, just raise this Exception to outside
                finally:
                    if cmd:
                        cmd.logout()
            
            # try to remove the temporary file in shared folder
            if os.path.isfile(shared_file):
                os.remove(shared_file)
            else:
                shutil.rmtree(shared_file, ignore_errors=True)

    def _rollback(self, rollbackJsonConfig):
        self.loadJsonConfig(rollbackJsonConfig, mergeWithExisting=True)
        self.initParameters()
        
        BaseClass.setOverride()
        
        self.output("Rollback for " + self.name)
        self.output(self.backupVersion)
        self.output('\n')

        self.stopServiceInClusterNodes(["cms", "workflow", "tomcat"])
        self.stopLocalServices(["cms", "workflow", "tomcat"])
        
        #Roll back files
        self.rollbackFiles()
        self.rollbackCreatedDirectories()
        
        #roll back database items
        self.rollbackTemplate()
        self.insertSelectorKeys(self.getConfig("selectorKeys"))
        self.insertCustomFieldsGroups(self.getConfig("customFields"))
        self.insertResources(self.getConfig("resources"))
        if CMSVersionDeploymentPolicy.getCMSVersion() < 4.0:
            self.insertAlerts(self.getConfig("alerts"))
#         else:
#             self.rollbackRemoteFiles()
        self.restartAlerts()
       
        self.rollbackRecourceGroup()
        self.rollbackResourceType()
        self.rollbackQueriesData()
        self.rollbackQueries()

        self.restartLocalServices(["cms", "workflow"])
        self.waitForOtherServersAvailable()

        self.rollBackReports()
        self.insertPartners(self.getConfig("partners"))
        self.handleMeSubsProfiles(self.getConfig("meprofiles"))
        self.rollbackContentClasses()
        self.rollbackRuleSet(self.getConfig("ruleSetsBackupPath"),self.config["cms"]["user"],self.config["cms"]["password"] )
        self.insertSites(self.getConfig("sites"))
        self.executeScripts(self.getConfig("rollbackPostScripts"))
 
        #sync files to slave nodes
        if (self.getConfig('cluster_app_nodes') and 
        (not self.getConfig('ignoreFileSycn') or 
         (self.getConfig('ignoreFileSycn').upper() != 'Y' and
          self.getConfig('ignoreFileSycn').upper() != 'YES'))):
            print self.getConfig('cluster_app_nodes')
            
            #put script to share folder if the "sync script" not exists
            sync_script_dir = '/opt/tandbergtv/cms/prepack/filesync/script'
            sync_script = sync_script_dir + os.sep + 'sync.py'
            if not os.path.exists(sync_script):
                if not os.path.isdir(sync_script_dir):
                    os.makedirs(sync_script_dir)
                running_script_dir = os.path.dirname(os.path.realpath(__file__))
                to_copy_script = os.path.dirname(running_script_dir) + os.sep + 'components/BASE/conf/opt/tandbergtv/cms/prepack/filesync/script/sync.py'
                shutil.copyfile(to_copy_script, sync_script)
                
            sync_successed = FileSync(self.getConfig('cluster_app_nodes')).dosync()
            if not sync_successed:
                return False
 
        self.restartLocalServices(["tomcat", "cms", "workflow"])
        self.restartServiceInClusterNodes(["tomcat", "cms", "workflow"])        
        # Finally, let the user know they're done
        self.output('Rollback complete.')
        return True
        
    def clean(self):
        CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().cleanDatabaseConnection()
        print 'clean database connection'
        
    def rollback(self, auto = False):
        if not self.helper.getLatestBackupVersion():
            self.output("No backup is currently available for roll back, please contact the system administrator.")
            return False
            
        self.backupVersion = self.helper.getLatestBackupVersion()
        
        #TODO add more informations here, eg. time stamp.
        fromVersion, toVersion = self.helper.getUpgradeFromTo(self.backupVersion)
        if auto:
            self.output("System is rolling back current Prepack from version " + toVersion + " to version " + fromVersion + ".")
        else:
            InstallHelper.printSeperateLine()
            self.output("System will roll back from current status to last snapshot, please refer below as detail:")
            InstallHelper.printHighlightMessage("Current Prepack Status: ")
            self.printCurrentPrepackInstallStatus()
            InstallHelper.printHighlightMessage("Last Snapshot Prepack Status: ")
            self.printBackupPrepackInstallStatus(self.backupVersion)
            InstallHelper.printSeperateLine()
            if not self.helper.promptYesOrNo("Are you sure you want to proceed? Type 'yes' to proceed, type 'no' to skip... "):
                return False
        rollbackJsonConfigPath = self.getBackupDir() + "/" + self.ROLLBACK_JSON
        self.helper.logActions(InstallHelper.ROLLBACK, self.backupVersion, InstallHelper.START)
        try:
            result = self._rollback(rollbackJsonConfigPath)
            if result:
                self.helper.logActions(InstallHelper.ROLLBACK, self.backupVersion, InstallHelper.FINISH)
                self.rollbackPrepackVersionInfo()
                self.helper.rollBackToLastVersion(self.backupVersion)
            return result
        except Exception:
            print traceback.format_exc()
            return False
        
    
    def printCurrentPrepackInstallStatus(self):
        prepackVersion = self.helper.getCurrentPrepackVersion();
        patches = self.helper.getPatchesForCurrentPrepackVersion()
        installedComponents = self.helper.loadInstalledComponents()
        self.printInstallStatus(prepackVersion, patches, installedComponents)
        
    def printBackupPrepackInstallStatus(self, backupVersion):
        prepackVersion = self.helper.getBackupPrepackVersion(backupVersion);
        patches = self.helper.getPatchesForBackupPrepackVersion(backupVersion)
        installedComponents = self.helper.loadBackupInstalledComponents(backupVersion)
        self.printInstallStatus(prepackVersion, patches, installedComponents)
    
    def printInstallStatus(self, prepackVersion, patches, installedComponents):
        preparckVersionDisplay = 'Scratch'
        if prepackVersion:
            preparckVersionDisplay = prepackVersion
        print 'Prepack Version: ' + preparckVersionDisplay
        patchesDisplay = 'None'
        if patches:
            patchesDisplay = ','.join(patches)
        print 'Installed Patches: ' + patchesDisplay
        print 'Installed Components: '
        if installedComponents:
            InstallComponentCollector().printInstallComponents(InstallComponent.fromJsonObject(installedComponents))
        else:
            print 'None'
   
###########################################################################
#
# Main
#
###########################################################################
def output(outputstr):
    print outputstr
    
def main():
    # Check and make sure script is run by 'root'
    userlogin = os.environ.get('LOGNAME')
    if userlogin != "root":
        output('You must run this as user root.')
        output('Exiting...')
        sys.exit(1)
    
    optparser = optparse.OptionParser()
    optparser.add_option("-i", "--sys-info",
                         action="store", type="string", dest="systemInfoFile",
                         help="Configuration for system information.")
    (options, args) = optparser.parse_args()
    print("options: {0}".format(options))
    
    rol = None
    try:
        rol = Rollback(options.systemInfoFile)
        result = rol.rollback()
        if not result:
            output('Rollback Failed Or Skipped.')
            sys.exit(2)
    except KeyboardInterrupt:
        output('')
        output('<CTRL>-C detected...Exiting...')
        sys.exit(2)
    except Exception:
        print traceback.format_exc()
    finally:
        if rol:
            rol.clean()
    sys.exit(0)
             
if __name__ == "__main__":
    logFile = Tee.Tee("rollback.log", "a")
    main()
