from InstallTestBase import InstallTestBase
import unittest
import os
from rollback import Rollback

class RollbackTest(InstallTestBase):
    
    def setUp(self):
        InstallTestBase.setUp(self)
        sysInfoPath = self._constructSysInfoFile('')
        self.rollback=RollbackMock(sysInfoPath)
        self._mockRollback()
    
    def _mockRollback(self):
        self._mockPrepackDecider(self.rollback)
        self._mockRawInputForHelper(self.rollback)
    
    def testFreshInstallOnScratch(self):
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["1 2", "1", "1", "1 2", "yes", "yes"])
        self._doInstall(installer)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        
    def testUpgradeOnExistingVersionWithSameFromVersionWithoutExtraComponentsSelectedAndWithAvailableUpgrade(self):
        self.testFreshInstallOnScratch()
          
        self._initUpgradeComponents("1.1-2.0", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("UPGRADE", "2.0")
        self._defineUpgradePaths(installer, ["1.1-2.0"])
        self._resetInputs(["", "", "yes", "", "", "yes", "yes"])
        self._doInstall(installer)
        self.assertEqual("2.0", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1-2.0")))
    
    def testPatchOnExistingVersionWithSameMainVersion(self):
        self.testFreshInstallOnScratch()
          
        self._initUpgradeComponents("1.1.001", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("PATCH", "1.1.001")
        self._doInstall(installer)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
          
        self._initUpgradeComponents("1.1.002", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("PATCH", "1.1.002")
        self._doInstall(installer)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual("1.1.002", installer.helper.getPatchesForCurrentPrepackVersion()[1])
        self.assertEqual(3, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.002")))

    def testFreshInstallOnExistingVersionWithSameVersionButWithExtraComponentsSelected(self):
        self.testFreshInstallOnScratch()
        
        installer = self._createInstaller("", "1.1")
        self._resetInputs(["yes","","","yes","1","1","","yes","yes"])
        logFile = self._doInstall(installer, outputToFile=True)
        self._verifyLogFromLogFile(logFile, [
            "Version 1.1 is already installed, Do you still want to rerun this? type 'yes' to continue rerun; type 'no' to skip.======>",
            "System will backup this snapshot because the installer is rerun to remove existing components or install more components"
        ])
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1_rerun")))

    def testRollBackFreshInstall(self):
        self.testFreshInstallOnScratch()
        self._resetInputs(["yes"])
        self.rollback.rollback()
        self.assertTrue(self.rollback.helper.getCurrentPrepackVersion() is None)
        self.assertEqual(0, len(self._getBackupFolders()))
        self.assertEqual(0, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(0, len(self._getBackupAbandonFolders()))
        self.assertEqual(0, len(self._getBackupAbandonFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getRollbackFolders()))
        self.assertEqual(1, len(self._getRollbackFolderForVersion("1.1")))
        
    def testRollBackOnUpgrade(self):
        self.testUpgradeOnExistingVersionWithSameFromVersionWithoutExtraComponentsSelectedAndWithAvailableUpgrade()
        
        self._resetInputs(["yes"])
        self.rollback.rollback()
        self.assertEqual("1.1", self.rollback.helper.getCurrentPrepackVersion())
        self.assertTrue(len(self.rollback.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getRollbackFolders()))
        self.assertEqual(1, len(self._getRollbackFolderForVersion("1.1-2.0")))
        
    def testRollBackOnPatches(self):
        self.testPatchOnExistingVersionWithSameMainVersion()
        
        self._resetInputs(["yes"])
        self.rollback.rollback()
        self.assertEqual("1.1", self.rollback.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(self.rollback.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.001", self.rollback.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.001")))
        self.assertEqual(1, len(self._getRollbackFolders()))
        self.assertEqual(1, len(self._getRollbackFolderForVersion("1.1.002")))
        
        self._resetInputs(["yes"])
        self.rollback.rollback()
        self.assertEqual("1.1", self.rollback.helper.getCurrentPrepackVersion())
        self.assertEqual(0, len(self.rollback.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(2, len(self._getRollbackFolders()))
        self.assertEqual(1, len(self._getRollbackFolderForVersion("1.1.001")))
        self.assertEqual(1, len(self._getRollbackFolderForVersion("1.1.002")))

    def testRollBackForInstallerRerun(self):
        self.testFreshInstallOnExistingVersionWithSameVersionButWithExtraComponentsSelected()
        
        self._resetInputs(["yes"])
        self.rollback.rollback()
        self.assertEqual("1.1", self.rollback.helper.getCurrentPrepackVersion())
        self.assertTrue(len(self.rollback.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(1, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getRollbackFolders()))
        self.assertEqual(1, len(self._getRollbackFolderForVersion("1.1_rerun")))

    def testRollbackFirstAndReInstallForFreshInstall(self):
        self.testRollBackFreshInstall()
        self.testFreshInstallOnScratch()
        
    def testRollbackFirstAndReInstallForUpgrade(self):
        self.testRollBackOnUpgrade()
        
        self._initUpgradeComponents("1.1-2.0", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("UPGRADE", "2.0")
        self._defineUpgradePaths(installer, ["1.1-2.0"])
        self._resetInputs(["", "", "yes", "", "", "yes", "yes"])
        self._doInstall(installer)
        self.assertEqual("2.0", installer.helper.getCurrentPrepackVersion())
        self.assertTrue(len(installer.helper.getPatchesForCurrentPrepackVersion()) == 0)
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1-2.0")))
        
    def testRollbackFirstAndReInstallForPatch(self):
        self.testRollBackOnPatches()
        
        self._initUpgradeComponents("1.1.002", ["DEVICES/QC/BATON"])
        installer = self._createInstaller("PATCH", "1.1.002")
        self._doInstall(installer)
        self.assertEqual("1.1", installer.helper.getCurrentPrepackVersion())
        self.assertEqual(1, len(installer.helper.getPatchesForCurrentPrepackVersion()))
        self.assertEqual("1.1.002", installer.helper.getPatchesForCurrentPrepackVersion()[0])
        self.assertEqual(2, len(self._getBackupFolders()))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1")))
        self.assertEqual(1, len(self._getBackupFolderForVersion("1.1.002")))

class RollbackMock(Rollback):
    def __init__(self, systemInfoFile):
        Rollback.__init__(self, systemInfoFile)
    
    def _rollback(self, rollbackJsonConfig):
        self.loadJsonConfig(rollbackJsonConfig)
        self.initParameters()
        self.rollbackFiles()
        return True
    
    def initPolicy(self):
        pass

if __name__ == "__main__":
    unittest.main() 