#!/bin/sh
# This script is used for automatically fetch ip address and ssh login credential.
# For CMS5.X, fetch ip from /root/config.json
# For CMS6.X, fetch ip from DNS service
#set -x
csNodes=()
appNodes=()
isConfirmed=false
cmsVersion=$1
sysInfoFile=$2
preFillPassword=$3
password=""
existingAppIpPrefix=""
currentUser=""
passwordRetry=0
currentHostIp=""

# Get Ip for current host
function getIpForCurrentHost() {
  local existingAppIpPrefix=$(echo ${appNodes[0]} | awk 'BEGIN {FS="."}{print $1}')

  local currentPublicAndPrivateIp=(`ifconfig | grep -Eo 'inet (addr:)?([0-9]*\.){3}[0-9]*' | grep -Eo '([0-9]*\.){3}[0-9]*' | grep -v '127.0.0.1'`)
  for curIp in ${currentPublicAndPrivateIp[@]}
  do
      local currentIpPrefix=$(echo $curIp | awk 'BEGIN {FS="."}{print $1}')
      if [[ $currentIpPrefix = $existingAppIpPrefix ]];then
         currentHostIp=$curIp
      fi
  done
}

# Get ip address from sys info file (patch_sysInfo.json or install_sysInfo.json)
function getIpFromInstallSys() {
  if [ ! -f $sysInfoFile ]
  then
    showInstallError "$sysInfoFile does not exist!"
  fi
  nodeType=$1
  shift
  local _result=$@
  local nodes=(`cat $sysInfoFile | python -c "import sys,json;data=json.loads(sys.stdin.read());
clusterNode=data['$nodeType'];
for cNode in clusterNode:
    print cNode"`)
   eval $_result='(${nodes[@]})'
}

# Clear up the ip address list under "cluster_service_nodes" or "cluster_app_nodes"
function clearIpList() {
  local nodeType=$1
  python -c "import sys,json;
installSysFile=open(\"$sysInfoFile\", \"r+\")
jsonData=json.load(installSysFile)
installSysFile.close()
clusterNodes=jsonData['$nodeType']
del clusterNodes[:]
installSysFile=open(\"$sysInfoFile\", \"w+\")
installSysFile.write(json.dumps(jsonData, indent=4))
installSysFile.close()"
}


# Append the ip address to "cluster_service_nodes" or "cluster_app_nodes"
function appendIpList() {
  local nodeType=$1
  local ipEntry=$2
  python -c "import sys,json;
installSysFile=open(\"$sysInfoFile\", \"r+\")
jsonData=json.load(installSysFile)
installSysFile.close()
clusterNodes=jsonData['$nodeType']
clusterNodes.append($ipEntry)
installSysFile=open(\"$sysInfoFile\", \"w+\")
installSysFile.write(json.dumps(jsonData, indent=4))
installSysFile.close()"
}

# Remove current host ip from appNodes array is exist
function excludeCurrentIpFromAppNodes() {
   for i in "${!appNodes[@]}"
   do
       if [[ ${appNodes[i]} = "${currentHostIp}" ]];then
         unset 'appNodes[i]'
       fi
   done
}

# Add current host ip to appNodes array is not exist
function appendCurrentIpToAppNodes() {
  local isFound=false
  for i in "${!appNodes[@]}"
  do
      if [[ ${appNodes[i]} = "${currentHostIp}" ]];then
            isFound=true
            break
      fi
  done

  if [ "$isFound" == false ];then
         appNodes+=($currentHostIp)
  fi
}

# Feed the ip back to sys info json
function feedInIpToInstallSys() {
   clearIpList "cluster_service_nodes"
   clearIpList "cluster_app_nodes"

   excludeCurrentIpFromAppNodes

   for csIp in "${csNodes[@]}"
   do
       appendIpList "cluster_service_nodes" \"$csIp\"
   done

   for appIp in "${appNodes[@]}"
   do
      appendIpList "cluster_app_nodes" \"$appIp\"
   done
}

# Get ip address from /root/config.json (private ip)
function getIpFromCmsConfigJson() {
  if [ ! -f /root/config.json ]; then
     showInstallError "/root/config.json does not exist!"
  fi
  nodeType=$1
  shift
  local _result=$@
  local nodes=(`cat /root/config.json | python -c "import sys,json;data=json.loads(sys.stdin.read());
servers=data['environment']['servers'];
list=[]
for se in servers:
    vms=se['vms']
    for vm in vms:
        if 'clusterGroup' in vm and  vm['clusterGroup'] == '$nodeType':
            list.append(vm['customerIP'])
print ' '.join(list)"`)
  eval $_result='(${nodes[@]})'
}

# Get ip address from DNS service
function getIpFromDNS() {
  domainName=$1
  shift
  local _result=$@
  local ipList=(`dig +short @cs1 $domainName`)
  if [ ${#ipList[@]} -eq 0 ]; then
     #Try to query from cs2
     ipList=(`dig +short @cs2 $domainName`)
     if [ ${#ipList[@]} -eq 0 ]; then
        showInstallError "$domainName cannot be found in DNS server!"
     fi
  fi
  eval $_result='(${ipList[@]})'
}

# Show error and exit the script
function showInstallError() {
  errorMsg=$1
  echo
  echo "Installation skipped! Caused by error: $errorMsg"
  echo "Please refer to relevant installation guide or user guide..."
  exit 1
}

function promptConfirmation() {
  if [ ${#csNodes[@]} -eq 0 ] || [ ${#appNodes[@]} -eq 0 ]; then
     isConfirmed=false
     return
  fi

  printf "\n########################\n"
  printf   "# CS NODE IPs:         #"
  printf "\n########################\n"
  for csIp in "${csNodes[@]}"
  do
     echo "--|-$csIp"
  done

  getIpForCurrentHost
  appendCurrentIpToAppNodes

  printf "\n########################\n"
  printf   "# APP NODE IPs:        #"
  printf "\n########################\n"
  for appIp in "${appNodes[@]}"
  do
     echo "--|-$appIp"
  done

  printf "\n"
  printf "Please confirm IP address, type 'yes' or 'no'. Your input==> ******************************************************************************************************************************* "

  while :;
  do
     read confirmYesOrNo
     #Upper case
     confirmYesOrNo=$(echo $confirmYesOrNo | tr [a-z] [A-Z])
     if [ "$confirmYesOrNo" == "YES" ];then
         isConfirmed=true
         return
     elif [ "$confirmYesOrNo" == "NO" ];then
         isConfirmed=false
         return
     else
         echo "Please type 'yes' or 'no'"
         continue
     fi
  done
}

function autoIpFeedIn() {
   getIpFromInstallSys "cluster_service_nodes" csNodes
   getIpFromInstallSys "cluster_app_nodes" appNodes

   promptConfirmation

   if [ "$isConfirmed" != "true" ]; then
       isAfterCMS60=$(echo "$cmsVersion >= 6.0" | bc)
       if [ "$isAfterCMS60" -eq 1 ]
       then
           echo ""
           echo "Trying to get Ip from DNS service..."
           getIpFromDNS "cs-nodes.service.consul" csNodes
           getIpFromDNS "app-nodes.service.consul" appNodes
       else
           echo ""
           echo "Trying to get Ip from /root/config.json..."
           getIpFromCmsConfigJson "cs" csNodes
           getIpFromCmsConfigJson "app" appNodes
        fi
   fi

   if [ "$isConfirmed" != "true" ]; then
      promptConfirmation
   fi

   if [ "$isConfirmed" != "true" ]; then
        echo
        echo "Installation skipped!"
        exit 0
   else
      feedInIpToInstallSys
   fi
}

function promptForPwdInput() {
    if [[ $passwordRetry < 3 ]]; then
       unset password
       if [ -z $preFillPassword ]; then
           prompt="Please enter "$currentHostIp"'s password: "
           while IFS= read -p "$prompt" -r -s -n 1 char
           do
                if [[ $char == $'\0' ]]; then
                   if [[ ${#password} > 0  ]]; then
                     break
                   fi
                echo
                echo "Password is blank, please enter password!"
                continue
                fi
                prompt='*'
                password+="$char"
           done
       else
           password=$preFillPassword
       fi
        pwdVerify
    else
       showInstallError "Password validation failed!"
    fi
}


function pwdVerify() {
    currentUser=$(who am i  | awk '{print $1}')
    python ./scripts/SSHCommander.pyc -i ${csNodes[0]} -u $currentUser -p $password
    if [[ $? != 0 ]]; then
       if [ ! -z $preFillPassword ]; then
            # Only one chance for prefill password
            showInstallError "Password validation failed!"
       fi
       passwordRetry=$((passwordRetry+1))
       echo
       echo "Password incorrect! Please try again."
       promptForPwdInput
    fi
    
}

echo "Start to perfrom ip address auto feed in..."
autoIpFeedIn
echo
echo "Start to verify password...."
promptForPwdInput

echo
echo "Ready to execute prepack installer..."
