from InstallComponent import InstallComponent
from InstallHelper import InstallHelper
from InstallComponentBuilder import InstallComponentBuilder

class ComponentSelectionConfig(object):
    ROOT_SELECTION_ITEMS = "selectionItems"
    BACK="BACK"
    
    def __init__(self):
        self.rootSelection = []
        self.selectionConfigs = []

    def getSelectionConfig(self, name):
        for selectionConfig in self.selectionConfigs:
            if selectionConfig.name == name:
                return selectionConfig
    
    def getBackDisplayName(self):
        return self.getSelectionConfig(self.BACK).displayName
    
    def appendSelectionConfig(self, selectionConfig):
        self.selectionConfigs.append(selectionConfig)
        
    def mergeSelectionConfig(self, anotherSelectionConfigs):
        if not anotherSelectionConfigs:
            return self
        
        for selection in anotherSelectionConfigs.rootSelection:
            exists = False
            for existingSelection in self.rootSelection:
                if existingSelection == selection:
                    exists = True
            if exists:
                self.rootSelection.append(selection)
                
        for selection in anotherSelectionConfigs.selectionConfigs:
            exists = False
            for existingSelection in self.selectionConfigs:
                if existingSelection == selection:
                    exists = True
            if  not exists:
                self.selectionConfigs.append(selection) 
        return self               
    
    def constructInstallComponents(self):
        return InstallComponent.createRootInstallComponentForList(self._constructInstallComponentsRecursively(self.rootSelection))
    
    def _constructInstallComponentsRecursively(self, selections):
        installComponents = []
        for selectionItem in selections:
            installComponent = InstallComponent(selectionItem)
            selectionConfig = self.getSelectionConfig(selectionItem)
            if selectionConfig and selectionConfig.selections:
                for childInstallComponent in self._constructInstallComponentsRecursively(selectionConfig.selections):
                    installComponent.addSubComponent(childInstallComponent)
            installComponents.append(installComponent)
        return installComponents
    
    def collectAvailableComponentsForSelectedItems(self,selectedItems):
        fullInstallComponents = self.constructInstallComponents()
        allAvailableComponents = InstallComponent.createEmptyInstallComponent()
        for selectItem in selectedItems:
            selectionConfig = self.getSelectionConfig(selectItem)
            if selectionConfig:
                if selectionConfig.availableComponents or selectionConfig.unAvailableComponents:
                    allAvailableComponents.merge(self._collectAvailableComponents(selectionConfig))
                else:
                    allAvailableComponents.merge(fullInstallComponents)
        return allAvailableComponents
    
    def _collectAvailableComponents(self, selectionConfig):
        builder = InstallComponentBuilder()
        fullInstallComponents = self.constructInstallComponents()
        if selectionConfig.availableComponents:
            return builder.buildInstallComponents(selectionConfig.availableComponents)
        elif selectionConfig.unAvailableComponents:
            return fullInstallComponents.resolveLeaves().copy().substract(builder.buildInstallComponents(selectionConfig.unAvailableComponents))
    
    def collectAutoSelectComponentsForSelectedItems(self,selectedItems):
        builder = InstallComponentBuilder()
        allAutoSelectComponents = InstallComponent.createEmptyInstallComponent()
        for selectItem in selectedItems:
            selectionConfig = self.getSelectionConfig(selectItem)
            if selectionConfig and selectionConfig.autoSelectComponents:
                allAutoSelectComponents.merge(builder.buildInstallComponents(selectionConfig.autoSelectComponents))
        return allAutoSelectComponents
    
    def collectAvailableComponentsExcludeAutoSelectComponents(self,selectedItems):
        fullInstallComponents = self.constructInstallComponents().resolveLeaves()
        allAutoSelectComponents = self.collectAutoSelectComponentsForSelectedItems(selectedItems)
        allAvailableComponents = self.collectAvailableComponentsForSelectedItems(selectedItems)
        return allAvailableComponents.resolveLeaves(fullInstallComponents).substract(allAutoSelectComponents).removeNonLeafComponentIfNoSubcomponents()

    @staticmethod
    def fromJson(config):
        componentSelectionConfig = ComponentSelectionConfig()
        for selection in config:
            if selection == ComponentSelectionConfig.ROOT_SELECTION_ITEMS:
                componentSelectionConfig.rootSelection = config[selection]
            else:
                componentSelection = ComponentSelection(selection)
                componentSelection.convertFromConfig(config[selection])
                componentSelectionConfig.appendSelectionConfig(componentSelection)
        return componentSelectionConfig
    
    @staticmethod
    def constructFromInstallComponents(installComponents, fullComponentSelectionConfig):
        '''
            construct the selections via the installed component, aim to prompt to select for component deletion.
            @param installComponents        installComponents object
        '''
        newComponentSelectionConfig = ComponentSelectionConfig()
        for installComponent in installComponents.getSubComponents():
            newComponentSelectionConfig.rootSelection.append(installComponent.componentName)
            selectionConfig = fullComponentSelectionConfig.getSelectionConfig(installComponent.componentName)
            if selectionConfig:
                ComponentSelectionConfig._handleChildComponents(installComponent, selectionConfig, newComponentSelectionConfig,fullComponentSelectionConfig)
        return newComponentSelectionConfig
    
    @staticmethod     
    def _handleChildComponents(installComponent, selectionConfig, newComponentSelectionConfig, fullComponentSelectionConfig):
        '''
            handle child component recursively
        '''
        newSelectionConfig = selectionConfig.copyWithoutSelection()
        if installComponent.hasSubComponent():
            for childInstallComponent in installComponent.getSubComponents():
                newSelectionConfig.appendSelection(childInstallComponent.componentName)
                childSelectionConfig = fullComponentSelectionConfig.getSelectionConfig(childInstallComponent.componentName)
                if childSelectionConfig:
                    ComponentSelectionConfig._handleChildComponents(childInstallComponent, childSelectionConfig, newComponentSelectionConfig, fullComponentSelectionConfig)
        newComponentSelectionConfig.appendSelectionConfig(newSelectionConfig)

class ComponentSelection(object):
    DISPLAY_NAME = "displayName"
    SELECTIONS = "selections"
    SINGLE_SELECTION = "singleSelection"
    AVAILABLE_COMPONENTS = "availableComponents"
    UNAVAILABLE_COMPONENTS = "unAvailableComponents"
    AUTOSELECT_COMPONENTS = "autoSelectComponents"
    
    def __init__(self, componentName):
        self.name = componentName
        self.displayName = ""
        self.selections = []
        self.singleSelection = False
        self.availableComponents = None
        self.unAvailableComponents = None
        self.autoSelectComponents = None
    
    def convertFromConfig(self, config):
        self.displayName = config[self.DISPLAY_NAME] if config.has_key(self.DISPLAY_NAME) else ""
        self.selections = config[self.SELECTIONS] if config.has_key(self.SELECTIONS) else []
        self.singleSelection = config[self.SINGLE_SELECTION] if config.has_key(self.SINGLE_SELECTION) else False
        self.availableComponents = config[self.AVAILABLE_COMPONENTS] if config.has_key(self.AVAILABLE_COMPONENTS) else None
        self.unAvailableComponents = config[self.UNAVAILABLE_COMPONENTS] if config.has_key(self.UNAVAILABLE_COMPONENTS) else None
        self.autoSelectComponents = config[self.AUTOSELECT_COMPONENTS] if config.has_key(self.AUTOSELECT_COMPONENTS) else None
    
    def appendSelection(self, selection):
        self.selections.append(selection)
    
    def copyWithoutSelection(self):
        componentSelection = ComponentSelection(self.name)
        componentSelection.displayName = self.displayName
        componentSelection.singleSelection = self.singleSelection
        componentSelection.availableComponents = self.availableComponents
        componentSelection.unAvailableComponents = self.unAvailableComponents
        componentSelection.autoSelectComponents = self.autoSelectComponents
        return componentSelection