#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Jerish Lin
# Created: Nov 4, 2013
# Description:
# Helper Class to check versions, etc.

import os
import time
from distutils.version import LooseVersion

from K8sUtil import *
import sys


class InstallHelper(object):
    PREPACK_DIR = "/opt/tandbergtv/cms/conf/prepack"

    BACKUP_DIR = PREPACK_DIR + os.sep + "backup-restore"

    PREPACK_VERSION_FILE = PREPACK_DIR + os.sep + "prepackVersion.dat"
    INSTALL_ACTION_LOG = PREPACK_DIR + os.sep + "installAction.log"
    INSTALL_STATUS_FILE = "installStatus.log"

    INSTALLED_COMPONENTS_FILE_NAME = "installedcomponent.json"
    INSTALLED_COMPONENTS = PREPACK_DIR + os.sep + INSTALLED_COMPONENTS_FILE_NAME
    INSTALLATION_TEMP = PREPACK_DIR + os.sep + "installation_temp"
    COMPONENTS_TO_HANDLE_COMBINED_CONFIG_FILE_NAME = "combined-config.json"
    FLAG_EVER_IMP_FILE_PATH = PREPACK_DIR + os.sep + 'content_class_ever_imported.dat'

    DEPLOY = "Deploy"
    START = "Start"
    FINISH = "Finish"

    LEGACY_PREPACK_DUMMY_VERSION = "0.5"

    def __init__(self):
        self.cmsRole = os.getenv("CMS_ROLE")
        self.ctEnabled = os.getenv("INTERNAL_CT_ENABLED")
        if not self.cmsRole:
            print "[ERROR] CMS Role is empty"
            sys.exit(2)
        print "CMS Role: {}, CT Enabled: {}".format(self.cmsRole, self.ctEnabled)

    def __overWriteFile(self, text, filePath):
        d = os.path.dirname(filePath)
        if not os.path.isdir(d):
            os.makedirs(d)
        with open(filePath, "w") as f:
            f.write(text)

    def __appendToFileAsNewLine(self, text, filePath):
        d = os.path.dirname(filePath)
        if not os.path.isdir(d):
            os.makedirs(d)
        newline = ''
        if os.path.exists(filePath) and open(filePath, 'r').read():
            newline = '\n'
        with open(filePath, "a") as f:
            f.write(newline + text)

    def __grepFile(self, words, filePath):
        foundRecords = []
        if os.path.exists(filePath):
            allRecords = open(filePath, 'r').readlines()
            for record in allRecords:
                found = True
                for word in words:
                    if record.find("[" + word + "]") == -1:
                        found = False
                        break
                if found:
                    foundRecords.append(record)
        return foundRecords

    def __removeLinesWithWords(self, words, filePath):
        if os.path.exists(filePath):
            allRecords = open(filePath, 'r').readlines()
            f = open(filePath, 'w')
            for record in allRecords:
                found = True
                for word in words:
                    if record.find("[" + word + "]") == -1:
                        found = False
                        break
                if not found:
                    f.write(record)

    def __readLastLine(self, filePath):
        if os.path.exists(filePath):
            alllines = open(filePath, 'r').readlines()
            if len(alllines) > 0:
                return alllines[len(alllines) - 1].replace('\n', '')
        return None

    def saveNewPrepackVersion(self, version):
        self.__overWriteFile(version, self.PREPACK_VERSION_FILE)
        K8sUtil.patch(CONFIGMAP, PREPACK_CONFIGMAP, CM_KEY_VERSION, version)

    def __getPrepackVersion(self, prepackVersionFile):
        prePackVersion = self.__readLastLine(prepackVersionFile)
        if not prePackVersion:
            return None
        if LooseVersion(prePackVersion) < LooseVersion("3.0"):
            return self.LEGACY_PREPACK_DUMMY_VERSION
        return prePackVersion

    def getBackupDir(self, backupVersion):
        return self.BACKUP_DIR + os.sep + backupVersion

    @staticmethod
    def saveCurrentInstalledComponents(currentInstalledComponents):
        InstallHelper.writeJsonToFile(currentInstalledComponents, InstallHelper.INSTALLED_COMPONENTS)

    def saveCombinedConfigObject(self, configObject):
        combinedPath= self.INSTALLATION_TEMP + os.sep + self.COMPONENTS_TO_HANDLE_COMBINED_CONFIG_FILE_NAME + "_" + time.strftime('%Y-%m-%d-%H-%M-%S')
        self.writeJsonToFile(configObject, combinedPath)
        return combinedPath

    def logActions(self, step, version, action):
        self.__appendToFileAsNewLine(time.strftime('%Y-%m-%d-%H-%M-%S') + ": [" + step + "] at Version [" + version + "] [" + action + "]", self.INSTALL_ACTION_LOG)
        self.logInstallStatus(step, version, action)

    def logInstallStatus(self, step, version, action):
        installStatusFile = self.getInstallStatusFile(version)
        InstallHelper.mkdirForFile(installStatusFile)
        if not len(self.__grepFile([step, action], installStatusFile)):
            self.__appendToFileAsNewLine("[" + step + "] [" + action + "]", installStatusFile)

    def getInstallStatusFile(self, backupVersion):
        return self.getBackupDir(backupVersion) + os.sep + self.INSTALL_STATUS_FILE

    @staticmethod
    def loadFromJson(jsonFile):
        if not os.path.exists(jsonFile):
            return None
        json_data = open(jsonFile)
        jsonObject = json.load(json_data)
        json_data.close()
        return jsonObject

    @staticmethod
    def writeJsonToFile(jsonObject, jsonFile):
        InstallHelper.mkdirForFile(jsonFile)
        with open(jsonFile, 'w') as f:
            json.dump(jsonObject, f, ensure_ascii=False, indent=4)

    @staticmethod
    def mkdirForFile(path):
        dirPath = os.path.dirname(path)
        if not os.path.isdir(dirPath):
            os.makedirs(dirPath)

    @staticmethod
    def printHighlightMessage(message):
        holderLength = 0
        if type(list()) == type(message):
            for m in message:
                if len(m) > holderLength:
                    holderLength = len(m)
            print "#"*(holderLength+4)
            for m in message:
                print "# " + m + " "*(holderLength-len(m)+1) + "#"
            print "#"*(holderLength+4)
        else:
            print "#"*(len(message)+4)
            print "# " + message +" #"
            print "#"*(len(message)+4)

    @staticmethod
    def printSeperateLine():
        print "*"*150
        print ""

    @staticmethod
    def printShortSeperateLine():
        print "-"*100
        print ""

    def appEnable(self):
        return self.cmsRole.lower() == 'core'

    def ctEnable(self):
        return self.cmsRole.lower() == 'externalct' or self.ctEnabled.lower() == 'true'
