#!/usr/bin/env python
import os
import shutil
import yaml
from KubernetesConstants import *


class Alerts:
	ALERT_CONFIG_MAP = "cms-logstash-alerts"
	ALERT_CONF_PATH = "/etc/cms/cms-logstash-alerts/"
	ALERT_CONF_FILE_NAME = "alerts.yaml"
	ALERT_CONF_FILE = ALERT_CONF_PATH + ALERT_CONF_FILE_NAME
	
	def __init__(self, isOverride=False):
		self.isOverride = isOverride
	
	def insertAlerts(self, alerts, stubCmmObj):
		self._loadConfig()

		for rec in alerts:
			alert = rec["ALERTS"][0]
			if stubCmmObj.getIsDelete(rec):
				self.delete({"IDENTIFIER": alert["IDENTIFIER"]})
			elif self.getAlertByIdentifier(alert["IDENTIFIER"]):
				if self.isOverride:
					self.update({'IDENTIFIER': alert["IDENTIFIER"], 'PATTERN': rec["PATTERN"],
						'THRESH_COUNT': alert["THRESH_COUNT"],
						'THRESH_SECS': alert["THRESH_SECS"], 'FREQ_COUNT': alert["FREQ_COUNT"],
						'FREQ_SECS': alert["FREQ_SECS"],
						'EMAIL': alert["EMAIL"], 'SNMP_TRAP': alert["SNMP_TRAP"]})
			else:
				self.insert({'IDENTIFIER': alert["IDENTIFIER"], 'PATTERN': rec["PATTERN"],
					'THRESH_COUNT': alert["THRESH_COUNT"],
					'THRESH_SECS': alert["THRESH_SECS"], 'FREQ_COUNT': alert["FREQ_COUNT"],
					'FREQ_SECS': alert["FREQ_SECS"],
					'EMAIL': alert["EMAIL"], 'SNMP_TRAP': alert["SNMP_TRAP"]})
		
		self.saveToFile()

	@staticmethod
	def isAlertFileExist():
		return os.path.isfile(Alerts.ALERT_CONF_FILE)
	
	def _loadConfig(self):
		self._localConfig = './alerts.%s.yaml' % str(os.getpid())
		
		self.syncConfigToLocal()
		with open(self._localConfig, 'r') as f:
			self._allContent = yaml.load(f.read())
			
		if self._allContent is None:
			self._allContent = []
			
	def syncConfigToLocal(self):
		print "Start to copy alert configuration file %s to %s" % (Alerts.ALERT_CONF_FILE, self._localConfig)
		shutil.copyfile(Alerts.ALERT_CONF_FILE, self._localConfig)
		
	def saveToFile(self):
		with open(self._localConfig, 'w') as f:
			yaml.dump(self._allContent, f, default_flow_style=False)
		
		replaceCommand = "kubectl create configmap %s -n %s --from-file=%s=%s -o yaml --dry-run |kubectl replace -f -" \
			% (Alerts.ALERT_CONFIG_MAP, CMS_NAMESPACE, Alerts.ALERT_CONF_FILE_NAME, self._localConfig)
		os.system(replaceCommand)
		print "Saved alerts to config map %s" % Alerts.ALERT_CONFIG_MAP
	
	def getAlertByIdentifier(self, idf):
		alertIdentifier = idf
		obj = None
		
		for elem in self._allContent:
			if 'alert' not in elem:
				continue
			
			alert = elem['alert']
			if 'name' in alert and alert['name']==alertIdentifier:
				obj = alert
				break
		
		return obj
	
	def insert(self, data):
		print('Adding alert: ' + str(data))
		freq = {'count': int(data['FREQ_COUNT']), 'time': Alerts.formatSeconds(int(data['FREQ_SECS']))}
		# NOTE: WORK-AROUND:
		#thres = {'count': int(data['THRESH_COUNT']), 'time': AlertPolicyForCMS4X.formatSeconds(int(data['THRESH_SECS']))}
		obj = {}
		obj['name'] = str(data['IDENTIFIER'])
		obj['pattern'] = str(data['PATTERN'])
		obj['frequency'] = freq
		if 'SNMP_TRAP' in data and data['SNMP_TRAP']:
			obj['trap'] = int(data['SNMP_TRAP'])
		if 'EMAIL' in data and data['EMAIL']:
			obj['email'] = str(data['EMAIL'])
		
		self._allContent.append({'alert': obj})
		return True
	
	def update(self, data):
		print('Updating alert: ' + str(data))
		obj = self.getAlertByIdentifier(data['IDENTIFIER'])
		if not obj:
			return
		
		obj['pattern'] = str(data['PATTERN'])
		if 'SNMP_TRAP' in data and data['SNMP_TRAP']:
			obj['trap'] = int(data['SNMP_TRAP'])
		if 'EMAIL' in data and data['EMAIL']:
			obj['email'] = str(data['EMAIL'])
		
		freq = {}
		freq['count'] = int(data['FREQ_COUNT'])
		freq['time'] = Alerts.formatSeconds(int(data['FREQ_SECS']))
		obj['frequency'] = freq
	
	# NOTE: WORK-AROUND:
	#thres = {}
	#thres['count'] = int(data['THRESH_COUNT'])
	#thres['time'] = AlertPolicyForCMS4X.formatSeconds(int(data['THRESH_SECS']))
	#obj['threshold'] = thres
	
	def delete(self, data):
		print('Deleting alert: ' + str(data))
		alertIdentifier = data['IDENTIFIER']
		for elem in self._allContent:
			if 'alert' not in elem:
				continue
			
			alert = elem['alert']
			if 'name' in alert and alert['name']==alertIdentifier:
				self._allContent.remove(elem)
				break
	
	@staticmethod
	def formatSeconds(secs):
		hr = secs / 3600
		secs = secs % 3600
		min = secs / 60
		secs = secs % 60
		return '%02d:%02d:%02d'% (hr, min, secs)
