#!/usr/bin/env python
import time
from lxml import etree
from HttpRequestDecorator import HttpRequestDecorator
import json
import sys
import re


class RulesetsRestMgr:
	RULE_SERVICE_LOCATION = "https://app:8443/services/rules"
	RULE_UTIL_LOCATION = "https://app:8443/services/rulesutil/ruleset"
	URI_PATH_EXPORT_RULESET_BY_ID = "https://app:8443/services/rulesservice/getRuleSets/"
	URI_PATH_EXPORT_RULESET_BY_UUID = "https://app:8443/services/rulesservice/getRuleSetsByUuid/"
	URI_PATH_IMPORT_RULESET = "https://app:8443/services/rulesservice/saveRuleSet/"
	MAX_RETRY_TIMES = 3
	INTERVAL_SECONDS_TO_RETRY = 10
	
	def __init__(self):
		self.header = {'Content-Type': 'application/json'}
		self.credential = self._buildRulesetEntityHeader()
		# use to replace rule status in xml, only use by import rule
		self.statusCompiler = None
	
	def queryRulesetsWithStatus(self):
		rulesetStatusMapping = {}
		
		filter = {'filter': 'enable:EQUALS:true'}
		result = self.listRulesets(filter)
		if not result[0]:
			return False, None
		for rule in result[1]:
			rulesetStatusMapping[rule['uuid']] = 'true'
		
		filter = {'filter': 'enable:EQUALS:false'}
		result = self.listRulesets(filter)
		if not result[0]:
			return False, None
		for rule in result[1]:
			rulesetStatusMapping[rule['uuid']] = 'false'
		
		return True, rulesetStatusMapping
		
	def listRulesets(self, parameters):
		header = {'Accept': 'application/json'}
		
		url = self.RULE_SERVICE_LOCATION
		if parameters:
			url += "?"
			for (k, v) in parameters.items():
				url += k + "=" + v + "&"
			url = url[0: len(url)-1]
		
		print "list rulesets: ", url
		
		resp = HttpRequestDecorator.request(url, "GET", None, headers=header)
		if resp[0]["status"] != '200':
			print '[ERROR]', resp
			return False, None
		
		ruleSets = []
		resourceList = json.loads(resp[1])['resourceList']
		if resourceList:
			ruleSets = resourceList['ruleSet']
		return True, ruleSets
	
	def deleteRulSetsByUuid(self, uuids):
		"""
		This method is intend to delete the ruleSet specified by the uuids
			uuids: an array containing all the uuids of rulesets
		"""
		success = True
		
		if uuids:
			for uuid in uuids:
				success = self.deleteByuuid(uuid)
				
				if not success:
					break
		return success
	
	def deleteByuuid(self, uuid):
		url = self.RULE_SERVICE_LOCATION + "/" + uuid
		resp = HttpRequestDecorator.request(url, "DELETE", None, headers=self.header)
		if resp[0]["status"] != '200':
			print resp
			return False
		return True
	
	def getUuidsFromXml(self, path):
		"""
		This method is intend to delete the ruleset by the UUID
			path: the file path of the ruleset xml 
		"""
		ruleSetXml = etree.parse(path)
		ruleSets = ruleSetXml.getroot().find("ruleSets").getiterator("ruleSet")
		
		uuIds = []
		for ruleSet in ruleSets:
			uuid = ruleSet.find("uuid").text
			uuIds.append(uuid)
		
		return uuIds
	
	def sortByUuids(self, uuids, newlyImportedRuleSets):
		if not uuids:
			return True
		
		# Get all rule sets installed in CMS.
		rulesetsUrl = self.RULE_SERVICE_LOCATION + "?package=NORMALIZATION"
		resp = HttpRequestDecorator.request(rulesetsUrl, "GET", None, headers=self.header)
		if resp[0]["status"] != '200':
			print resp
			return False
		resourceList = json.loads(resp[1]).get("resourceList", "")
		installedRuleSets = [r['uuid'] for r in resourceList['ruleSet']]
		
		runBeforeUrl = self.RULE_SERVICE_LOCATION + "/{}?runBefore={}"
		runAfterUrl = self.RULE_SERVICE_LOCATION + "/{}?runAfter={}"
		lastSortedRuleSetUuid = None
		for uuid in uuids:
			if uuid not in installedRuleSets:
				continue
			if uuid not in newlyImportedRuleSets:
				# If the uuid is installed but not newly imported, then record it and no need to move.
				lastSortedRuleSetUuid = uuid
				continue
			
			if lastSortedRuleSetUuid:
				# The first sorted rule set had been updated
				moveResult = self.retryRuleSetMove(runAfterUrl.format(uuid, lastSortedRuleSetUuid))
			elif uuid != installedRuleSets[0]:
				moveResult = self.retryRuleSetMove(runBeforeUrl.format(uuid, installedRuleSets[0]))
			else:
				# The uuid we suppose to be the first one in CMS, and it's now, so not need to move.
				moveResult = True
			
			if moveResult:
				lastSortedRuleSetUuid = uuid
			else:
				return False
		return True
	
	def retryRuleSetMove(self, url, interval=5, timeout=60):
		timeToWait = timeout
		print "[INFO] Move rule set, url: ", url
		while True:
			try:
				resp = HttpRequestDecorator.request(url, "PATCH", None, headers=self.header)
				if resp[0]["status"] == '200':
					return True
				else:
					print "[WARN] Fail to move Rule Set, wait for " + str(interval) + " seconds then try again.", resp
			except Exception as e:
				print "[WARN] Fail to move Rule Set, wait for " + str(interval) + " seconds then try again.", e.message
			timeToWait = timeToWait - interval
			if timeToWait <= 0:
				break
			time.sleep(interval)
		print "[ERROR] Rule set move fail, url:", url
		return False
	
	def importRulesets(self, xmlPath, rulesetStatusMapping):
		if xmlPath is None:
			return
		xml = open(xmlPath, 'rb').read()
		if xml.find("<ruleSets/>") > -1:
			print "No Rules to import."
			return
		head = {'content-type': 'application/xml'}
		url = self.URI_PATH_IMPORT_RULESET
		
		if not self.statusCompiler:
			self.statusCompiler = re.compile("<enabled>.*</enabled>")
		
		nextRuleSetEnd = xml.index('</ruleSet>')
		while nextRuleSetEnd > -1:
			ruleSetXml = self.credential + xml[xml.index('<ruleSet '):nextRuleSetEnd + 10] + "</ruleSetToSave>"
			rulename = ruleSetXml[ruleSetXml.index('<ruleName>') + 10:ruleSetXml.index('</ruleName>')]
			ruleuuid = ruleSetXml[ruleSetXml.index('<uuid>') + 6:ruleSetXml.index('</uuid>')]

			existedRuleSet = self.getRuleSetByUUID(ruleuuid)
			if existedRuleSet[0]:
				ruleSetXml = self.combineSitesForRuleSet(existedRuleSet[1], ruleSetXml)

			if ruleuuid in rulesetStatusMapping:
				status = rulesetStatusMapping[ruleuuid]
				if status:
					ruleSetXml = self.statusCompiler.sub("<enabled>%s</enabled>" % status, ruleSetXml)
			
			print rulename
			xml = xml[nextRuleSetEnd + 10:]
			if not self._importRuleset(url, head, ruleSetXml.encode("utf-8")):
				print "import %s fail" % rulename
				print ruleSetXml
				sys.exit(1)
			
			try:
				nextRuleSetEnd = xml.index('</ruleSet>')
			except ValueError:
				nextRuleSetEnd = -1
	
	def _importRuleset(self, url, head, ruleSetXml):
		resp = HttpRequestDecorator.request(url, "POST", ruleSetXml.encode("utf-8"), headers=head)
		retryTimes = 0
		while retryTimes < RulesetsRestMgr.MAX_RETRY_TIMES:
			if self._checkIfImportSuccess(resp):
				return True
			else:
				print "Times:", retryTimes, resp
				time.sleep(RulesetsRestMgr.INTERVAL_SECONDS_TO_RETRY)
			retryTimes += 1
			return False

	def getRuleSetByUUID(self, uuid):
		header = {'Accept': 'application/xml'}
		url = self.RULE_SERVICE_LOCATION + "/" + uuid
		print "get rule set by uuid: ", url
		resp = HttpRequestDecorator.request(url, "GET", None, headers=header)
		if resp[0]["status"] != '200':
			print '[ERROR]', resp
			return False, None

		ruleSet = etree.fromstring(resp[1])
		return True, ruleSet

	def combineSitesForRuleSet(self, existedRuleSet, updatingRuleSetXml):
		if existedRuleSet is None or updatingRuleSetXml is None:
			return None

		updatingRuleSet = etree.fromstring(updatingRuleSetXml)
		updatingSites = updatingRuleSet.xpath(".//sites")[0]
		updatingSiteNames = updatingRuleSet.xpath(".//site/name/text()")
		for existedSiteName in existedRuleSet.xpath(".//site/name/text()"):
			if existedSiteName is not None and existedSiteName not in updatingSiteNames:
				updatingSites.append(existedSiteName.getparent().getparent())

		updatingRuleSetXml = etree.tostring(updatingRuleSet, encoding='utf8', method="xml", pretty_print=True)
		return updatingRuleSetXml

	@staticmethod
	def _checkIfImportSuccess(resp):
		return resp[0]["status"] == '204' or resp[0]["status"] == '200'
	
	@staticmethod
	def _buildRulesetEntityHeader():
		return "<ruleSetToSave><overwrite>true</overwrite>"




if __name__ == '__main__':
	ruleMgr = RulesetsRestMgr()
	# parameter = {'filter': 'enable:EQUALS:true', 'package': 'Targeting'}
	# result = ruleMgr.listRulesets(parameter)
	# print result
	
	"\n\n --------------------- \n\n"
	print ruleMgr.queryRulesetsWithStatus()
	
