#!/usr/bin/env python
#
# Copyright (c) 2012 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Bill Hood
# Author: Jerish Lin
# Created: Dec 11, 2012
# Description:
# Generic PrePackage Deployment Script
#

from __builtin__ import str

import DistTemplateConfiguration
import FTPConfiguration
import ProviderPropConfiguration
import WatchFolderConfiguration
from CMSVersionDeploymentPolicy import CMSVersionDeploymentPolicy
from Common import Common
from K8sUtil import *
from FileAssembler import ContentClassHandler
from FilemanagerRestMgr import FilemanagerRestMgr
from DeployFileHelper import *
from DeployTemplatesHelper import *
from DeploySecretsHelper import DeploySecretsHelper
from CertificateMigration import CertificateMigration


class Deploy(Common):
    
    def __init__(self, configObject):
        Common.__init__(self)
        self.config = configObject
        self.initParameters()
        self.combinedPath = None
        self.isFresh = self.isFreshInstall()
        self.output("[INFO] Is Fresh PrePack installation: {}".format(self.isFresh))

    def setCombinedConfigJsonPath(self, path):
        self.combinedPath = path

    def deleteConfigFiles(self, configFileToRemoveDirectories):
        if configFileToRemoveDirectories:
            for configFileDir in configFileToRemoveDirectories:
                self.deleteConfigFilesViaDirectory(configFileDir)
    
    @staticmethod
    def deleteConfigFilesViaDirectory(directory):
        for path, dirs, files in os.walk(os.path.abspath(directory)):
            for filename in files:
                targetFilePath = DeployFileHelper.getTargetFilePath(directory, os.path.join(path, filename))
                if os.path.exists(targetFilePath):
                    os.remove(targetFilePath)

    def configureWatchfolders(self, watchfolders):
        if watchfolders is None:
            return
        wfc = WatchFolderConfiguration.WatchFolderConfiguration()
        self.backupFile(wfc.filePath)
        for wf in watchfolders:
            if not wf.has_key("commandParameter"):
                commandParameter = None
            else:
                commandParameter = wf["commandParameter"]     
    
            if not wf.has_key("messageParameter"):
                messageParameter = None
            else:
                messageParameter = wf["messageParameter"]     
    
            if not wf.has_key("routingDir"):
                routingDir = ""
            else:
                routingDir = wf["routingDir"] 
                
            if not wf.has_key("failureDir"):
                failureDir = ""
            else:
                failureDir = wf["failureDir"] 
                
            if not wf.has_key("looseFileNamePattern"):
                looseFileNamePattern = ""
            else:
                looseFileNamePattern = wf["looseFileNamePattern"] 
                
            wfc.handle(wf["path"],wf["filter"],str(wf["frequency"]),wf["processClass"],wf["messageUID"],commandParameter,messageParameter,routingDir,failureDir, looseFileNamePattern, wf["events"],str(wf["threads"]),self.getIsDelete(wf))
    
    def configureDistTemplates(self, templates):
        if templates is None:
            return
        
        dt = DistTemplateConfiguration.DistTemplateConfiguration()
        self.backupFile(dt.filePath)
        for template in templates:
            dt.handle(template["NAME"], template["GROUPS"],self.getIsDelete(template))
    
    def configureFTPConfig(self, ftpConfig):
        if ftpConfig is None:
            return
        
        fc = FTPConfiguration.FTPConfiguration()
        for f in ftpConfig:
            fc.handle(f["HOST"], f["USER"], f["PASSWORD"],self.getIsDelete(f))

    def configureProviderProperties(self, providerProp):
        if providerProp is None:
            return
        pc = ProviderPropConfiguration.ProviderPropConfiguration()
        for p in providerProp:
            for psk in p["secretValues"]:
              pc.handle(p["secretKey"], psk["Key"], psk["Value"], self.getIsDelete(psk))

    
    def executeDataPatchInCmsDB(self, sqlFiles):
        if sqlFiles:
            connection = CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().getCmsDatabaseConnection()
            for sf in sqlFiles:
                print 'Starting to execute datapath: ' + sf
                with open(sf, 'r') as f:
                    connection.cursor().execute(f.read())
            connection.commit()

    @staticmethod
    def updateDeleteFileFilterRegex():
        """
                update filemanager neptune setting for retire title
        """
        deleteFileFilterRegex = ".*\\.(DTD|dtd|ce|m3u8|properties)"
        filemanagerRestMgr = FilemanagerRestMgr()
        regexResult = filemanagerRestMgr.getDeleteFileFilterRegex()

        if not regexResult[0]:
            return
        
        if regexResult[1]:
            print "File delete filter regex has already existed:", regexResult[1]
        else:
            filemanagerRestMgr.updateDeleteFileFilterRegex(deleteFileFilterRegex)

    def configureBatonReportSymbolicLink(self):
        # TODO baton report folder link creation
        return True
    
    def isFreshInstall(self):
        upgrade = os.getenv("UPGRADE")
        if upgrade and upgrade.lower() == 'true':
            print 'Upgrade mode specified by user input.'
            return False
        success, data = K8sUtil.getData(CONFIGMAP, PREPACK_CONFIGMAP)
        if not success:
            self.output('Failed to fetch configmap.')
            return False
        return CM_KEY_VERSION not in data

    def deployCommon(self):
        self.output("Install Prepack Version: " + self.installerVersion)

        # execute Scripts before installation
        self.executeScripts(self.getConfig("preInstallationScripts"))

        if self.helper.appEnable():
            # in order restart from twice to once, move some DB operation before the first restart.
            self.insertCustomFieldsGroups(self.getConfig("customFields"))
            self.handleMeSubsProfiles(self.getConfig("meprofiles"))
            self.executeSqlInWorkflowDB(self.getConfig("queries"))

            # using stop-start instead of restart: prevent jar missing when deploy some rpm files
            self.stopServices(["cms", "workflow"])
            
        no_replace_file_tuple = DeployFileHelper.get_no_replace_files(self.getConfig("noReplaceFiles"))
        DeployFileHelper.remove_deployed_files_with_exclude(no_replace_file_tuple)
        
        if self.helper.appEnable():
            print "[INFO] deploy app service related files"
            DeployFileHelper.deployRpmFiles(self.isFresh, self.getConfig("rpmFilesDirs"), no_replace_file_tuple)
        if self.helper.ctEnable():
            print "[INFO] deploy ct service related files"
            DeployFileHelper.deployRpmFiles(self.isFresh, self.getConfig("ctRpmFilesDirs"), no_replace_file_tuple)
            
        DeployFileHelper.record_deployed_files()
        
        if self.helper.appEnable():
            self.startServices(["cms", "workflow"])

    def deployCt(self):
        self.output("deploy for content transfer services")

        self.output("Creating required directories")
        directoriesToCreate = self.getConfig("ctDirectoriesToCreate")
        if directoriesToCreate is not None and len(directoriesToCreate):
            for directory in directoriesToCreate:
                self.makeDir(directory)

        DeployFileHelper.deployConfFiles(self.isFresh, self.getConfig("ctConfigFileDirs"))

        self.configureWatchfolders(self.getConfig("watchfolders"))
        self.configureFTPConfig(self.getConfig("ftpConfig"))

        if self.getConfig("batonReportEnable"):
            self.configureBatonReportSymbolicLink()

        self.updateDeleteFileFilterRegex()

    def deployApp(self):
        self.output("deploy for application services")

        DeployTemplatesHelper.deployTemplates()
        templatesToInstall = DeployTemplatesHelper.getTemplatesToInstall()

        # Wait for CMS workflow to ingest adaptors and templates
        DeployTemplatesHelper.waitForAllTemplatesIngested(templatesToInstall)

        if not self.isFresh:
            # should migrate first then deploy secrets
            CertificateMigration.migrate_certificate(self.getConfig("certToMigrate"))
        DeploySecretsHelper.deployFromJson(self.getConfig("secrets"))
        self.configureProviderProperties(self.getConfig("mePluginApiKey"))

        # TODO Consider `xxxToDelete` in containerization upgrade case
        self.disableProcessDefinitions(self.getConfig("templateToDelete"))
        self.deleteRuleSets(self.getConfig("ruleSetsToDelete"))

        self.insertResources(self.getConfig("resources"))
        self.updateResourceGroupsAllocation(self.getConfig("resourceGroupsAllocation"))

        self.handle_sites(self.getConfig("sites"))
        self.handle_sites(self.getConfig("updateSites"), True)

        self.insertAlerts(self.getConfig("alerts"))
        self.insertSelectorKeys(self.getConfig("selectorKeys"))

        DeployFileHelper.deployConfFiles(self.isFresh, self.getConfig("configFileDirs"))
        self.deleteConfigFiles(self.getConfig("configFileToDeleteDirs"))
        self.configureDistTemplates(self.getConfig("distTemplates"))

        ContentClassHandler.importContentClass(self.getConfig("contentClasses"), self.getConfig("contentClassesToDelete"))
        self.insertPartners(self.getConfig("partners"))
        self.importRules(self.getConfig("ruleSets"))

        # Sort the rule sets only in fresh installation: to prevent customized rule sets out of order
        if self.isFresh:
            self.sortRuleSet(self.getConfig("ruleSets"))
        else:
            # A short term solution to handle rule sets sorting in upgrade mode
            self.sortRuleSet(self.getConfig("ruleSetsToSorting"))

    def deploy(self):

        self.output("deploy for " + self.name)
        self.deployCommon()

        if self.helper.appEnable():
            self.deployApp()
            
        if self.helper.ctEnable():
            self.deployCt()
        
        self.executeScripts(self.getConfig("postInstallationScripts"))
        if not self.isFresh:
            self.executeScripts(self.getConfig("postUpgradeScripts"))

        if self.helper.appEnable():
            self.restartServices(["cms"])

        # Finally, let the user know they're done
        return True

    def deploySecondary(self):
        self.output("deploy for " + self.name + " on secondary site.")
        self.deployCommon()
        
        if self.helper.ctEnable():
            self.output("Creating required directories")
            directoriesToCreate = self.getConfig("ctDirectoriesToCreate")
            if directoriesToCreate is not None and len(directoriesToCreate):
                for directory in directoriesToCreate:
                    self.makeDir(directory)

            # may need to add a config item to indicate whether baton report is required.
            if self.getConfig("batonReportEnable"):
                self.configureBatonReportSymbolicLink()

        # Finally, let the user know they're done
        self.output('GEO secondary deployment complete.')
        return True

