#!/usr/bin/env python

import datetime
import json
from decimal import Decimal

def ispgsqldb(cursor):
    return 'psycopg2' in str(cursor.__class__)

class Dao(object):
    """
    DataBase Access Object
    """

    ######################################################################## insert
    @staticmethod
    def insert(cursor, data, table, *columns):
        """ insert data to specify table
        @param cursor       Database cursor
        
        @param data         A dictionary or a list of dictionaries;
        
        @param table        The table name
        
        @param columns      Specify the columns to be inserted to table
                            - None (default). all of columns base on the keys of dictionary will be inserted to table
        """
        if not data:
            return
        dicts = []
        if isinstance(data, dict):
            dicts = [data]
        elif isinstance(data, list):
            dicts = data
        else:
            print str(data) + ' is not support data type'
            return
        insert_columns = columns if columns else dicts[0].keys()
        columnsholder = ', '.join(insert_columns)
        if ispgsqldb(cursor):
            valuesholder = ', '.join(map(lambda x: '%%(%s)s' % x , insert_columns))
        else:
            valuesholder = ', '.join(map(lambda x: ':%s' % x , insert_columns))
        sql = 'INSERT INTO %s (%s) VALUES (%s)' % (table, columnsholder, valuesholder)
        if columns:
            # when execute the "insert SQL" by {cursor.executemany(sql, new_dictionaries)}, Python analysis all of element in dictionary to detect the column type,
            # It will raise error if the dictionary key do not contain the corresponding column
            # cx_Oracle.DatabaseError: ORA-01036: illegal variable name/number
            # If user specify which columns be insert,it is require to delete the useless elements for each dictionary to avoid the error
            cursor.executemany(sql, Dao.copyDicts(dicts, insert_columns))
        else:
            cursor.executemany(sql, dicts)
        print str(cursor.rowcount) + ' record(s) is inserted'

    ######################################################################## delete
    @staticmethod
    def delete(cursor, data, table, *id_columns):
        """ data data from specify table
        @param cursor       Database cursor
        
        @param data         A dictionary or a list of dictionaries;
        
        @param table        The table name
        
        @param columns      Specify the columns to be inserted to table
                            - None (default). all of columns base on the keys of dictionary will be inserted to table
        """
        if not data:
            return
        dicts = []
        if isinstance(data, dict):
            dicts = [data]
        elif isinstance(data, list):
            dicts = data
        else:
            print 'Un-support data type' + str(data)
            return

        if not id_columns:
            raise RuntimeError("You should assign at least one id column for table! the id columns usually be a unique key(s) in database")
        
        if ispgsqldb(cursor):
            criteriaHolders = ' and '.join(map(lambda x: '%s=%%(%s)s' % (x,x) , id_columns))
        else:
            criteriaHolders = ' and '.join(map(lambda x: '%s=:%s' % (x,x) , id_columns))
        
        sql = "DELETE FROM %s WHERE %s" % (table, criteriaHolders)
        print sql
        
        # when execute the "delete SQL" by {cursor.executemany(sql, new_dicts)}, Python analysis all of element in dictionary to detect the column type,
        # It will raise error if the dictionary key do not contain the corresponding column
        # cx_Oracle.DatabaseError: ORA-01036: illegal variable name/number
        # If user specify which columns be insert,it is require to delete the useless elements for each dictionary to avoid the error
        cursor.executemany(sql, Dao.copyDicts(dicts, id_columns))
        print str(cursor.rowcount) + ' record(s) is deleted'


    ######################################################################## update
    @staticmethod
    def update (cursor, data,  table, *id_columns, **kwargs):
        """ update data to specify table
        @param cursor       Database cursor
        
        @param data         A dictionary or a list of dictionaries;
        
        @param table        The table name
        
        @param id_columns   Specify a list of columns to identify a unique row in table
        
        @Keywords:
            update_columns  : list   Specify the columns to be updated:
                                    - None (default). Update all of columns base on the keys of dictionary
        """
        if not data:
            return
        dicts = []
        if isinstance(data, dict):
            dicts = [data]
        elif isinstance(data, list):
            dicts = data
        else:
            print 'un-support data type' + str(data)
            return
        if not id_columns:
            raise RuntimeError("You should assign at least one id column for table! the id columns usually be a unique key(s) in database")
        update_columns = kwargs.get('update_columns', dicts[0].keys())
        if ispgsqldb(cursor):
            setHolders = ', '.join(map(lambda x: '%s=%%(%s)s' % (x,x) , update_columns))
            criteriaHolders = ' and '.join(map(lambda x: '%s=%%(%s)s' % (x,x) , id_columns))
        else:
            setHolders = ', '.join(map(lambda x: '%s=:%s' % (x,x) , update_columns))
            criteriaHolders = ' and '.join(map(lambda x: '%s=:%s' % (x,x) , id_columns))
        sql = "UPDATE %s SET %s WHERE %s" % (table, setHolders, criteriaHolders)
        print sql
        if kwargs.get('update_columns'):
            dict_keys = list(id_columns) + update_columns
            cursor.executemany(sql, Dao.copyDicts(dicts, dict_keys))
        else:
            cursor.executemany(sql, dicts)
        print str(cursor.rowcount) + ' record(s) was updated'

    ######################################################################## others
    @staticmethod
    def export(cursor, sql, formatDate=False):
        """
        export the SQL result set to python dictionaries which reference to each rows in table.
        the column name map to the dictionary key,the column value map to the dictionary value,
        """
        cursor.execute(sql)
        obj = [dict((cursor.description[i][0].upper(), Dao.date2Str(formatDate, cursor.description[i][1], value)) for i, value in enumerate(row)) for row in cursor.fetchall()]
        
        return json.loads(json.dumps(obj, cls=IntEncoder, indent=4))

    @staticmethod
    def date2Str(formatDate, dbDataType, value):
        if formatDate:
            if hasattr(dbDataType,"__name__"):
                if (dbDataType.__name__ == 'DATETIME' or dbDataType.__name__ == 'TIMESTAMP') and value is not None:
                    return value.isoformat()
                else:
                    return value
            else:
                '''type_code 1114 is timestamp for postgresql'''
                if dbDataType == 1114: 
                    return value.isoformat()
                else:
                    return value
        else:
            return value

    @staticmethod
    def copyDicts(dicts, keys):
        new_dictionaries = []
        for d in dicts:
            new_dict = {}
            for k in keys:
                new_dict[k] = d[k]
            new_dictionaries.append(new_dict)
        return new_dictionaries

class DateTimeEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, datetime.datetime):
            # datetime.datetime.now().__repr__()
            # obj.strftime('%Y-%m-%d %H:%M:%S')
            return obj.isoformat()
        elif isinstance(obj, datetime.date):
            return obj.isoformat()
        elif isinstance(obj, datetime.timedelta):
            return (datetime.datetime.min + obj).time().isoformat()
        elif isinstance(obj, Decimal):
            return int(obj)
        else:
            return super(DateTimeEncoder, self).default(obj)

        
class IntEncoder(json.JSONEncoder):
    '''
        During export, Postgres will export numeric type to Decimal, 
        But Decimal can not be serialized to file, so need to convert Decimal to int type.
    '''
    def default(self, obj):
        if isinstance(obj, Decimal):
            return int(obj)
        else:
            return super(IntEncoder, self).default(obj)        
