import base64
import os
import sys
from lxml import etree
from migration import PropertiesUtils
from K8sUtil import K8sUtil
from KubernetesConstants import *

MAPPING_XML = "mapping.xml"
MIGRATED_RECORD_FILE = "/opt/tandbergtv/cms/conf/prepack/cert-migrated-record"
ENTRY_XPATH_IN_MAPPING = "/properties/entry"


class CertificateMigration:
	
	@staticmethod
	def migrate_certificate(cert_to_migrate_infos):
		if not cert_to_migrate_infos:
			return
		success, exist_secret_keys = K8sUtil.getData(SECRET, EXTERNAL_CREDENTAIL_SECRET)
		if not success:
			print "[ERROR] Fetching existing secret key from %s failure, exit installation" % EXTERNAL_CREDENTAIL_SECRET
			sys.exit(1)
		
		for certificate_info in cert_to_migrate_infos:
			if not CertificateMigration.__need_migrate(certificate_info):
				continue
			
			all_migrated_files = CertificateMigration.__migrate(certificate_info, exist_secret_keys)
			CertificateMigration.__remove_migrated_file(all_migrated_files)
			CertificateMigration.__record_migrated(certificate_info["name"])
	
	@staticmethod
	def __migrate(certificate_info, exist_secret_keys):
		certificate_path = certificate_info["certificate_path"]
		default_mapping = certificate_info["default_migrate_mapping"]
		migrate_entries = certificate_info["migrate_entries"]
		all_migrated_files = []
		
		resource_properties_mappings = CertificateMigration.__get_properties_files_from_mapping(certificate_path, default_mapping)
		for resource_proprties_mapping in resource_properties_mappings:
			migrated_files = CertificateMigration.__migrate_properties_file(exist_secret_keys, migrate_entries, certificate_path, resource_proprties_mapping)
			if migrated_files:
				all_migrated_files.extend(migrated_files)
		return set(all_migrated_files)

	@staticmethod
	def __migrate_properties_file(exist_secret_keys, migrate_entries, certificate_path, resource_proprties_mapping):
		properties_file_path = certificate_path + resource_proprties_mapping["properties_file"]
		if not os.path.isfile(properties_file_path):
			return
		props = PropertiesUtils.PropertiesUtils(properties_file_path)
		remove_entry_regex = r'%s\s*=[^\n]*\n?'
		migrated_files = []
		
		for migrate_entry in migrate_entries:
			property_key = migrate_entry["property_key"]
			secret_key = resource_proprties_mapping["resource_name"] + "." + migrate_entry["secret_key_suffix"]
			secret_value = props.get(property_key, None)
			isFile = ("type" in migrate_entry and migrate_entry["type"] == "file")
			
			if secret_key in exist_secret_keys or not secret_value:
				continue
				
			print "[INFO] Start to migrate certificate to secret for", property_key
			
			certificate_file = None
			if isFile:
				file_name = secret_value
				certificate_file = certificate_path + file_name
				if not (file_name and os.path.isfile(certificate_file)):
					continue
				with open(certificate_file) as f:
					secret_value = f.read()
			
			# patching secret
			successs = K8sUtil.patch(SECRET, EXTERNAL_CREDENTAIL_SECRET, secret_key, base64.b64encode(secret_value))
			if not successs:
				print "[ERROR] Adding certificate %(key)s to secret %(secret)s failure, exit installation" \
					% {'key': secret_key, 'secret': EXTERNAL_CREDENTAIL_SECRET}
				sys.exit(1)
			
			if isFile:
				migrated_files.append(certificate_file)
			# remove entry from properties file
			PropertiesUtils.writeFile(properties_file_path, remove_entry_regex % property_key, "")
			print "[INFO] Deleted entry %(key)s from %(file)s" % {'key': secret_key, 'file': properties_file_path}
		return migrated_files
	
	@staticmethod
	def __get_properties_files_from_mapping(certificate_path, default_properties):
		migrate_mappings = []
		mapping_file = certificate_path + MAPPING_XML
		if os.path.isfile(mapping_file):
			parser = etree.XMLParser(remove_blank_text=True, resolve_entities=False)
			tree = etree.parse(mapping_file, parser)
			entries = tree.xpath(ENTRY_XPATH_IN_MAPPING)
			for entry in entries:
				# {resource_name, properties_file_name}
				migrate_mappings.append({"resource_name": entry.attrib["key"], "properties_file": entry.text})
		
		if len(migrate_mappings) == 0:
			migrate_mappings.append(default_properties)
		return migrate_mappings
	
	@staticmethod
	def __has_migrated(adaptor_name):
		if os.path.isfile(MIGRATED_RECORD_FILE):
			with open(MIGRATED_RECORD_FILE) as f:
				for migrated_record in f:
					if migrated_record.strip() == adaptor_name:
						return True
		return False
	
	@staticmethod
	def __remove_migrated_file(migrated_files):
		for file_path in migrated_files:
			if os.path.isfile(file_path):
				os.remove(file_path)
				print "[INFO] Deleted file", file_path

	@staticmethod
	def __record_migrated(name):
		with open(MIGRATED_RECORD_FILE, 'a+') as f:
			f.write(name + "\n")

	@staticmethod
	def __need_migrate(certificate_info):
		name = certificate_info["name"]
		certificate_path = certificate_info["certificate_path"]
		
		if not (certificate_path and len(certificate_path) > 0):
			print "[INFO] The certificate_path is empty. skip the migrate for %s." % name
			return False
		if not (name and len(name) > 0):
			return False
		if CertificateMigration.__has_migrated(name):
			return False
		return True
