#!/usr/bin/env python
#
# Copyright (c) 2013 Ericsson, Inc.  All Rights Reserved.
#
# This module contains unpublished, confidential, proprietary
# material.  The use and dissemination of this material are
# governed by a license.  The above copyright notice does not
# evidence any actual or intended publication of this material.
#
# Author: Simon Meng
# Created: Dec 2, 2013
# Description: A class for merge Content Class
from operator import itemgetter
import time
import os
from urllib import urlencode
import xml.etree.ElementTree as ET
from xml.etree.ElementTree import Element
from HttpRequestDecorator import HttpRequestDecorator
from CMSVersionDeploymentPolicy import CMSVersionDeploymentPolicy

class ContentClassAssembler(object):
    
    def __init__(self, sortTags=False, fix_isHidden_field=False, fix_includeInBatch_field=False):
        self.__default_xml_tree = None
        self.fix_isHidden_field = fix_isHidden_field
        self.fix_includeInBatch_field = fix_includeInBatch_field
        # Define the arrays for identify which field needs to merge options
        self.merge_option_field_paths = ['/Fields/CustomFields/CustomField[@name=CP_Profile]/@value', '/Fields/CustomFields/CustomField[@name=Title_State_Machine_Profile]/@value', '/Fields/CustomFields/CustomField[@name=purgeStrategy]/@value']
        if sortTags:
            ET._serialize_xml = _serialize_xml_with_order # support Python 2.7+ only
        
    #===========================================================================
    # Load default content class
    #===========================================================================
    def loadDefault(self, filePath):
        """ load preset(default) content class
        @param filePath     The XML path of content class
        
        """
        self.__default_xml_tree = ET.parse(filePath)
    
    #===========================================================================
    # Merge with specify content class
    #===========================================================================
    def mergeWith(self, filePath):
        """ Merge with specify content class
        @param filePath     The XML path of content class
        
        """
        xml_tree = ET.parse(filePath)
        if self.__default_xml_tree is None:
            self.__default_xml_tree = xml_tree
            return
        self.__mergeAssets(xml_tree, self.__default_xml_tree)

    def deleteBy(self, filePath):
        """ Delete by specify content class
        @param filePath     The XML path of content class
        
        """
        if self.__default_xml_tree is None:
            print 'Please load default content class first'
            return
        assets = ET.parse(filePath).getroot().findall("metadata/asset")
        el_metadata = self.__default_xml_tree.getroot().find('metadata')
        tar_assets = self.__default_xml_tree.getroot().findall("metadata/asset")
        tar_asset_dict = dict((a.attrib["path"], a) for a in tar_assets)
        for asset in assets:
            asset_path = asset.attrib["path"]
            if tar_asset_dict.has_key(asset_path):
                tar_asset = tar_asset_dict.get(asset_path)
                if not list(asset):
                    el_metadata.remove(tar_asset)
                else:
                    self.__delFields(asset, tar_asset)
                        
    #===========================================================================
    # Delete fields
    #===========================================================================
    def __delFields(self, src_asset, tar_asset):
        src_fields = src_asset.findall("field")
        tar_fields = tar_asset.findall("field")
        tar_fields_dict = dict((f.attrib["path"], f) for f in tar_fields)
        for src_field in src_fields:
            path = src_field.attrib["path"]
            if tar_fields_dict.has_key(path):
                tar_field = tar_fields_dict.get(path)
                src_opts = src_field.findall('options/option')
                tar_opts = tar_field.findall('options/option')
                if not src_opts:
                    tar_asset.remove(tar_field)
                else:
                    tar_opts_field = tar_field.find('options')
                    tar_opts_dict = dict((opt.find('value').text, opt) for opt in tar_opts)
                    
                    #if tar_field.find('options'):
                    if path in self.merge_option_field_paths:
                        for opt in src_opts:
                            val = opt.find('value').text
                            if val in tar_opts_dict:
                                tar_opts_field.remove(tar_opts_dict.get(val))
                
    #===========================================================================
    # Merge assets
    #===========================================================================
    def __mergeAssets(self, src_xml_tree, tar_xml_tree):
        src_assets = src_xml_tree.getroot().findall("metadata/asset")
        tar_assets = tar_xml_tree.getroot().findall("metadata/asset")
        tar_asset_dict = dict((a.attrib["path"], a) for a in tar_assets)
        el_metadata = tar_xml_tree.getroot().find('metadata')
        for asset in src_assets:
            asset_path = asset.attrib["path"]
            if asset_path not in tar_asset_dict.keys():
                el_metadata.append(asset)
            else:
                tar_asset = tar_asset_dict.get(asset_path)
                self.__mergeAssetAttr(asset, tar_asset)
                self.__mergeFields(asset, tar_asset)
                
    #===========================================================================
    # Merge asset attribute to target asset
    #===========================================================================    
    def __mergeAssetAttr(self, src_asset, tar_asset):
        # Update to biggest 'max' attribute
        if 'max' in src_asset.attrib:
            if 'max' not in tar_asset.attrib or src_asset.attrib["max"] > tar_asset.attrib["max"]:
                tar_asset.attrib["max"] = src_asset.attrib["max"]
        
        # Update to smallest 'min' attribute
        if 'min' in src_asset.attrib:
            if 'min' not in tar_asset.attrib or src_asset.attrib["min"] < tar_asset.attrib["min"]:
                tar_asset.attrib["min"] = src_asset.attrib["min"]
        
        
    #===========================================================================
    # Replace the options, autofillProvider of tar_field by src_field,
    # merge the options if the field in the `tar_fields_dict`
    #===========================================================================
    def __mergeFields(self, src_asset, tar_asset):
        src_fields = src_asset.findall("field")
        tar_fields = tar_asset.findall("field")
        tar_fields_dict = dict((f.attrib["path"], f) for f in tar_fields)
        for field in src_fields:
            field_path = field.attrib["path"]
            if field_path not in tar_fields_dict.keys():
                tar_asset.append(field)
            else:
                tar_field = tar_fields_dict.get(field_path)
                if field_path in self.merge_option_field_paths:
                    self.__mergeOption(field, tar_field)
                else:
                    self.__overrideOption(field, tar_field)
                self.__overrideFieldAttr(field, tar_field)

    #===========================================================================
    # override target attributes by source
    #===========================================================================
    def __overrideFieldAttr(self, src_field, tar_field):
        if 'autofillProvider' in src_field.attrib and 'isAutofillable' in src_field.attrib:
            if src_field.attrib['autofillProvider'] and src_field.attrib['isAutofillable'] == 'true':
                tar_field.attrib['isAutofillable'] = 'true'
                tar_field.attrib['autofillProvider'] = src_field.attrib['autofillProvider']

    #===========================================================================
    # override target options by source
    #===========================================================================
    def __overrideOption(self, src_field, tar_field):
        src_opts_field = src_field.find('options')
        if src_opts_field is None:
            return
        
        tar_opts = tar_field.find('options')
        if tar_opts is not None:
            tar_field.remove(tar_opts)
        tar_field.append(src_opts_field)
            
    #===========================================================================
    # Merge file options
    #===========================================================================
    def __mergeOption(self, src_field, tar_field):   
        src_opts_field = src_field.find('options')
        if src_opts_field is None:
            return
        tar_opts_field = tar_field.find('options')
        if tar_opts_field is None:
            tar_field.append(src_opts_field)
            return
        
        src_opts = src_opts_field.findall('option')
        tar_opts = tar_opts_field.findall('option')
        tar_opts_dict = dict((opt.find('value').text, opt) for opt in tar_opts)
        for opt in src_opts:
            val = opt.find('value').text
            if val not in tar_opts_dict:
                tar_opts_field.append(opt)
            else:
                tar_opt = tar_opts_dict.get(val)
                index = tar_opts.index(tar_opt)
                tar_opts_field.remove(tar_opt)
                tar_opts_field.insert(index, opt)
                
    #===========================================================================
    # Write XML object to file
    #===========================================================================
    def write2file(self, outputPath):
        if self.__default_xml_tree is None:
            print "No XML Object to persistence!"
            return
        self.__default_xml_tree.write(outputPath)

def _get_attr_path(e):
    items  =  e.items()
    path = ''
    if items:
        for p in items:
            if p[0] == 'path':
                return p[1]
    return path
  
    
def _serialize_xml_with_order(write, elem, encoding, qnames, namespaces):
    tag = elem.tag
    text = elem.text
    if tag is ET.Comment:
        write("<!--%s-->" % ET._encode(text, encoding))
    elif tag is ET.ProcessingInstruction:
        write("<?%s?>" % ET._encode(text, encoding))
    else:
        tag = qnames[tag]
        if tag is None:
            if text:
                write(ET._escape_cdata(text, encoding))
            for e in elem:
                _serialize_xml_with_order(write, e, encoding, qnames, None)
        else:
            write("<" + tag)
            items = elem.items()
            if items or namespaces:
                if namespaces:
                    for v, k in sorted(namespaces.items(),
                                       key=lambda x: x[1]):  # sort on prefix
                        if k:
                            k = ":" + k
                        write(" xmlns%s=\"%s\"" % (
                            k.encode(encoding),
                            ET._escape_attrib(v, encoding)
                            ))
                        
                for k, v in items:  # lexical order (without order)
                    if k == 'path':
                        if isinstance(k, ET.QName):
                            k = k.text
                        if isinstance(v, ET.QName):
                            v = qnames[v.text]
                        else:
                            v = ET._escape_attrib(v, encoding)
                        write(" %s=\"%s\"" % (qnames[k], v))
                        break
                for k, v in sorted(items):  # lexical order (without order)
                    if k != 'path':
                        if isinstance(k, ET.QName):
                            k = k.text
                        if isinstance(v, ET.QName):
                            v = qnames[v.text]
                        else:
                            v = ET._escape_attrib(v, encoding)
                        write(" %s=\"%s\"" % (qnames[k], v))
            if text or len(elem):
                write(">")
                if text:
                    write(ET._escape_cdata(text, encoding))
                tags = [(_get_attr_path(e), e) for e in elem.getchildren()]
                tags.sort(key=itemgetter(0))
                tags = [e[1] for e in tags]
                for e in tags:
                    _serialize_xml_with_order(write, e, encoding, qnames, None)
                write("</" + tag + ">")
            else:
                write(" />")
    if elem.tail:
        write(ET._escape_cdata(elem.tail, encoding))


class ContentClassHandler(object):
    
    @staticmethod
    def importContentClass(contentClasses, contentClassesToDelete):
        mer_ct_cls_list = []
        del_ct_cls_list = []
        
        if contentClasses is not None:
            mer_ct_cls_list = contentClasses
        if contentClassesToDelete is not None:
            del_ct_cls_list = contentClassesToDelete
        
        if len(mer_ct_cls_list) or len(del_ct_cls_list):
            contentClassesList = {}
            for mer_ct_cls_item in mer_ct_cls_list:
                if mer_ct_cls_item["NAME"] not in contentClassesList.keys():
                    contentClassesList.update({mer_ct_cls_item["NAME"]: mer_ct_cls_item["DESCRIPTION"]})
            for del_ct_cls_item in del_ct_cls_list:
                if del_ct_cls_item["NAME"] not in contentClassesList.keys():
                    contentClassesList.update({del_ct_cls_item["NAME"]: del_ct_cls_item["DESCRIPTION"]})
            for ct_cls_item in contentClassesList.keys():
                mer_ct_cls = []
                ct_cls_id = ContentClassHandler.getContentClassIdByName(ct_cls_item)
                if ct_cls_id > 0:
                    cur_cls_path = 'current_content_class_' + str(ct_cls_item) + time.strftime(
                            '%Y%m%d%H%M%S') + '.xml'
                    ContentClassHandler.exportById(cur_cls_path, ct_cls_id)
                    # The merge is base on the export's if ever import content class success
                    mer_ct_cls = [cur_cls_path]
                for mer_ct_cls_item in mer_ct_cls_list:
                    if ct_cls_item == mer_ct_cls_item["NAME"]:
                        mer_ct_cls = mer_ct_cls + [c['FILENAME'] for c in mer_ct_cls_item["CONTENTCLASSFILES"]]
                del_ct_cls = []
                for del_ct_cls_item in del_ct_cls_list:
                    if ct_cls_item == del_ct_cls_item["NAME"]:
                        del_ct_cls = del_ct_cls + [c['FILENAME'] for c in del_ct_cls_item["CONTENTCLASSFILES"]]
                xmlPath = 'final_merged_content_class_' + ct_cls_item + '_' + time.strftime('%Y%m%d%H%M%S') + '.xml'
                print 'merge content class for ' + ct_cls_item + ': ', mer_ct_cls, del_ct_cls
                ContentClassHandler.mergeContentClasses(mer_ct_cls, del_ct_cls, xmlPath)
                ContentClassHandler.addHeadOfContentClass(ct_cls_item, contentClassesList[ct_cls_item], xmlPath)
                ContentClassHandler.__importContentClass(xmlPath)
        else:
            print 'Content class no change.'
        print 'Import content class completed!'

    #===========================================================================
    # Merge Multiple content class to combine to single Content Class
    #===========================================================================
    @staticmethod
    def mergeContentClasses(mer_content_classes, del_content_classes, outputPath):
        assembler = ContentClassAssembler(fix_isHidden_field=False)
        if mer_content_classes is not None:
            for c in mer_content_classes:
                assembler.mergeWith(c)
        if del_content_classes is not None:
            for c in del_content_classes:
                assembler.deleteBy(c)
        assembler.write2file(outputPath)
    
    #===============================================================================
    # Export Content Class by id
    # The Default Content Class id is "1"
    #===============================================================================
    @staticmethod
    def exportById(outputPath, ct_cls_id):
        """ export Content Class by id
        @param ct_cls_id            The content class id
        """
        head = {'content-type':'application/x-www-form-urlencoded'}
        data = {'id':ct_cls_id, 'partner':False}
        req_url = "https://app:8443/services/resteasy/contentclassservice/contentclassexport"
        resp = HttpRequestDecorator.request(req_url , "POST", urlencode(data), headers=head)
        if resp[0]["status"] != '200':
            print resp
            if "_ssl.c:492: error:100AE081" in resp:
                print "This Error is caused by a bug in Red Hat/CentOS, please contact the OS adminitrator to upgrade the OpenSSL rpm"
            raise RuntimeError('Failed to export content class')
        f = open(outputPath, 'wb')
        source=unicode(resp[1],'utf-8')
        f.write(source.encode('utf-8'))
        f.close()
       
    #===============================================================================
    # Import Content Class
    # 
    #===============================================================================
    @staticmethod
    def __importContentClass(xmlPath):
        """ Import Content Class
        @param xmlPath       The path of content class
        """
        if not os.path.exists(xmlPath):
            print xmlPath, " not exists!"
            return
        print "import content class ", xmlPath
        head = {'content-type':'application/x-www-form-urlencoded'}
        xmlstr = open(xmlPath, 'rb').read ()
        data = dict(xml=xmlstr, partner="ignore")
        req_url = "https://app:8443/services/resteasy/contentclassservice/contentclassimport"
        resp = HttpRequestDecorator.request(req_url , "POST", urlencode(data), headers=head)
        if resp[0]["status"] != '200':
            print resp
            raise RuntimeError('Failed to import content class ' + xmlPath)

    @staticmethod
    def getContentClassIdByName(content_class_name):
        """ Get Content Class ID by content class name through DataBase
        @param className     content class name
        """
        cursor=CMSVersionDeploymentPolicy.getPrepackDepolymentPolicy().getCmsDatabaseConnection().cursor()
        statement= "SELECT ID FROM PMM_CONTENT_CLASS WHERE NAME='%s'"
        cursor.execute(statement%content_class_name)
        contentClassId=cursor.fetchall()
        if len(contentClassId):
            return contentClassId[0][0]
        return -1

    @staticmethod
    def addHeadOfContentClass(content_class_name,content_class_description,xmlPath):
        """ Add Head information to merged content class file of each content class
        @param content_class_name     content class name
        @param content_class_description  content class description
        @param xmlPath       store file path after added head information
        """        
        contentClass = ET.parse(xmlPath)
        root=contentClass.getroot()
        if content_class_name== 'Default':
            root.set('isDefault','true')
        else:
            root.set('isDefault','false')      
        if root.find('name') is None:
            name=Element("name")
            name.text=content_class_name
            root.insert(0,name)     
        if root.find('description') is None:
            description=Element("description")
            description.text=content_class_description
            root.insert(1,description)
        if root.find('usage') is None:
            usage=Element("usage")
            usage.text="None"
            root.insert(2,usage)
        contentClass.write(xmlPath,"utf-8")
