package com.tandbergtv.watchpoint.studio.debugger.launching;

import static com.tandbergtv.watchpoint.studio.debugger.WatchpointTemplateDebuggerPlugin.getDefault;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.jdt.internal.launching.StandardVMDebugger;
import org.eclipse.jdt.launching.AbstractJavaLaunchConfigurationDelegate;
import org.eclipse.jdt.launching.ExecutionArguments;
import org.eclipse.jdt.launching.IVMRunner;
import org.eclipse.jdt.launching.VMRunnerConfiguration;
import org.osgi.framework.Bundle;

import com.tandbergtv.watchpoint.studio.debugger.model.JpdlCustomVMDebugger;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.interpreterwrapper.InterpreterWrapperServer;

/**
 * The Template launch configuration delegate.
 * 
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio -
 *         Francisco Bento da Silva Neto</a>
 * 
 */
public class TemplateLaunchConfigurationDelegate extends AbstractJavaLaunchConfigurationDelegate {

    public void launch(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor)
            throws CoreException {

        // define ports to be used by the wrapper
        int commandPort = -1;
        int eventPort = -1;
        if (mode.equals(ILaunchManager.DEBUG_MODE)) {
            commandPort = findFreePort();
            eventPort = findFreePort();
        }

        IVMRunner runner = getVMRunner(configuration, mode);
        // replace the runner by one which adds the custom target, if the runner is the one expected
        if (runner instanceof StandardVMDebugger) {
            runner = new JpdlCustomVMDebugger(verifyVMInstall(configuration), mode, commandPort, eventPort);
        }

        String workingPath = verifyWorkingDirectory(configuration) != null ? verifyWorkingDirectory(configuration)
                .getAbsolutePath() : null;

        // Environment variables
        String[] env = getEnvironment(configuration);

        // Program & VM arguments
        String pgmArgs = getTemplateSimulatorProgramArguments(configuration, commandPort, eventPort);
        String vmArgs = getVMArguments(configuration);
        ExecutionArguments execArgs = new ExecutionArguments(vmArgs, pgmArgs);

        // VM-specific attributes
        Map<?, ?> vmAttributesMap = getVMSpecificAttributesMap(configuration);

        // Classpath
        String[] classpath = getTemplateSimulatorClasspath(configuration);

        // Create VM config
        VMRunnerConfiguration runConfig = new VMRunnerConfiguration(InterpreterWrapperServer.class.getName(),
                classpath);

        runConfig.setProgramArguments(execArgs.getProgramArgumentsArray());
        runConfig.setEnvironment(env);
        runConfig.setVMArguments(execArgs.getVMArgumentsArray());
        runConfig.setWorkingDirectory(workingPath);
        runConfig.setVMSpecificAttributesMap(vmAttributesMap);

        // Bootpath
        runConfig.setBootClassPath(getBootpath(configuration));

        // stop in main
        prepareStopInMain(configuration);

        // Launch the configuration - 1 unit of work
        runner.run(runConfig, launch, monitor);
    }

    private String getTemplateSimulatorProgramArguments(ILaunchConfiguration configuration, int commandPort, int eventPort) throws CoreException {
        StringBuilder arguments = new StringBuilder();
        String templateInWorkspace = configuration.getAttribute(
                TemplateLauncherConstants.TEMPLATE_LAUNCHER_PERSISTENT_DATA_TEMPLATE, "");
        IFile templateFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(templateInWorkspace));

        // templateFile
        arguments
                .append(" --templateFile \"")
                .append(templateFile.getParent().getFile(new Path(WorkflowTemplateDTO.TEMPLATE_RUNTIME_FILE_NAME))
                        .getLocation().toPortableString()).append("\"");

        // presentationTemplateFile
        arguments.append(" --presentationTemplateFile \"")
            .append(templateFile.getLocation().toPortableString()).append("\"");
        
        // configFile
        arguments.append(" --configFile \"");
        if (configuration.getFile() != null) {
            arguments.append(configuration.getFile().getLocation().makeAbsolute().toPortableString());
        } else {
            arguments.append(configuration.getLocation().makeAbsolute().toPortableString());
        }
        arguments.append("\"");
        
        // commandPort
        arguments.append(" --commandPort ").append(commandPort);

        // eventPort
        arguments.append(" --eventPort ").append(eventPort);

        arguments.append(getProgramArguments(configuration));

        return arguments.toString();
    }

    private String[] getTemplateSimulatorClasspath(ILaunchConfiguration configuration) throws CoreException {
        List<String> classpath = new ArrayList<String>();
        Bundle bundle = getDefault().getBundle();
        List<URL> libs = Collections.list(bundle.findEntries("lib/", "*.jar", true));

        try {
            classpath.add(getBundlePath(bundle, "/"));
            classpath.add(getBundlePath(bundle, "/") + "bin");

            for (URL url : libs) {
                String lib = url.getFile();

                classpath.add(getBundlePath(bundle, lib));
            }

            classpath.addAll(Arrays.asList(getClasspath(configuration)));
        } catch (Exception e) {
            e.printStackTrace();
        }

        return classpath.toArray(new String[0]);
    }

    private String getBundlePath(Bundle bundle, String bundleEntry) throws IOException {
        URL url = FileLocator.find(bundle, new Path(bundleEntry), null);

        url = FileLocator.resolve(url);

        return url.getFile();
    }

    private static int findFreePort() {
        ServerSocket socket = null;
        try {
            socket = new ServerSocket(0);
            return socket.getLocalPort();
        } catch (IOException e) {
        } finally {
            if (socket != null) {
                try {
                    socket.close();
                } catch (IOException e) {
                }
            }
        }
        return -1;
    }
}
