package com.tandbergtv.watchpoint.studio.debugger.ui;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jdt.debug.ui.launchConfigurations.JavaLaunchTab;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.debugger.WatchpointTemplateDebuggerPlugin;
import com.tandbergtv.watchpoint.studio.debugger.launching.TemplateLauncherConstants;
import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.model.ParameterizableSimulationData;
import com.tandbergtv.watchpoint.studio.debugger.model.SimulationType;
import com.tandbergtv.watchpoint.studio.debugger.util.DebuggerUtils;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * A launch configuration tab that displays and edits project and
 * main type name launch configuration attributes.
 * <p>
 * This class may be instantiated.
 * </p>
 * @since 3.2
 * @noextend This class is not intended to be subclassed by clients.
 */
public class TemplateLaunchMainTab extends JavaLaunchTab {

	protected String templatePath = EMPTY_STRING; 
	
	public static final String EMPTY_STRING = "";
	
	// UI widgets
	private Button fSearchButton;
	protected Text fTemplate;
	protected VariableValuesTable variableValuesTable;
	protected WorkflowTemplate template;
	protected NodeSimulationConfig nodeSimulationConfig; 

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
    	Composite comp = new Composite(parent, SWT.NONE);
    	comp.setLayout(new GridLayout(1, false));
    	comp.setFont(parent.getFont());
    	GridData gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 1;
		comp.setLayoutData(gd);
		((GridLayout)comp.getLayout()).verticalSpacing = 0;
		createTemplateEditor(comp);
		createVerticalSpacer(comp, 1);
		createStartVariablesEditor(comp);
		setControl(comp);
	}
	
	/**
	 * Creates the widgets for specifying the Template to run.
	 * 
	 * @param parent the parent composite
	 */
	protected void createTemplateEditor(final Composite mainPanel) {
    	Group group = new Group(mainPanel, SWT.NONE);
    	group.setLayout(new GridLayout(2, false));
    	group.setText("Template");
    	group.setFont(mainPanel.getFont());
    	GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 1;
		group.setLayoutData(gd);
		
    	fTemplate = new Text(group, SWT.SINGLE | SWT.BORDER);
    	fTemplate.setFont(mainPanel.getFont());
    	gd = new GridData(GridData.FILL_HORIZONTAL);
    	gd.horizontalSpan = 1;
    	fTemplate.setLayoutData(gd);
		fTemplate.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setTemplate(fTemplate.getText());
				
				// Hack for adjusting toolbars/resizing  
				ScrolledComposite sc = (ScrolledComposite) mainPanel.getParent();
				sc.setMinSize(mainPanel.computeSize(SWT.DEFAULT, SWT.DEFAULT));
				
				updateLaunchConfigurationDialog();
			}
		});
		
		fSearchButton = new Button(group, SWT.PUSH);
		fSearchButton.setFont(mainPanel.getFont());
		fSearchButton.setText("Search...");
		gd = new GridData();
		fSearchButton.setLayoutData(gd);	
		gd.horizontalAlignment = GridData.FILL;
		
		fSearchButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				handleSearchButtonSelected();
			}
		});
	}
	
	protected void createStartVariablesEditor(Composite parent) {
		Group group = new Group(parent, SWT.NONE);
    	group.setLayout(new GridLayout(2, false));
    	group.setText("Start Variables");
    	group.setFont(parent.getFont());
    	GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 1;
		group.setLayoutData(gd);
		
		variableValuesTable = new VariableValuesTable(group);
		variableValuesTable.setLabelProvider(new VariableNameTableLabelProvider());
		variableValuesTable.addSimulationModifyListener(new SimulationModifyListener() {
			public void modifySimulation(Object o) {
				WPVariable var = (WPVariable) o;
				getSimulationData(nodeSimulationConfig).setVariable(var.getName(), var.getValue());
				updateLaunchConfigurationDialog();
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.AbstractLaunchConfigurationTab#getImage()
	 */
	@Override
	public Image getImage() {
		return Utility.getImageDescriptor(WatchpointTemplateDebuggerPlugin.PLUGIN_ID, "template.png").createImage();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getName()
	 */
	public String getName() {
		return "Main Template";
	}
	
	/**
	 * @see org.eclipse.debug.ui.AbstractLaunchConfigurationTab#getId()
	 * 
	 * @since 3.3
	 */
	@Override
	public String getId() {
		return "com.tandbergtv.watchpoint.studio.debugger.templateMainTab"; //$NON-NLS-1$
	}
	
	/**
	 * Show a dialog that lists all main types
	 */
	protected void handleSearchButtonSelected() {
		WorkflowTemplateDTO templateDTO = chooseTemplate();
		if (templateDTO != null) {
			fTemplate.setText(templateDTO.getPath());
		} else {
			template = null;
		}
	}	
	
	@SuppressWarnings("unchecked")
	protected void setTemplate(String newTemplatePath) {
		if (newTemplatePath != null && !newTemplatePath.equals(templatePath)) {
			// means that Template has changed, parses the template and update the variables table
			template = DebuggerUtils.parseWorkspaceTemplate(newTemplatePath);
			if (template != null) {
				List<Variable> variablesData = Collections.emptyList();
				if (template.getStartState().getTask() != null) {
					variablesData = Arrays.asList(template.getStartState().getTask().getController().getVariables()); 
				}
				variableValuesTable.setVariablesData(variablesData);
				
				nodeSimulationConfig = new NodeSimulationConfig();
				nodeSimulationConfig.setNodeElement(template.getStartState());
				nodeSimulationConfig.setTemplateNodeName(template.getStartState().getName());
				nodeSimulationConfig.setSimulationData(new ParameterizableSimulationData());
				nodeSimulationConfig.setSimulationType(SimulationType.SIMULATED_VALUES);

			} else {
				variableValuesTable.setVariablesData(Collections.EMPTY_LIST);
				nodeSimulationConfig = null;
			}
			templatePath = newTemplatePath;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.internal.debug.ui.launcher.AbstractJavaMainTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	@Override
	public void initializeFrom(ILaunchConfiguration config) {
		super.initializeFrom(config);
		try {
			fTemplate.setText(config.getAttribute(TemplateLauncherConstants.TEMPLATE_LAUNCHER_PERSISTENT_DATA_TEMPLATE, EMPTY_STRING));
			if (template != null) {
				nodeSimulationConfig = NodeSimulationConfigFactory.createFactory().getNodeForSimulation(template.getStartState(), config.getAttributes(), null);
				nodeSimulationConfig.setSimulationType(SimulationType.SIMULATED_VALUES);
				ParameterizableSimulationData simulationData = getSimulationData(nodeSimulationConfig);
				for (Variable item : variableValuesTable.getVariablesData()) {
					WPVariable startNodeVariable = (WPVariable) item;
					if (simulationData.hasVariable(startNodeVariable.getName())) {
						String value = (String) simulationData.getVariable(startNodeVariable.getName());
						startNodeVariable.setValue(value);
					}
				}
				variableValuesTable.refresh();
			}
		} catch (CoreException e) {}
	}	
	
	public ParameterizableSimulationData getSimulationData(NodeSimulationConfig nodeSimulationConfig) {
		ParameterizableSimulationData result = null;
		if (!(nodeSimulationConfig.getSimulationData() instanceof ParameterizableSimulationData)) {
			result = new ParameterizableSimulationData(nodeSimulationConfig.getSimulationData());
		} else {
			result = (ParameterizableSimulationData) nodeSimulationConfig.getSimulationData();
		}
		nodeSimulationConfig.setSimulationData(result);
		return result;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.AbstractLaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	@Override
	public boolean isValid(ILaunchConfiguration config) {
		setErrorMessage(null);
		setMessage(null);
		
		boolean valid = template != null;
		if (!valid) {
			if (!fTemplate.getText().isEmpty()) {
				setErrorMessage("Invalid Template.");
			} else { 
				setErrorMessage("Template not specified.");
			}
		} 
		
		return valid;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy config) {
		config.setAttribute(TemplateLauncherConstants.TEMPLATE_LAUNCHER_PERSISTENT_DATA_TEMPLATE, fTemplate.getText().trim());
		
		if (template != null && variableValuesTable.getVariablesData() != null) {
			ParameterizableSimulationData simulationData = getSimulationData(nodeSimulationConfig);
			for (Variable item : variableValuesTable.getVariablesData()) {
				WPVariable startNodeVariable = (WPVariable) item;
				String value = "";
				if (startNodeVariable.getValue() != null) {
					value = startNodeVariable.getValue();
				}
				simulationData.setVariable(startNodeVariable.getName(), value);
			}
			@SuppressWarnings("rawtypes")
			Map nodeAttributes = nodeSimulationConfig.getSimulationData().getAttributes();
			
			String nodeName = nodeSimulationConfig.getNodeElement().getName();
			if (nodeSimulationConfig.getSimulationType() != null) {
				config.setAttribute(TemplateLauncherConstants.getNodeSimulationTypeKey(nodeName), nodeSimulationConfig.getSimulationType().getValue());
			} else {
				config.removeAttribute(TemplateLauncherConstants.getNodeSimulationTypeKey(nodeName));
			}
			String dataPrefix = TemplateLauncherConstants.getNodeSimulationDataKey(nodeName);
			for (Object keyObj : nodeAttributes.keySet()) {
				String key = keyObj.toString();
				
				String newKey = dataPrefix + "." + key;
				config.setAttribute(newKey, (String) nodeAttributes.get(keyObj));				
			}
		}
		
		// Saves the project name for Java launcher compatibility.
		String project = EMPTY_STRING;
		try {
			IFile templateFile = getWorkspaceRoot().getFile(new Path(fTemplate.getText()));
			if (templateFile.getProject() != null) {
				project = templateFile.getProject().getName();
				config.setMappedResources(new IResource[] {templateFile});
			}
		} catch (Exception e) {}
		config.setAttribute(IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, project);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#setDefaults(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void setDefaults(ILaunchConfigurationWorkingCopy config) {
		config.setAttribute(TemplateLauncherConstants.TEMPLATE_LAUNCHER_PERSISTENT_DATA_TEMPLATE, EMPTY_STRING);
	}
	
	/**
	 * Chooses a template
	 * 
	 * @return the template selected
	 */
	private WorkflowTemplateDTO chooseTemplate() {
		ILabelProvider labelProvider = new LabelProvider() {
			@Override
			public String getText(Object element) {
				WorkflowTemplateDTO template = (WorkflowTemplateDTO) element;
				return template.getName() + " - " + template.getPath();
			}
			@Override
			public Image getImage(Object element) {
				return Utility.getImageDescriptor("template.png").createImage();
			}
			
		};
		ElementListSelectionDialog dialog= new ElementListSelectionDialog(getShell(), labelProvider);
		dialog.setTitle("Template Selection"); 
		dialog.setMessage("Select a Template to run.");
		
		IWorkflowTemplateService templateService = ServiceFactory.createFactory().createWorkflowTemplateService();
		List<WorkflowTemplateDTO> templates = templateService.getTemplateList();
		
		dialog.setElements(templates.toArray());
		String template = fTemplate.getText();
		if (template != null) {
			dialog.setInitialSelections(new Object[] { template });
		}
		if (dialog.open() == Window.OK) {			
			return (WorkflowTemplateDTO) dialog.getFirstResult();
		}		
		return null;		
	}
	
	/**
	 * Convenience method to get the workspace root.
	 */
	protected IWorkspaceRoot getWorkspaceRoot() {
		return ResourcesPlugin.getWorkspace().getRoot();
	}
	
	public String getTemplateLocation() {
		return fTemplate.getText();
	}
	
	public WorkflowTemplate getTemplate() {
		return template;
	}
	
}
