/*
 * Created on Jun 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess;

import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;

/**
 * The Data Access Interface for the Resource Type entity.
 * 
 * @author Vijay Silva
 */
public interface ResourceTypeDAI extends DataAccessInterface<ResourceType, Long>
{
	/**
	 * Finds all Resource Types but only returns a subset of the Properties of the Resource Type.
	 * The returned properties are the properties required when viewing the basic resource type.
	 * 
	 * The list of Viewable properties are: Id, name, systemId, description, createDate, outOfTheBox
	 * flag, version and dirty flag.
	 * 
	 * @return The list of all Resource Types (with limited properties).
	 */
	List<ResourceType> findAllViewable();

	/**
	 * Finds a Resource Type using the System Id
	 * 
	 * @param systemId
	 *            The System Id of the Resource Type
	 * 
	 * @return The Resource Type with matching System Id, or null if none exists.
	 */
	ResourceType findBySystemId(String systemId);

	/**
	 * Finds a Resource Type by Name
	 * 
	 * @param name
	 *            The name of the Resource Type
	 * 
	 * @return The Resource Type with matching name, or null if none exists.
	 */
	ResourceType findByName(String name);
	
	/**
	 * Get the Resource Type for a specified project.
	 * 
	 * @param projectName
	 * @return
	 */
	List<ResourceType> findByProjectName(String projectName);
	
	/**
	 * Get the Resource Type by its Path.
	 * 
	 * @param name
	 *            The Resource Type Path
	 * 
	 * @return Resource Type with matching path.
	 */
	ResourceType findByPath(String path);

	/**
	 * Gets all the messages for a resource type specified by its id.
	 * 
	 * @param resourceTypeId
	 *            the id of the resource type
	 * @return all the messages for the resource type
	 */
	List<Message> getMessagesById(long resourceTypeId);

	/**
	 * Gets all the messages corresponding to the given uids
	 * 
	 * @param uids
	 *            the set of uids whose messages are to be retrieved
	 * @return messages corresponding to the given uids
	 */
	List<Message> getMessagesByUid(Set<String> uids);

	/**
	 * Checks if the the resource type specified by id is out of the box.
	 * 
	 * @param resourceTypeId
	 *            The Resource Type Id
	 * @return true, if the Resource Type is Out-of-the-box; false, otherwise
	 */
	boolean isOutOfTheBox(long resourceTypeId);

	/**
	 * Gets the count of Resource Types with a matching System Id
	 * 
	 * @param systemId
	 *            The System Id
	 * 
	 * @return The Count of Resource Types with matching system Id (should be either 0 or 1).
	 */
	int getCountBySystemId(String systemId);

	/**
	 * Gets the count of Resource Types with a matching Name
	 * 
	 * @param name
	 *            The name of the Resource Type
	 * 
	 * @return The Count of Resource Types with matching name (should be either 0 or 1).
	 */
	int getCountByName(String name);
}
