/*
 * Created on Jun 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess.jpa;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.NoResultException;
import javax.persistence.Query;

import org.apache.log4j.Logger;

import com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.ResourceTypeParameter;

/**
 * JPA Data Access Object for the ResourceType entity
 * 
 * @author Vijay Silva
 */
public class ResourceTypePDAO extends PersistenceDAO<ResourceType, Long> implements ResourceTypeDAI
{
	private static final Logger logger = Logger.getLogger(ResourceType.class);

	/**
	 * Constructor
	 * 
	 * @param persistenceContext
	 *            The Persistence Context
	 */
	public ResourceTypePDAO(JPAPersistenceContext persistenceContext)
	{
		super(ResourceType.class, persistenceContext);
	}

	/**
	 * Need to remove all Parameters and Messages that are not used anymore.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.jpa.PersistenceDAO#update(com.tandbergtv.watchpoint.studio.dto.IPersistable)
	 */
	@Override
	public ResourceType update(ResourceType entity)
	{
		// Delete all removed parameters, update the parameter ids
		List<ResourceTypeParameter> currentParameters = this.getParametersById(entity.getId());
		Map<ParameterMapKey, ResourceTypeParameter> parameterMap = this.buildParameterMap(entity);
		ParameterMapKey mapKey = new ParameterMapKey(null);
		for (ResourceTypeParameter currentParameter : currentParameters)
		{
			mapKey.parameter = currentParameter;
			ResourceTypeParameter parameter = parameterMap.get(mapKey);
			if (parameter != null)
				parameter.setId(currentParameter.getId());
			else
				this.getEntityManager().remove(currentParameter);
		}

		return super.update(entity);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.jpa.PersistenceDAO#refresh(com.tandbergtv.watchpoint.studio.dto.IPersistable)
	 */
	@Override
	protected void refresh(ResourceType entity)
	{
		super.refresh(entity);

		/* Cascade the refreshing for the Messages and Parameters */
		Set<ResourceTypeParameter> parameters = entity.getParameters();
		if (parameters != null)
		{
			for (ResourceTypeParameter parameter : parameters)
				this.getEntityManager().refresh(parameter);
		}

		Set<Message> messages = entity.getMessages();
		if (messages != null)
		{
			for (Message message : messages)
				this.getEntityManager().refresh(message);
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#findAllViewable()
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceType> findAllViewable()
	{
		String queryName = "ResourceType.Viewable";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#findBySystemId(java.lang.String)
	 */
	public ResourceType findBySystemId(String systemId)
	{
		String queryName = "ResourceType.BySystemId";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("systemId", systemId);

		ResourceType resourceType = null;
		try
		{
			resourceType = (ResourceType) query.getSingleResult();
		}
		catch (NoResultException nre)
		{
			String msg = "Cannot find a Resource Type with System Id: " + systemId
					+ ", return null.";
			logger.debug(msg, nre);
			resourceType = null;
		}

		return resourceType;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#findByName(java.lang.String)
	 */
	public ResourceType findByName(String name)
	{
		String queryName = "ResourceType.ByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);

		ResourceType resourceType = null;
		try
		{
			resourceType = (ResourceType) query.getSingleResult();
		}
		catch (NoResultException nre)
		{
			String msg = "Cannot find a Resource Type with Name: " + name + ", return null.";
			logger.debug(msg, nre);
			resourceType = null;
		}

		return resourceType;
	}
	
	public ResourceType findByPath(String path) {
		String queryName = "ResourceType.ByPath";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("path", path);

		ResourceType result = null;
		try {
			result = (ResourceType) query.getSingleResult();
		} catch (javax.persistence.NoResultException e) {
			// No Resource Type found - return null
		}
		
		return result;
	}
	
	@SuppressWarnings("unchecked")
	public List<ResourceType> findByProjectName(String projectName)
	{
		String queryName = "ResourceType.ByProjectName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("projectName", projectName);

		List<ResourceType> result = null;
		try {
			result = (List<ResourceType>) query.getResultList();
		} catch (NoResultException e) {
			// No Resource Type found - return null
		}
		
		return result;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#getCountByName(java.lang.String)
	 */
	public int getCountByName(String name)
	{
		String queryName = "ResourceType.CountByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#getCountBySystemId(java.lang.String)
	 */
	public int getCountBySystemId(String systemId)
	{
		String queryName = "ResourceType.CountBySystemId";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("systemId", systemId);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#getMessagesById(long)
	 */
	@SuppressWarnings("unchecked")
	public List<Message> getMessagesById(long resourceTypeId)
	{
		String queryName = "ResourceType.MessagesById";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("id", resourceTypeId);
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#getMessagesByUid(java.util.Set)
	 */
	@SuppressWarnings("unchecked")
	public List<Message> getMessagesByUid(Set<String> uids)
	{
		if (uids == null || uids.size() == 0)
			return new ArrayList<Message>();

		// Building the Query
		StringBuilder queryBuf = new StringBuilder();
		queryBuf.append("SELECT NEW com.tandbergtv.watchpoint.studio.dto.Message(msg.id, msg.name) ");
		queryBuf.append("FROM com.tandbergtv.watchpoint.studio.dto.Message msg ");
		queryBuf.append("WHERE msg.uid IN ( ");

		boolean first = true;
		for (String uid : uids)
		{
			if (uid == null || uid.trim().length() == 0)
				continue;

			if (!first)
				queryBuf.append(", ");
			else
				first = false;

			queryBuf.append("'");
			queryBuf.append(uid);
			queryBuf.append("'");
		}

		queryBuf.append(" ) ");

		// Create and execute the query
		Query query = this.getEntityManager().createQuery(queryBuf.toString());
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI#isOutOfTheBox(long)
	 */
	public boolean isOutOfTheBox(long resourceTypeID)
	{
		String queryName = "ResourceType.OutOfTheBoxById";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("id", resourceTypeID);
		return (Boolean) query.getSingleResult();
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Builds a Map of all the Parameters in the Resource Type
	 */
	private Map<ParameterMapKey, ResourceTypeParameter> buildParameterMap(ResourceType resourceType)
	{
		Map<ParameterMapKey, ResourceTypeParameter> parameterMap = new HashMap<ParameterMapKey, ResourceTypeParameter>();
		Set<ResourceTypeParameter> parameters = resourceType.getAllParameters();
		for (ResourceTypeParameter parameter : parameters)
		{
			parameterMap.put(new ParameterMapKey(parameter), parameter);
		}

		return parameterMap;
	}

	/*
	 * Get all the Parameters for a Resource Type
	 */
	@SuppressWarnings("unchecked")
	private List<ResourceTypeParameter> getParametersById(long resourceTypeId)
	{
		String queryName = "ResourceType.ParametersById";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("id", resourceTypeId);
		return query.getResultList();
	}

	// ========================================================================
	// ===================== INTERNAL CLASS
	// ========================================================================

	/*
	 * Internal class that is used as a Map Key for a Resource Type Parameter, defining that to such
	 * keys are equal if the parameter name and type are the same.
	 */
	private class ParameterMapKey
	{
		ResourceTypeParameter parameter;

		/*
		 * Constructor
		 */
		ParameterMapKey(ResourceTypeParameter parameter)
		{
			this.parameter = parameter;
		}

		/*
		 * @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode()
		{
			if (this.parameter == null || this.parameter.getName() == null)
				return super.hashCode();

			return (this.parameter.getName().hashCode() * 4)
					- (this.parameter.getType().hashCode() * 3) + 373;
		}

		/*
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj)
		{
			if (!(obj instanceof ParameterMapKey))
				return false;

			ResourceTypeParameter paramObj = ((ParameterMapKey) obj).parameter;

			if (this.parameter == null && paramObj == null)
				return super.equals(obj);

			String name = this.parameter.getName();
			return ((this.parameter.getType() == paramObj.getType()) && (name != null) && name.equals(paramObj.getName()));
		}
	}
}
