/*
 * Created on Mar 16, 2007 (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.jpf;

import static com.tandbergtv.watchpoint.studio.ui.util.Utility.PLUGIN_XML;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.java.plugin.ObjectFactory;
import org.java.plugin.PathResolver;
import org.java.plugin.PluginManager;
import org.java.plugin.PluginManager.PluginLocation;
import org.java.plugin.boot.DefaultPluginsCollector;
import org.java.plugin.boot.PluginsCollector;
import org.java.plugin.registry.Extension;
import org.java.plugin.registry.Extension.Parameter;
import org.java.plugin.registry.ExtensionPoint;
import org.java.plugin.registry.IntegrityCheckReport;
import org.java.plugin.registry.PluginDescriptor;
import org.java.plugin.registry.PluginRegistry;
import org.java.plugin.util.ExtendedProperties;

import com.tandbergtv.watchpoint.studio.external.jpf.model.FailureDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ResourceGroupDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ResourceTypeDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.validation.ResourceGroupPluginValidator;
import com.tandbergtv.watchpoint.studio.external.jpf.validation.ResourceTypeDescriptorValidator;
import com.tandbergtv.watchpoint.studio.external.jpf.validation.ResourceTypeExtensionValidator;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Manager class that maintains the registry of Resource Type Plugins
 * 
 * @author Vijay Silva
 */
public class JPFPluginManagerImpl implements JPFPluginManager {

	// Flag to keep track if the Manager is initialized
	private boolean isInitialized = false;

	// The JPF Plugin Manager
	private PluginManager pluginManager;

	// A cache of the Licensed ResourceTypeDescriptors mapped to System Ids.
	private Map<String, ResourceTypeDescriptor> resTypeDescriptors = new Hashtable<String, ResourceTypeDescriptor>();

	// A cache of the ResourceGroupDescriptors mapped to Resource Group's Name.
	private Map<String, ResourceGroupDescriptor> resGroupDescriptors = new Hashtable<String, ResourceGroupDescriptor>();

	private static final Logger logger = Logger.getLogger(JPFPluginManagerImpl.class);

	private static final String WATCHPOINT_PLUGIN_ID = "com.tandbergtv.workflow";

	private static final String RESOURCE_TYPE_EXTENSION_POINT_ID = "resourcetype";

	private static final String RESOURCE_GROUP_EXTENSION_POINT_ID = "resourcegroup";
	
	private static final String REPOSITORIES = "org.java.plugin.boot.pluginsRepositories";
	
	// =====================================================
	// ============== CLASS CONSTRUCTOR AND INITIALIZATION
	// =====================================================

	/*
	 * Class Constructor
	 */
	public JPFPluginManagerImpl(String path) {
		ExtendedProperties properties = new ExtendedProperties();
		
		try {
			String pluginxml = writePluginDescriptor();
			
			/* Include the host plugin's definition in the list of paths to be traversed */
			properties.put(REPOSITORIES, pluginxml + "," + path);
			this.pluginManager = ObjectFactory.newInstance(properties).createManager();
			
			PluginsCollector collector = new DefaultPluginsCollector();
			
			collector.configure(properties);
			
			Collection<PluginLocation> pluginLocations = collector.collectPluginLocations();

			pluginManager.publishPlugins(pluginLocations.toArray(new PluginLocation[pluginLocations.size()]));
			pluginManager.getRegistry().checkIntegrity(pluginManager.getPathResolver(), true);
		} catch (Exception e) {
			throw new RuntimeException("Failed to load plugins from " + path, e);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void initialize() {
		if (this.isInitialized)
			return;

		logger.debug("Initializing the Plugin Manager...");

		loadResourceTypeDescriptors();
		loadResourceGroupDescriptors();

		this.isInitialized = true;
		logger.debug("Finished initializing the Plugin Manager...");
	}

	/**
	 * {@inheritDoc}
	 */
	public void close() {
		this.resGroupDescriptors.clear();
		this.resTypeDescriptors.clear();
		this.pluginManager.shutdown();
		this.isInitialized = false;
	}

	/**
	 * Writes out the workflow plugin.xml to the plugin's state location. Returns the state location.
	 * 
	 * @throws Exception 
	 */
	private String writePluginDescriptor() throws Exception {
		File file = WatchPointStudioPlugin.getDefault().getStateLocation().toFile();
		File pluginxml = new File(file, Utility.PLUGIN_XML);

		if (!pluginxml.exists()) {
			FileOutputStream os = new FileOutputStream(pluginxml);
			BufferedInputStream bis = null;
			BufferedOutputStream bos = null;
			try {
				InputStream is = this.getClass().getResourceAsStream("/" + PLUGIN_XML);
				
				bis = new BufferedInputStream(is);
				bos = new BufferedOutputStream(os);
				
				while (true) {
					int datum = bis.read();
					if (datum == -1)
						break;
					bos.write(datum);
				}
				bos.flush();
			} finally {
				if (bis != null)
					bis.close();
				if (os != null)
					bos.close();
					os.close();
			}
		}
		
		return file.getAbsolutePath();
	}
	
	/*
	 * Load all the ResourceTypeDescriptors
	 */
	private void loadResourceTypeDescriptors() {
		logger.debug("Loading the Resource Type Descriptors...");

		PluginRegistry registry = this.pluginManager.getRegistry();
		PathResolver pathResolver = this.pluginManager.getPathResolver();
		IntegrityCheckReport report = registry.checkIntegrity(pathResolver, true);
		ExtensionPoint rtPoint = registry.getExtensionPoint(WATCHPOINT_PLUGIN_ID, RESOURCE_TYPE_EXTENSION_POINT_ID);

		List<String> failedPlugins = new ArrayList<String>();
		for (Extension extension : rtPoint.getAvailableExtensions()) {
			// Check if the extension is valid
			FailureDescriptor failedDescriptor = validateResourceTypeExtension(extension, report);

			// The extension is valid
			if (failedDescriptor == null) {
				PluginDescriptor pluginDesc = extension.getDeclaringPluginDescriptor();
				ClassLoader loader = this.pluginManager.getPluginClassLoader(pluginDesc);

				ResourceTypeDescriptor descriptor = 
					ExtensionParser.parseResourceTypeExtension(extension, loader);
				
				// Validate the descriptor object properties
				failedDescriptor = validateResourceTypeDescriptor(descriptor);

				logger.debug("Plugin " + descriptor.getSystemId());
				
				if (failedDescriptor == null)
					failedDescriptor = validateUniqueSystemId(descriptor);

				// Descriptor is valid
				if (failedDescriptor == null)
					resTypeDescriptors.put(descriptor.getSystemId(), descriptor);
			}

			if (failedDescriptor != null) { // The Plugin Extension is invalid
				failedPlugins.add(failedDescriptor.getPluginId());
			}
		}

		if (failedPlugins.size() > 0) {
			registry.unregister(failedPlugins.toArray(new String[0]));
			logger.debug("Found : " + failedPlugins.size() + " invalid plugins.");
			
			for (String failed : failedPlugins) {
				logger.info("Failed plugin " + failed);
			}
		}

		logger.debug("Finished loading: " + this.resTypeDescriptors.size() + " plugins");
	}

	private void loadResourceGroupDescriptors() {
		logger.debug("Loading Resource Group Descriptors...");

		PluginRegistry registry = this.pluginManager.getRegistry();
		PathResolver pathResolver = this.pluginManager.getPathResolver();
		
		IntegrityCheckReport report = registry.checkIntegrity(pathResolver, true);
		ExtensionPoint rtPoint = registry.getExtensionPoint(WATCHPOINT_PLUGIN_ID,
				RESOURCE_GROUP_EXTENSION_POINT_ID);

		List<String> failedPluginIDs = new ArrayList<String>();

		for (Extension extension : rtPoint.getAvailableExtensions()) {
			logger.debug("Extension ID: " + extension.getId().toString());

			// Validate the extension
			FailureDescriptor failedDescriptor = 
				ResourceGroupPluginValidator.validateExtension(extension, report);

			if (failedDescriptor == null) {
				PluginDescriptor pluginDesc = extension.getDeclaringPluginDescriptor();
				ClassLoader loader = this.pluginManager.getPluginClassLoader(pluginDesc);

				// Parse the Extension into a descriptor
				ResourceGroupDescriptor rgDescriptor = 
					ResourceGroupExtensionParser.parse(extension, loader);
				
				URL url = pathResolver.getRegisteredContext(extension.getId());
				
				String path = url.getPath();
				try {
					 // Decodes the URL to avoid escaping entities 
					 // on special characters like: My%20ResourceGroup
					 path = URLDecoder.decode(path, "UTF-8");
				} catch (UnsupportedEncodingException e) {
					logger.error("Error decoding URL: " + path);
				}
				rgDescriptor.setPath(path);

				logger.debug("Parsed Res Grp Descriptor: " + rgDescriptor.toString());

				// Validate the descriptor object
				boolean exists = rgDescriptor.getName() != null && this.resGroupDescriptors.containsKey(rgDescriptor.getName());
				failedDescriptor = 
					ResourceGroupPluginValidator.validateDescriptor(rgDescriptor, exists);

				if (failedDescriptor == null)
					resGroupDescriptors.put(rgDescriptor.getName(), rgDescriptor);
				else
					logger.debug("Resource group description validation failed");
			} else
				logger.debug("Resource group extension validation failed");

			// if the extension or the parsed descriptor failed validation,
			// add the plugin id to list of failed plugins ids
			if (failedDescriptor != null) {
				logger.debug("Failure Descriptor: " + failedDescriptor.toString());
				failedPluginIDs.add(failedDescriptor.getPluginId());
			}
		}

		// if there are any failed plugins, unregister them fom registry
		if (failedPluginIDs.size() > 0) {
			registry.unregister(failedPluginIDs.toArray(new String[0]));
			logger.debug("Found " + failedPluginIDs.size() + " invalid plugins.");
		}

		logger.debug("Finished loading " + resGroupDescriptors.size() + " Resource Group Descriptors...");
	}

	// =====================================================
	// ============== VALIDATION
	// =====================================================

	private FailureDescriptor validateResourceTypeExtension(Extension extension,
			IntegrityCheckReport report) {
		List<String> messages = ResourceTypeExtensionValidator.validate(extension, report);
		return prepareFailureDescriptor(extension, messages);
	}

	/*
	 * Generate the FailureDescriptor that provides information about why the Resource Type
	 * Extension has failed validation.
	 */
	private FailureDescriptor prepareFailureDescriptor(Extension extension, List<String> messages) {
		FailureDescriptor result = null;
		if (messages.size() > 0) {
			result = new FailureDescriptor();
			result.setPluginId(extension.getDeclaringPluginDescriptor().getId());
			result.setExtensionId(extension.getId());

			// Get the Id
			Parameter param = extension.getParameter(PluginParameters.SYSTEM_ID);
			if (param != null)
				result.setId(param.valueAsString());

			// Get the Name
			param = extension.getParameter(PluginParameters.NAME);
			if (param != null)
				result.setName(param.valueAsString());

			result.setSummaryMessage("The Plugin Extension has failed basic validation.");
			result.setErrorMessages(messages);
		}

		return result;
	}

	/*
	 * Method to validate if the Descriptor contains valid data, and all classes specified can be
	 * correctly loaded.
	 */
	private FailureDescriptor validateResourceTypeDescriptor(ResourceTypeDescriptor descriptor) {
		FailureDescriptor result = null;
		List<String> messages = ResourceTypeDescriptorValidator.validate(descriptor);

		if (messages.size() > 0) {
			result = new FailureDescriptor();
			result.setPluginId(descriptor.getPluginId());
			result.setExtensionId(descriptor.getExtensionId());
			result.setId(descriptor.getSystemId());
			result.setCreateDate(descriptor.getCreateDate());
			result.setName(descriptor.getName());
			result.setSummaryMessage("Plugin Extension was successfully loaded, but failed validation.");
			result.setErrorMessages(messages);
		}

		return result;
	}
	
	private FailureDescriptor validateUniqueSystemId(ResourceTypeDescriptor descriptor) {
		FailureDescriptor result = null;
		List<String> messages = new ArrayList<String>();
		String id = descriptor.getSystemId();
		
		if (this.resTypeDescriptors.containsKey(id)) {
			ResourceTypeDescriptor other = this.resTypeDescriptors.get(id);

			/* Reject the plugin that is older */
			if (descriptor.getVersion() > other.getVersion()) {
				messages.add("The System Id specified is already used by another Resource Type.");
				this.resTypeDescriptors.remove(id);
			}
		}
		
		if (messages.size() > 0) {
			result = new FailureDescriptor();
			result.setPluginId(descriptor.getPluginId());
			result.setExtensionId(descriptor.getExtensionId());
			result.setId(descriptor.getSystemId());
			result.setCreateDate(descriptor.getCreateDate());
			result.setName(descriptor.getName());
			result.setSummaryMessage("Plugin Extension was successfully loaded, but failed validation.");
			result.setErrorMessages(messages);
		}

		return result;
	}

	// ====================================================
	// =============== RESOURCE TYPE DESCRIPTOR MANAGMENT
	// ====================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.external.jpf.workflow.pluginmanager.JPFPluginManager#getResourceTypeDescriptor(String)
	 */
	public ResourceTypeDescriptor getResourceTypeDescriptor(String systemId) {
		ResourceTypeDescriptor result = null;

		ResourceTypeDescriptor descriptor = this.resTypeDescriptors.get(systemId);
		if (descriptor != null)
			result = (ResourceTypeDescriptor) descriptor.clone();

		return result;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.external.jpf.workflow.pluginmanager.JPFPluginManager#getResourceTypeDescriptors()
	 */
	public List<ResourceTypeDescriptor> getResourceTypeDescriptors() {
		List<ResourceTypeDescriptor> result = new ArrayList<ResourceTypeDescriptor>();

		for (ResourceTypeDescriptor descriptor : this.resTypeDescriptors.values()) {
			ResourceTypeDescriptor clone = (ResourceTypeDescriptor) descriptor.clone();
			result.add(clone);
		}

		return result;
	}

	// ====================================================
	// =============== RESOURCE GROUP DESCRIPTOR MANAGMENT
	// ====================================================

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tandbergtv.workflow.pluginmanager.PluginManagement#getResourceGroupDescriptors()
	 */
	public List<ResourceGroupDescriptor> getResourceGroupDescriptors() {
		List<ResourceGroupDescriptor> result = new ArrayList<ResourceGroupDescriptor>();

		for (ResourceGroupDescriptor descriptor : resGroupDescriptors.values()) {
			result.add((ResourceGroupDescriptor) descriptor.clone());
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tandbergtv.workflow.pluginmanager.PluginManagement#getResourceGroupDescriptor(java.lang.String)
	 */
	public ResourceGroupDescriptor getResourceGroupDescriptor(String name) {
		if (resGroupDescriptors != null && resGroupDescriptors.containsKey(name))
			return (ResourceGroupDescriptor) resGroupDescriptors.get(name).clone();

		return null;
	}
}
