/*
 * Created on Sep 4, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.wpexport.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Utility Class that provides methods for building the basic JPF (Java Plug-in Framework) elements
 * required in a plug-in file.
 * 
 * @author Vijay Silva
 */
public final class JPFExportUtil
{
	private static final String LIBRARY_ID_PREFIX = "_library_";

	private static final int VALUE_ATTRIBUTE_SIZE_LIMIT = 300;

	// Cannot Instantiate
	private JPFExportUtil()
	{
	}

	/**
	 * Creates the basic JPF Plug-in XML Document. The document contains the doctype element and the
	 * root plugin element.
	 * 
	 * @param pluginId
	 *            The Plug-Id to use for the new Plug-in.
	 * @param version
	 *            The Version of the new Plug-in.
	 * 
	 * @return The plug-in XML document
	 * 
	 * @throws ParserConfigurationException
	 *             Failure to create the Document Builder
	 */
	public static Document createPluginDocument(String pluginId, String version)
			throws ParserConfigurationException
	{
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);

		DocumentBuilder documentBuilder = factory.newDocumentBuilder();
		DOMImplementation domImpl = documentBuilder.getDOMImplementation();
		DocumentType docType = domImpl.createDocumentType(JPFConstants.ELEM_PLUGIN,
				JPFConstants.DOCTYPE_PUBLIC_ID, JPFConstants.DOCTYPE_SYSTEM_ID);
		Document pluginDocument = domImpl.createDocument("", JPFConstants.ELEM_PLUGIN, docType);

		/* Plug-in Element must have a 'id' and 'version' attribute defined */
		Element pluginElement = pluginDocument.getDocumentElement();
		pluginElement.setAttribute(JPFConstants.ATTR_PLUGIN_ID, pluginId);
		pluginElement.setAttribute(JPFConstants.ATTR_PLUGIN_VERSION, version);

		return pluginDocument;
	}

	/**
	 * Add a Plug-in to the list of plug-ins the current plug-in depends on in the 'requires'
	 * section of the document.
	 * 
	 * @param pluginDocument
	 *            The Plug-in Document for the current plug-in.
	 * @param pluginId
	 *            The Id of the plug-in which the current plug-in depends on.
	 */
	public static final void addPluginDependency(Document pluginDocument, String pluginId)
	{
		if (pluginId == null || pluginId.trim().length() == 0)
		{
			String msg = "The Plugin ID specified as a dependent cannot be null or blank.";
			throw new IllegalArgumentException(msg);
		}

		/* Create or find the Requires Element node */
		Element pluginElement = pluginDocument.getDocumentElement();
		List<String> names = Arrays.asList(JPFConstants.ELEM_RUNTIME,
				JPFConstants.ELEM_EXTENSION_POINT, JPFConstants.ELEM_EXTENSION);
		Element element = findOrCreateElement(JPFConstants.ELEM_REQUIRES, pluginElement, names);

		/* Check if the Import is already present */
		NodeList nodes = element.getChildNodes();
		if (nodes != null)
		{
			for (int i = 0; i < nodes.getLength(); i++)
			{
				Node childNode = nodes.item(i);
				if (childNode.getNodeType() == Node.ELEMENT_NODE)
				{ // Check if the import is already present and skip
					Element childElement = (Element) childNode;
					String name = childNode.getNodeName();
					String value = childElement.getAttribute(JPFConstants.ATTR_IMPORT_PLUGIN_ID);
					if (JPFConstants.ELEM_IMPORT.equals(name) && pluginId.equals(value))
						return;
				}
			}
		}

		/* Add the new Import */
		Element importElement = pluginDocument.createElement(JPFConstants.ELEM_IMPORT);
		importElement.setAttribute(JPFConstants.ATTR_IMPORT_PLUGIN_ID, pluginId);
		element.appendChild(importElement);
	}

	/**
	 * Add the list of runtime libraries as part of the plugin's runtime environment.
	 * 
	 * @param pluginDocument
	 *            The Document containing the plug-in XML
	 * @param codeLibraries
	 *            The list of plug-in relative code library paths
	 * @param resourceLibraries
	 *            The list of plug-in relative resource library paths
	 */
	public static final void addRuntimeLibraries(Document pluginDocument,
			List<String> codeLibraries, List<String> resourceLibraries)
	{
		/* Ensure that the list of library paths are not null */
		if (codeLibraries == null)
			codeLibraries = new ArrayList<String>();

		if (resourceLibraries == null)
			resourceLibraries = new ArrayList<String>();

		/* Skip if there are no libraries being added */
		int newLibraryCount = codeLibraries.size() + resourceLibraries.size();
		if (newLibraryCount == 0)
			return;

		/* Create or find the Runtime Element node */
		Element pluginElement = pluginDocument.getDocumentElement();
		List<String> names = Arrays.asList(JPFConstants.ELEM_EXTENSION_POINT,
				JPFConstants.ELEM_EXTENSION);
		Element element = findOrCreateElement(JPFConstants.ELEM_RUNTIME, pluginElement, names);

		/* Collect all the current libraries */
		Map<String, String> currentLibraries = new HashMap<String, String>();
		NodeList nodes = element.getChildNodes();
		if (nodes != null)
		{
			for (int i = 0; i < nodes.getLength(); i++)
			{
				Node childNode = nodes.item(i);
				if (childNode.getNodeType() == Node.ELEMENT_NODE
						&& JPFConstants.ELEM_LIBRARY.equals(childNode.getNodeName()))
				{
					Element childElement = (Element) childNode;
					String id = childElement.getAttribute(JPFConstants.ATTR_LIBRARY_ID);
					String path = childElement.getAttribute(JPFConstants.ATTR_LIBRARY_PATH);
					currentLibraries.put(id, path);
				}
			}
		}

		/* Add all new libraries */
		addLibraries(element, codeLibraries, JPFConstants.VALUE_LIBRARY_TYPE_CODE, currentLibraries);
		addLibraries(element, resourceLibraries, JPFConstants.VALUE_LIBRARY_TYPE_RESOURCE,
				currentLibraries);
	}

	/**
	 * Adds and returns a new Extension Element to the Plug-In document given the new extension Id,
	 * and the extension point and plug-in id of the extension point being extended.
	 * 
	 * @param pluginDocument
	 *            The Plug-in Document
	 * @param extensionId
	 *            The new extension's Id
	 * @param pluginId
	 *            The Id of the plug-in containing the extension point being extended
	 * @param extensionPointId
	 *            The Id of the extension point being extended
	 * @return The new extension element created and added to the document
	 */
	public static Element addExtension(Document pluginDocument, String extensionId,
			String pluginId, String extensionPointId)
	{
		Element extensionElement = pluginDocument.createElement(JPFConstants.ELEM_EXTENSION);
		extensionElement.setAttribute(JPFConstants.ATTR_EXTENSION_ID, extensionId);
		extensionElement.setAttribute(JPFConstants.ATTR_EXTENSION_PLUGIN_ID, pluginId);
		extensionElement.setAttribute(JPFConstants.ATTR_EXTENSION_PLUGIN_POINT_ID, extensionPointId);

		pluginDocument.getDocumentElement().appendChild(extensionElement);

		return extensionElement;
	}

	/**
	 * Add a new Parameter Element to the specified parent element and assign set the parameter Id
	 * with the given key. No value is set.
	 * 
	 * @param parent
	 *            The Parent Element
	 * @param key
	 *            The Parameter Id
	 * 
	 * @return The parameter element created and added to the parent element.
	 */
	public static Element addParameter(Element parent, String key)
	{
		return addParameter(parent, key, null);
	}

	/**
	 * Add a new Parameter Element to the specified parent element and assign set the parameter Id
	 * and value with the given key and value. If the value is null, does not set the value.
	 * 
	 * @param parent
	 *            The Parent Element to which the new parameter element must be added.
	 * @param key
	 *            The Parameter Id for the new parameter element
	 * @param value
	 *            The value for the new parameter element
	 * 
	 * @return The new Parameter Element created and added to the parent element.
	 */
	public static Element addParameter(Element parent, String key, String value)
	{
		Document document = parent.getOwnerDocument();
		Element parameterElement = document.createElement(JPFConstants.ELEM_PARAMETER);

		parameterElement.setAttribute(JPFConstants.ATTR_PARAMETER_ID, key);
		if (requiresValueElement(value))
		{
			Element valueElement = document.createElement(JPFConstants.ELEM_VALUE);
			/* fix the new lines in the value */
			value = value.replace(System.getProperty("line.separator"), "\n");
			valueElement.setTextContent(value);
			parameterElement.appendChild(valueElement);
		}
		else if (value != null)
		{
			parameterElement.setAttribute(JPFConstants.ATTR_PARAMETER_VALUE, value);
		}

		parent.appendChild(parameterElement);
		return parameterElement;
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/* Method to check if the value cannot be placed as an attribute and requires an element */
	private static boolean requiresValueElement(String value)
	{
		if (value == null)
			return false;

		return ((value.length() > VALUE_ATTRIBUTE_SIZE_LIMIT)
				|| value.contains(System.getProperty("line.separator")) || value.contains("\n"));
	}

	/*
	 * Inserts the element in the parent element ensuring that it is inserted before any element
	 * who's name is present in the elementsAfter list.
	 */
	private static Element findOrCreateElement(String elementName, Element parent,
			List<String> elementsAfter)
	{
		NodeList nodes = parent.getElementsByTagName(elementName);
		Element element = null;
		if (nodes == null || nodes.getLength() == 0)
		{
			element = parent.getOwnerDocument().createElement(elementName);
			Node refChild = parent.getFirstChild();
			NodeList childNodes = parent.getChildNodes();
			if (childNodes != null)
			{
				if (elementsAfter == null)
					elementsAfter = new ArrayList<String>();

				for (int i = 0; i < childNodes.getLength(); i++)
				{
					Node childNode = childNodes.item(i);
					if (childNode.getNodeType() == Node.ELEMENT_NODE)
					{
						if (!elementsAfter.contains(childNode.getNodeName()))
							refChild = childNode.getNextSibling();
						else
							break;
					}
				}
			}

			parent.insertBefore(element, refChild);
		}
		else
		{
			element = (Element) nodes.item(0);
		}

		return element;
	}

	/*
	 * Adds the libraries to the Runtime Element, generating unique library IDs. The current
	 * libraries are a map of existing library IDs to library paths.
	 */
	private static void addLibraries(Element runtimeElement, List<String> libraryPaths,
			String libraryType, Map<String, String> currentLibraries)
	{
		int libraryCounter = 1;

		for (String libraryPath : libraryPaths)
		{
			Document document = runtimeElement.getOwnerDocument();
			Element libraryElement = document.createElement(JPFConstants.ELEM_LIBRARY);
			String initialLibraryId = libraryType + LIBRARY_ID_PREFIX;

			String libraryId = null;
			String suffix = null;
			do
			{
				suffix = Integer.toString(libraryCounter++);
				if (suffix.length() == 1)
					suffix = "0" + suffix;
				libraryId = initialLibraryId + suffix;
			}
			while (currentLibraries.containsKey(libraryId));

			libraryElement.setAttribute(JPFConstants.ATTR_LIBRARY_PATH, libraryPath);
			libraryElement.setAttribute(JPFConstants.ATTR_LIBRARY_ID, libraryId);
			libraryElement.setAttribute(JPFConstants.ATTR_LIBRARY_TYPE, libraryType);

			runtimeElement.appendChild(libraryElement);
		}
	}
}
