/*
 * Created on Jun 26, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service.impl;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.tandbergtv.watchpoint.studio.application.StudioRuntimeException;
import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory;
import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.PersistenceContextFactory;
import com.tandbergtv.watchpoint.studio.dto.IPersistable;
import com.tandbergtv.watchpoint.studio.service.IService;
import com.tandbergtv.watchpoint.studio.service.ServiceErrorCode;
import com.tandbergtv.watchpoint.studio.service.ServiceException;

/**
 * Default Service Implementation
 * 
 * @author Vijay Silva
 */
public abstract class ServiceImpl implements IService
{
	private static final Logger logger = Logger.getLogger(ServiceImpl.class);

	private static final String PERSISTENCE_OPERATION_NAME = "performOperation";

	protected PersistenceContextFactory pcFactory;

	protected DataAccessFactory daFactory;

	/**
	 * The Constructor
	 */
	protected ServiceImpl()
	{
		this.daFactory = DataAccessFactory.createFactory();
		this.pcFactory = PersistenceContextFactory.createFactory();
	}

	/**
	 * @return the pcFactory
	 */
	public PersistenceContextFactory getPersistenceContextFactory() {
		return pcFactory;
	}

	/**
	 * @param pcFactory the pcFactory to set
	 */
	public void setPersistenceContextFactory(PersistenceContextFactory pcFactory) {
		this.pcFactory = pcFactory;
	}

	/**
	 * @return the daFactory
	 */
	public DataAccessFactory getDataAccessFactory() {
		return daFactory;
	}

	/**
	 * @param daFactory the daFactory to set
	 */
	public void setDataAccessFactory(DataAccessFactory daFactory) {
		this.daFactory = daFactory;
	}

	/**
	 * Method to create a new initialized Persistence Context
	 * 
	 * @return An Initialized Persistence Context
	 */
	protected IPersistenceContext createPersistenceContext()
	{
		IPersistenceContext context = this.pcFactory.createPersistenceContext();
		context.initialize();

		return context;
	}
	
	/**
	 * Method to create a Data Access Object given an entity and the Persistence Context
	 * 
	 * @param entity
	 *            The Entity requiring data access operations
	 * @param context
	 *            The Persistence Context
	 * 
	 * @return The Data Access Object
	 */
	protected DataAccessInterface<? extends IPersistable, ? extends Serializable> createDAO(
			IPersistable entity, IPersistenceContext context)
	{
		return this.daFactory.createDataAccessObject(entity, context);
	}

	/**
	 * Method to close the Persistence Context, closing any state maintained.
	 * 
	 * @param context
	 *            The Persistence Context to close.
	 */
	protected void closePersistenceContext(IPersistenceContext context)
	{
		try
		{
			context.close();
		}
		catch (Exception ex)
		{
			String msg = "Failed to close the Persistence Context: " + ex.getLocalizedMessage();
			logger.error(msg, ex);
		}
	}

	/**
	 * Start a new Transaction for the input persistence context.
	 * 
	 * @param context
	 *            The Persistence Context
	 */
	protected void beginTransaction(IPersistenceContext context)
	{
		context.beginTransaction();
	}

	/**
	 * Commit the current transaction of the input persistence context.
	 * 
	 * @param context
	 *            The persistence context
	 */
	protected void commitTransaction(IPersistenceContext context)
	{
		context.commitTransaction();
	}

	/**
	 * Rollback the current transaction of the input persistence context. Catch any exceptions
	 * caused when rolling back the transaction and log them.
	 * 
	 * @param context
	 *            The persistence context
	 */
	protected void rollbackTransaction(IPersistenceContext context)
	{
		try
		{
			context.rollbackTransaction();
		}
		catch (Exception ex)
		{
			String msg = "Failed when rolling back the current transaction: "
					+ ex.getLocalizedMessage();
			logger.error(msg, ex);
		}
	}

	/**
	 * Embeds the creation of the Persistence Context and the transaction management for any of the
	 * public service methods. Creates and initializes the Persistence Context, and then invokes the
	 * method with the additional persistence context argument via reflection.
	 * 
	 * @param parameterTypes
	 *            The types of the parameters for the public service method
	 * @param args
	 *            The list of arguments for the method
	 * 
	 * @return The object returned by the operation invoked via reflection.
	 */
	protected Object performOperation(Class<?>[] parameterTypes, Object... args)
	{
		Method method = this.findMethodToInvoke(parameterTypes, args);

		int argCount = (args != null) ? args.length : 0;
		Object[] newArgs = new Object[argCount + 1];
		for (int i = 0; i < argCount; i++)
			newArgs[i] = args[i];

		IPersistenceContext context = this.createPersistenceContext();
		newArgs[argCount] = context;

		try
		{
			this.beginTransaction(context);

			Object result = method.invoke(this, newArgs);

			this.commitTransaction(context);

			return result;
		}
		catch (Exception ex)
		{
			String operationName = this.getClass().getName() + "." + method.getName() + "().";
			String msg = "Failed to perform operation: " + operationName;
			logger.error(msg, ex);

			this.rollbackTransaction(context);
			throw this.generateException(method.getName(), ex);
		}
		finally
		{
			this.closePersistenceContext(context);
		}
	}

	/*
	 * Finds the method to invoke. The method name is expected to be the same as the public
	 * operation name, but the input parameters will include an additional 'IPersistenceContext'
	 * object.
	 */
	private Method findMethodToInvoke(Class<?>[] parameterTypes, Object... args)
	{
		String methodName = null;

		try
		{
			String className = ServiceImpl.class.getName();
			StackTraceElement[] stack = Thread.currentThread().getStackTrace();
			int index = -1;
			for (int i = 0; i < stack.length; i++)
			{
				String currClassName = stack[i].getClassName();
				String currMethodName = stack[i].getMethodName();
				if (className.equals(currClassName)
						&& PERSISTENCE_OPERATION_NAME.equals(currMethodName))
				{
					index = i + 1;
					break;
				}
			}

			methodName = stack[index].getMethodName();

			List<Class<?>> types = new ArrayList<Class<?>>();
			int paramTypeCount = (parameterTypes != null) ? parameterTypes.length : 0;
			for (int i = 0; i < paramTypeCount; i++)
				types.add(parameterTypes[i]);
			types.add(IPersistenceContext.class);

			return this.getClass().getDeclaredMethod(methodName, types.toArray(new Class[0]));
		}
		catch (Exception ex)
		{
			String msg = "Failed to find method: " + methodName + " in class: "
					+ this.getClass().getName() + " to invoke.";
			logger.error(msg, ex);
			throw new StudioRuntimeException(msg, ex);
		}
	}

	/*
	 * Throw an appropriate runtime exception
	 */
	private RuntimeException generateException(String operationName, Throwable cause)
	{
		// If 'InvocationTargetException', get the actual cause of failure
		if (cause instanceof InvocationTargetException)
		{
			cause = cause.getCause();
		}

		if (cause instanceof RuntimeException)
		{
			return (RuntimeException) cause;
		}

		String msg = "Failed to perform operation: " + this.getClass().getName() + "."
				+ operationName + "().";
		return new ServiceException(ServiceErrorCode.GENERAL_ERROR, msg, cause);
	}
}
