package com.tandbergtv.watchpoint.studio.ui;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;

/**
 * Utility class for managing temporary project, folder and files needed to store the template xml
 * and the gpd xml while user is operating on a template.
 * 
 * @author Imran Naqvi
 * 
 */
public class TemplateResourcesUtility {
	private static final String WATCH_POINT_TEMP_PROJECT = "WatchPointTempProject";

	private static final String TEMP_GPD_FILE = "gpd.xml";

	private static final String TEMPLATE_FOLDER = "TempTemplates";

	private static final String NODE_DEFINITION_FOLDER = "TempNodeDefinitions";

	private static final String TEMP_TEMPLATE_FILE = "processdefinition.xml";

	private static final String TEMP_NODE_DEFINITION_FILE = "nodedefinition.xml";

	private static IJavaProject PROJECT = null;

	/**
	 * Creates a process definition file from the template xml and a gpd file from the gpd xml. If
	 * the files already exist, their contents are replaced with the corresponding xmls. The files
	 * are created in the temp project within the temp folder. If either the project or the folder
	 * dont exist, they are created.
	 * 
	 * @param templateXML
	 * @param gpdXML
	 * @return The File for the Workflow Template
	 */
	public static IFile createTemplateFile(String templateXML, String gpdXML) {
		return createTempFiles(TEMPLATE_FOLDER, TEMP_TEMPLATE_FILE, templateXML, TEMP_GPD_FILE,
				gpdXML);
	}

	/**
	 * Creates a node definition file from the node definition xml and a gpd file from the gpd xml. If
	 * the files already exist, their contents are replaced with the corresponding xmls. The files
	 * are created in the temp project within the temp folder. If either the project or the folder
	 * dont exist, they are created.
	 * 
	 * @param nodeDefinitionXML
	 * @param gpdXML
	 * @return The File for the Node Definition
	 */
	public static IFile createNodeDefinitionFile(String nodeDefinitionXML, String gpdXML) {
		return createTempFiles(NODE_DEFINITION_FOLDER, TEMP_NODE_DEFINITION_FILE,
				nodeDefinitionXML, TEMP_GPD_FILE, gpdXML);
	}

	/**
	 * Gets the template gpd xml string from the gpd file in the template folder.
	 * 
	 * @return gpd xml
	 */
	public static String getTemplateGpdXml() {
		if (PROJECT == null)
			return null;

		return getGpdXml(TEMPLATE_FOLDER);
	}

	/**
	 * Gets the node definition gpd xml string from the gpd file in the node definition folder.
	 * 
	 * @return gpd xml
	 */
	public static String getNodeDefinitionGpdXml() {
		if (PROJECT == null)
			return null;

		return getGpdXml(NODE_DEFINITION_FOLDER);
	}

	private static String getGpdXml(String folderName){
		IFile gpdFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
				new Path(PROJECT.getPath() + "//" + folderName + "//" + TEMP_GPD_FILE));
		try {
			String line;
			BufferedReader in = new BufferedReader(new InputStreamReader(gpdFile.getContents()));
			StringBuffer buffer = new StringBuffer();
			while ((line = in.readLine()) != null) {
				buffer.append(line); 
				buffer.append(System.getProperty("line.separator"));
			}
			return buffer.toString();
		}
		catch (IOException e) {
			throw new UIException("Could not read from gpd file.", e);
		}
		catch (CoreException e) {
			throw new UIException("Could not open gpd file.", e);
		}
	}
	
	private static IFile createTempFiles(String folderName, String templateFileName,
			String templateXML, String gpdFileName, String gpdXML) {
		IFile gpdFile, processDefinitionFile = null;
		try {
			PROJECT = createJavaProject(WATCH_POINT_TEMP_PROJECT);

			processDefinitionFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
					new Path(PROJECT.getPath() + "//" + folderName + "//" + templateFileName));
			/* Delete process definition file if it already exists */
			if (processDefinitionFile.exists())
				processDefinitionFile.setContents(new ByteArrayInputStream(templateXML.getBytes()),
						true, false, null);
			else
				processDefinitionFile.create(new ByteArrayInputStream(templateXML.getBytes()),
						true, null);

			gpdFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
					new Path(PROJECT.getPath() + "//" + folderName + "//" + gpdFileName));

			/* Delete gpd file if it already exists */
			if (gpdFile.exists())
				gpdFile.setContents(new ByteArrayInputStream(gpdXML.getBytes()), true, false, null);
			else
				gpdFile.create(new ByteArrayInputStream(gpdXML.getBytes()), true, null);
		}
		catch (CoreException e) {
			e.printStackTrace();
		}
		return processDefinitionFile;
	}

	/**
	 * Creates a IJavaProject.
	 */
	@SuppressWarnings("restriction")
	private static IJavaProject createJavaProject(String projectName) throws CoreException {
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IProject project = root.getProject(projectName);
		if (!project.exists()) {
			project.create(null);
		}
		else {
			project.refreshLocal(IResource.DEPTH_INFINITE, null);
		}
		if (!project.isOpen()) {
			project.open(null);
		}
		/*Creating temp folder for templates*/
		String templateFolderName = TEMPLATE_FOLDER;
		IFolder templateFolder = project.getFolder(templateFolderName);
		if (!templateFolder.exists()) {
			org.eclipse.jdt.internal.ui.util.CoreUtility.createFolder(templateFolder, false, true,
					null);
		}
		/*Creating temp folder for node definitions*/
		String nodeDefFolderName = NODE_DEFINITION_FOLDER;
		IFolder nodeDefFolder = project.getFolder(nodeDefFolderName);
		if (!nodeDefFolder.exists()) {
			org.eclipse.jdt.internal.ui.util.CoreUtility.createFolder(nodeDefFolder, false, true,
					null);
		}


		if (!project.hasNature(JavaCore.NATURE_ID)) {
			addNatureToProject(project, JavaCore.NATURE_ID, null);
		}
		try {
			IJavaProject jproject = JavaCore.create(project);
			jproject.open(null);

			return jproject;
		}
		catch (Exception e) {
			throw new IllegalStateException("error de project java : " + e);
		}

	}

	/**
	 * Add a Nature to a Project
	 * 
	 * @param proj
	 * @param natureId
	 * @param monitor
	 * @throws CoreException
	 */
	private static void addNatureToProject(IProject proj, String natureId, IProgressMonitor monitor)
			throws CoreException {
		IProjectDescription description = proj.getDescription();
		String[] prevNatures = description.getNatureIds();
		String[] newNatures = new String[prevNatures.length + 1];
		System.arraycopy(prevNatures, 0, newNatures, 0, prevNatures.length);
		newNatures[prevNatures.length] = natureId;
		description.setNatureIds(newNatures);
		proj.setDescription(description, monitor);
	}

}
