package com.tandbergtv.watchpoint.studio.ui.editor;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;

import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.fs.dao.ResourceGroupParser;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes;
import com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter;
import com.tandbergtv.watchpoint.studio.external.wpexport.WatchPointDTOExporterFactory;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.util.UIValidator;
import com.tandbergtv.watchpoint.studio.ui.util.UIValidator.Validation;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * The editor for WatchPoint templates. This class extends JBPM's JpdlEditor and uses WatchPoint
 * templates instead of regular process definitions.
 * 
 * @author Imran Naqvi
 * 
 */
public class ResourceGroupEditor extends EditorPart {

	private FileEditorInput input; 
	
	private boolean dirty = false;
	
	private FormToolkit formToolkit = null;
	private Section section = null;
	private Composite mainFormContainer = null;

	private Text nameText, allocationStrategyText, currentImageText, imageText;
	private Combo resourceTypeCombo, functionalTypeCombo, protectionKeyCombo;
	private Group allocationStyleGroup;
	private Button internalRadio, externalRadio, clearCurrentImageCheck, imageFileBrowseButton;
	private Map<String, ProtectionKey> protectionKeyNameObjectMap = new HashMap<String, ProtectionKey>();
	private Map<String, ResourceType> resourceTypeNameObjectMap = new HashMap<String, ResourceType>();

	private ModifyListener validationListener;
	
	private IResourceGroupService rgService;
	private IResourceTypeService rtService;
	private ResourceGroup group;

	private InputFileListener inputFileListener;
	
	public ResourceGroupEditor() {
		this.rgService = ServiceFactory.createFactory().createResourceGroupService();
		this.rtService = ServiceFactory.createFactory().createResourceTypeService();
		
		this.validationListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (!dirty) {
					dirty = true;
					firePropertyChange(IEditorPart.PROP_DIRTY);
				}
				validate();
			}
		};
	}
	
	class InputFileListener implements IResourceChangeListener, IResourceDeltaVisitor {
		public void resourceChanged(IResourceChangeEvent event) {
			if (event.getType() == IResourceChangeEvent.POST_CHANGE) {
				IResourceDelta delta = event.getDelta();
				try {
					delta.accept(this);
				} catch (CoreException e) {
				}
			}
		}

		public boolean visit(IResourceDelta delta) throws CoreException {
			IResource resource = delta.getResource();
			if (resource instanceof IFile) {
				IFile file = (IFile) resource;
				if (file.equals(((IFileEditorInput) getEditorInput()).getFile())) {
					if (delta.getKind() == IResourceDelta.REMOVED || delta.getKind() == IResourceDelta.REPLACED)
						close();
					return false;
				}
			}
			return true;
		}
	}
	
	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		super.setInput(input);
		super.setSite(site);
		this.input = (FileEditorInput) input;
		this.group = ResourceGroupParser.parse(this.input.getFile());
		Utility.setWindow(site.getWorkbenchWindow());
		inputFileListener = new InputFileListener();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(inputFileListener);
	}

	@Override
	public boolean isSaveOnCloseNeeded() {
		return isDirty();
	}
	
	@Override
	public void dispose() {
		if (inputFileListener != null) {
			ResourcesPlugin.getWorkspace().removeResourceChangeListener(inputFileListener);
			inputFileListener = null;
		}
		super.dispose();
	}

	@Override
	public String getPartName() {
		if (this.group == null) {
			return super.getPartName();
		}
		return this.group.getName();
	}

	/* ------------------------------------------------------------------------------- */
	
	public boolean isPageComplete() {
		return ((section.getDescription() == null || section.getDescription().isEmpty()) && nameText.getText().length() > 0);
	}
	
	//-----------------------------------------
	//		GETTER METHODS FOR PAGE VALUES
	//-----------------------------------------
	
	public String getResourceGroupName() {
		return nameText.getText();
	}
	
	public ResourceType getResourceType() {
		return resourceTypeNameObjectMap.get(resourceTypeCombo.getText());
	}
	
	public String getFunctionalType() {
		return functionalTypeCombo.getText();
	}

	/*
	 * @return true, if internal; false, if external
	 */
	public boolean isAllocationStyleInternal() {
		return internalRadio.getSelection();
	}
	
	public String getAllocationStrategy() {
		return allocationStrategyText.getText();
	}

	public boolean isClearCurrentImage() {
		return clearCurrentImageCheck.getSelection();
	}

	
	public String getImageFile() {
		return imageText.getText();
	}
	
	public ProtectionKey getProtectionKey() {
		String protectionKeyName = protectionKeyCombo.getText();
		return (protectionKeyNameObjectMap.containsKey(protectionKeyName))
				? protectionKeyNameObjectMap.get(protectionKeyName)
				: new ProtectionKey(protectionKeyName);
	}
		
	//-----------------------------------------
	//			CONTROL CREATION
	//-----------------------------------------

	public void createControl(Composite parent) {
		createClientArea(parent);
		createNameControl(mainFormContainer, null);
		createrResourceTypeControl(mainFormContainer, nameText);
		createFunctionalTypeControl(mainFormContainer, resourceTypeCombo);
		createAllocationStyleControl(mainFormContainer, functionalTypeCombo);
		createAllocationStrategyControl(mainFormContainer, allocationStyleGroup);
		createCurrentImageControl(mainFormContainer, allocationStrategyText);
		createImageControl(mainFormContainer, currentImageText);
		createProtectionKeyCombo(mainFormContainer, imageText);
		Dialog.applyDialogFont(mainFormContainer);
	}

	private String convertNullToEmpty(String str) {
		return ((str == null) ? "" : str);
	}
	private String convertBlankToNull(String str) {
		if(str != null && str.trim().length() > 0) {
			return str;	
		}
		else {
			return null;
		}
	}

	
	/*
	 * Sets the layout of the client area.
	 * 
	 * @param parent
	 *            the parent composite of the client area
	 * @return the composite representing the client area
	 */
	private Composite createClientArea(Composite parent) {
		GridData gridData = new GridData();
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessVerticalSpace = true;
		Composite mainContainer = getFormToolkit().createComposite(parent);
		mainContainer.setLayout(new GridLayout());
		mainContainer.setLayoutData(gridData);
		createScrolledForm(mainContainer);
		
		parent.setLayout(new GridLayout());
		
		
		return mainContainer;
	}
	
	/**
	 * This method initializes scrolledForm	
	 *
	 */
	private void createScrolledForm(Composite mainContainer) {
		GridData gridData5 = new GridData();
		gridData5.horizontalAlignment = GridData.FILL;
		gridData5.verticalAlignment = GridData.FILL;
		gridData5.grabExcessVerticalSpace = true;
		gridData5.grabExcessHorizontalSpace = true;
		ScrolledForm scrolledForm = getFormToolkit().createScrolledForm(mainContainer);
		scrolledForm.setText("Resource Group");
		scrolledForm.setLayoutData(gridData5);
		scrolledForm.getBody().setLayout(new GridLayout());
		createSection(scrolledForm);
	}
	
	/**
	 * This method initializes section	
	 *
	 */
	private void createSection(ScrolledForm form) {
		GridData gridData1 = new GridData();
		gridData1.horizontalAlignment = GridData.FILL;
		gridData1.grabExcessHorizontalSpace = true;
		gridData1.verticalAlignment = GridData.CENTER;
		section = getFormToolkit().createSection(form.getBody(), Section.DESCRIPTION | ExpandableComposite.TITLE_BAR | SWT.BORDER);
		createComposite();
		section.setText("Properties");
		section.setLayoutData(gridData1);
		section.setClient(mainFormContainer);
	}
	
	/**
	 * This method initializes composite1	
	 *
	 */
	private void createComposite() {
		GridData gridData4 = new GridData();
		gridData4.grabExcessHorizontalSpace = true;
		gridData4.verticalAlignment = GridData.CENTER;
		gridData4.horizontalAlignment = GridData.FILL;
		
		GridData gridData3 = new GridData();
		gridData3.horizontalAlignment = GridData.END;
		gridData3.verticalAlignment = GridData.CENTER;
		
		GridData gridData2 = new GridData();
		gridData2.horizontalAlignment = GridData.END;
		gridData2.verticalAlignment = GridData.CENTER;
		
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.marginTop = 10;
		
		mainFormContainer = new Composite(section, SWT.NONE);
		mainFormContainer.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));
		mainFormContainer.setLayout(gridLayout);
	}
	
	/**
	 * This method initializes formToolkit	
	 * 	
	 * @return org.eclipse.ui.forms.widgets.FormToolkit	
	 */
	private FormToolkit getFormToolkit() {
		if (formToolkit == null) {
			formToolkit = new FormToolkit(Display.getCurrent());
		}
		return formToolkit;
	}
	
	/*
	 * Creates the name text box along with its label.
	 * 
	 * @param parent
	 *            the parent composite of the text
	 */
	private void createNameControl(Composite parent, Control prevControl) {
		createLabel("Name : ", parent, prevControl);
		nameText = createText(parent, prevControl);
		nameText.setTextLimit(50);
		nameText.setText(convertNullToEmpty(group.getName()));
		nameText.addModifyListener(validationListener);
		nameText.setEditable(false);
	}
	
	/*
	 * Creates the node definition type combo box along with its label.
	 * 
	 * @param composite
	 *            the parent composite of the combo
	 */
	private void createrResourceTypeControl(Composite composite, Control prevControl) {
		createLabel("Resource Type : ", composite, prevControl);
		resourceTypeCombo = createCombo(composite, prevControl, true);
		fillResourceTypeCombo();
		
		resourceTypeCombo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				String functionalTypeComboText = functionalTypeCombo.getText();
				populateFunctionalTypeCombo();
				functionalTypeCombo.setText(functionalTypeComboText);
			}
		});
	}
	
	/*
	 * Fills the node definition type combo. 
	 */
	private void fillResourceTypeCombo() {
		List<ResourceType> resourceTypes = this.rtService.getResourceTypeList();
		int foundAt = -1;
		if(resourceTypes != null) {
			//sort resource types based on name (ignore case)
			Collections.sort(resourceTypes, new Comparator<ResourceType>() {
				public int compare(ResourceType o1, ResourceType o2) {
					return compareIgnoreCase(o1.getName(), o2.getName());
				}
			});
			int i = 0;
			for(ResourceType resourceType : resourceTypes) {
				if (group.getSystemId() != null && group.getSystemId().equals(resourceType.getSystemId())) {
					foundAt = i;
				}
				resourceTypeCombo.add(resourceType.getName());
				resourceTypeNameObjectMap.put(resourceType.getName(), resourceType);
				i++;
			}
		}
		if (foundAt != -1) {
			resourceTypeCombo.select(foundAt);
		} else {
			// Not found, set the SystemID in the combo
			String systemID = group.getSystemId();
			if (systemID == null || systemID.equals(String.valueOf(ResourceType.DEFAULT_ID))) {
				systemID = ""; // shows empty value.
			}
			resourceTypeCombo.add(systemID);
			resourceTypeCombo.setText(systemID);
		}
		if (group.getSystemId() != null && !group.getSystemId().equals(String.valueOf(ResourceType.DEFAULT_ID))) {
			// Once systemID is set, it cannot change.
			resourceTypeCombo.setEnabled(false);
		}

	}

	private void createFunctionalTypeControl(Composite parent, Control prevControl) {
		createLabel("Functional Type : ", parent, prevControl);
		functionalTypeCombo = createCombo(parent, prevControl, false);
		functionalTypeCombo.setTextLimit(256);
		populateFunctionalTypeCombo();
		functionalTypeCombo.setText(convertNullToEmpty(group.getFunctionalType()));
		functionalTypeCombo.addModifyListener(validationListener);
	}
	
	private void populateFunctionalTypeCombo() {
		ResourceType resourceType = getResourceType();
		if(resourceType == null)
			return;
		List<String> functionalTypes = rgService.getAllFunctionalTypesForResourceType(resourceType.getSystemId());
		if(functionalTypes != null) {
			//sort functional types (ignore case)
			Collections.sort(functionalTypes, new Comparator<String>() {
				public int compare(String o1, String o2) {
					return compareIgnoreCase(o1, o2);
				}
			});
			functionalTypeCombo.setItems(functionalTypes.toArray(new String[] {}));
		}
	}
	
	private void createAllocationStyleControl(Composite parent, Control prevControl) {
		createLabel("Allocation Style : ", parent, prevControl);
		allocationStyleGroup = createGroup(parent, prevControl);
		
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		allocationStyleGroup.setLayout(gridLayout);
		
		internalRadio = new Button(allocationStyleGroup, SWT.RADIO);
		internalRadio.setText("Internal");
		internalRadio.setSelection(true);
		internalRadio.setSelection(group.isInternallyAcquired());
		SelectionListener sl = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				allocationStrategyText.setEnabled(internalRadio.getSelection());
				if(externalRadio.getSelection())
					allocationStrategyText.setText("");
				validate();
			}
		};
		internalRadio.addSelectionListener(sl);
		
		externalRadio = new Button(allocationStyleGroup, SWT.RADIO);
		externalRadio.setText("External");
		externalRadio.setSelection(!group.isInternallyAcquired());
		externalRadio.addSelectionListener(sl);
	}
	
	private void createAllocationStrategyControl(Composite parent, Control prevControl) {
		createLabel("Allocation Strategy: ", parent, prevControl);
		allocationStrategyText = createText(parent, prevControl);
		allocationStrategyText.setTextLimit(256);
		allocationStrategyText.setText(convertNullToEmpty(group.getAllocationStrategyClassName()));
		allocationStrategyText.setEnabled(group.isInternallyAcquired());
		allocationStrategyText.addModifyListener(validationListener);
	}
	
	private void createCurrentImageControl(Composite parent, Control prevControl) {
		createLabel("Current Image: ", parent, prevControl);
		
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.marginWidth = 0;
		
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayoutData(getComponentGridData());
		composite.setLayout(gridLayout);
		
		currentImageText = createText(composite, prevControl); 
		currentImageText.setEnabled(false);
		currentImageText.setLayoutData(getComponentGridData());

		clearCurrentImageCheck = new Button(composite, SWT.CHECK);
		clearCurrentImageCheck.setText("Clear");
		clearCurrentImageCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				dirty = true;
				firePropertyChange(IEditorPart.PROP_DIRTY);
			}
		});
	}
	
	private void createImageControl(Composite parent, Control prevControl) {
		createLabel("Image : ", parent, prevControl);

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.marginWidth = 0;
		
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayoutData(getComponentGridData());
		composite.setLayout(gridLayout);
		
		imageText = new Text(composite, SWT.BORDER);
		imageText.addModifyListener(validationListener);
		imageText.setLayoutData(getComponentGridData());
		
		imageFileBrowseButton = new Button(composite, SWT.PUSH);
		imageFileBrowseButton.setText("Browse");
		setCurrentImageText(group.getImageFileName());
		imageFileBrowseButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				String path = getPathFromUser();
				if(path != null)
					imageText.setText(path);
			}
		});
	}
	
	private void setCurrentImageText(String text) {
		currentImageText.setText(convertNullToEmpty(text));
		if(text == null || text.isEmpty()) {
			clearCurrentImageCheck.setEnabled(false);
			clearCurrentImageCheck.setSelection(false);
		} else {
			clearCurrentImageCheck.setEnabled(true);
			clearCurrentImageCheck.setSelection(false);
		}
	}
	
	private String getPathFromUser() {
		FileDialog fileDialog = new FileDialog(Display.getCurrent().getActiveShell(),
				SWT.SYSTEM_MODAL);
		fileDialog.setText("Please select the image file.");
		return fileDialog.open();
	}

	private void createProtectionKeyCombo(Composite parent, Control prevControl) {
		createLabel("Protection Key : ", parent, prevControl);
		protectionKeyCombo = createCombo(parent, prevControl, false);
		protectionKeyCombo.setTextLimit(50);
		if(group.getProtectionKey() != null)
			protectionKeyCombo.setText(convertNullToEmpty(group.getProtectionKey().getName()));
		protectionKeyCombo.addModifyListener(validationListener);
		protectionKeyCombo.addFocusListener(new FocusAdapter() {
			public void focusLost(FocusEvent e) {
					protectionKeyCombo.setText(protectionKeyCombo.getText().toUpperCase());				
			}
		});
		fillProtectionKeyCombo();
	}
	
	private void fillProtectionKeyCombo() {
		List<ProtectionKey> protectionKeys = rgService.getAllProtectionKeys();
		if(protectionKeys != null)
			//sort protection keys by their name (ignore case)
			Collections.sort(protectionKeys, new Comparator<ProtectionKey>() {
				public int compare(ProtectionKey o1, ProtectionKey o2) {
					return compareIgnoreCase(o1.getName(), o2.getName());
				}
			});
			for(ProtectionKey protectionKey : protectionKeys) {
				protectionKeyCombo.add(protectionKey.getName());
				protectionKeyNameObjectMap.put(protectionKey.getName(), protectionKey);
			}
	}
	
	//	-------------------------------------------------------------------------------
	//							GENERAL METHODS
	//	-------------------------------------------------------------------------------
	
	private void validate() {
		if(validateValue("Name", nameText.getText(), UIValidator.Validation.NOT_BLANK, UIValidator.Validation.TRIMMED)) {
			return;
		}
		
		if(validateValue("Functional Type", functionalTypeCombo.getText(), UIValidator.Validation.TRIMMED)) {
			return;
		}
		
		if(internalRadio.getSelection()) {
			if(validateValue("Allocation Strategy", allocationStrategyText.getText(), UIValidator.Validation.TRIMMED)) {
				return;
			}
		}
		
		if(validateValue("Image", imageText.getText(), UIValidator.Validation.TRIMMED)) {
			return;
		}
		
		if(validateValue("Protection Key", protectionKeyCombo.getText(), UIValidator.Validation.NOT_BLANK, UIValidator.Validation.TRIMMED)) {
			return;
		}
		
		clearErrorMessage();
	}
	
	/*
	 * 		Validates the field. 
	 * 
	 * @param propertyName
	 * @param propertyValue
	 * @param validations
	 * @return true if field has errors
	 */
	private boolean validateValue(String propertyName, String propertyValue, Validation... validations) {
		String errorMessage = UIValidator.validateValue(propertyName, propertyValue, validations);
		boolean hasErrors = false;
		if (errorMessage != null) {
			updateErrorMessage(propertyName, errorMessage);
			hasErrors = true;
		}
		return hasErrors;
	}
	
	private void updateErrorMessage(String field, String errorMessage) {
		errorMessage = " * " + errorMessage;
		section.setDescription(errorMessage);
	}
	
	private void clearErrorMessage() {
		section.setDescription("");
	}
	
	private Label createLabel(String text, Composite parent, Control prevControl) {
		Label label = new Label(mainFormContainer, SWT.WRAP);
		label.setText(text);
		label.setLayoutData(getLSGridData());
		
		return label;
	}
	
	private Text createText(Composite parent, Control prevControl) {
		Text text = new Text(parent, SWT.BORDER);
		text.setLayoutData(getComponentGridData());
		return text;
	}
	
	private Combo createCombo(Composite parent, Control prevControl, boolean readOnly) {
		int style = SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL;
		if(readOnly)
			style |= SWT.READ_ONLY;
		Combo combo = new Combo(parent, style);
		combo.setLayoutData(getComponentGridData());
		return combo;
	}
	
	private Group createGroup(Composite parent, Control prevControl) {
		Group group = new Group(parent, SWT.NULL);
		group.setLayoutData(getComponentGridData());
		return group;
	}
	
	private GridData getLSGridData() {
		GridData gridDataLabel = new GridData();
		gridDataLabel.horizontalAlignment = GridData.END;
		gridDataLabel.verticalAlignment = GridData.CENTER;
		
		return gridDataLabel;
	}
	
	private GridData getComponentGridData() {
		GridData gridDataComponent = new GridData();
		gridDataComponent.grabExcessHorizontalSpace = true;
		gridDataComponent.verticalAlignment = GridData.CENTER;
		gridDataComponent.horizontalAlignment = GridData.FILL;
		
		return gridDataComponent;
	}
	
	private int compareIgnoreCase(String str1, String str2) {
		if(str1 == null && str2 == null)
			return 0;
		if(str1 == null)
			return -1;
		if(str2 == null)
			return 1;
		return str1.compareToIgnoreCase(str2);

	}

	@Override
	public boolean isDirty() {
		return dirty;
	}
	
	@Override
	public void doSave(IProgressMonitor progress) {
		if (!isPageComplete()) {
			MessageDialog.openError(Display.getCurrent().getActiveShell(), "Validation Error", 
					"Resource Group contains errors. Please, clear all errors before saving.");
			return;
		}
		group.setName(getResourceGroupName());
		String systemID = null;
		ResourceType resourceType = getResourceType();
		if (resourceType == null) {
			systemID = resourceTypeCombo.getText();
			if (systemID == null || systemID.isEmpty()) {
				// User has not set systemID yet, use the DEFAULT_ID
				// Once systemID is set, it cannot change.
				systemID = String.valueOf(ResourceType.DEFAULT_ID);
			}
		} else {
			systemID = resourceType.getSystemId();
		}
		group.setSystemId(systemID);
		group.setFunctionalType(convertBlankToNull(getFunctionalType()));
		group.setInternallyAcquired(isAllocationStyleInternal());
		group.setAllocationStrategyClassName(convertBlankToNull(getAllocationStrategy()));
		if(isClearCurrentImage()) {
			group.setImageFileName(null);
			setCurrentImageText(null);
		}
		if(getImageFile() != null && getImageFile().length() > 0) {
			try {
				// Gets the group folder
				IFolder groupFolder = (IFolder) input.getFile().getParent();
				// Gets the resource folder
				IFolder resourcesFolder = groupFolder.getFolder(WatchPointPluginConstants.PLUGIN_RESOURCE_FOLDER_NAME);
				if (!resourcesFolder.exists()) {
					resourcesFolder.create(true, true, null);
				}
				// Gets the images folder
				IFolder imagesFolder = resourcesFolder.getFolder(ResourceGroup.RESOURCE_GROUP_IMAGES_FOLDER);
				if (!imagesFolder.exists()) {
					imagesFolder.create(true, true, null);
				}
				String imageFileName = (new File(getImageFile())).getName();
				IFile imageFile = imagesFolder.getFile(imageFileName);
				ByteArrayInputStream baos = new ByteArrayInputStream(FileUtil.readFile(getImageFile()));
				if (!imageFile.exists()) {
					imageFile.create(baos, true, null);
				} else {
					imageFile.setContents(baos, true, false, null);
				}
				imageFileName = ResourceGroup.RESOURCE_GROUP_IMAGES_FOLDER + "/" + imageFileName;
				group.setImageFileName(imageFileName);
				setCurrentImageText(imageFileName);
				imageText.setText("");
			} catch (IOException e) {
				String message = "Exception while reading the image file: " + getImageFile();
				throw new UIException(message, e);
			} catch (CoreException e) {
				String message = "Exception while creating image folder: " + getImageFile();
				throw new UIException(message, e);
			}
		}
		group.setProtectionKey(getProtectionKey());
		
		// Increment group version
		group.setVersion(group.getVersion()+1);

		Map<String, Object> exportInputAttrs = new HashMap<String, Object>();
        exportInputAttrs.put(ExporterInputAttributes.EXPORT_FOLDER_PATH, input.getFile().getFullPath().toPortableString());
        exportInputAttrs.put(ExporterInputAttributes.TARGET_FILESYSTEM, ExporterInputAttributes.TargetFilesystem.ECLIPSE_WORKSPACE);
        
		// Use the Export Service to export the Template
		WatchPointDTOExporterFactory exporterFactory = WatchPointDTOExporterFactory.createFactory();
		IWatchPointDTOExporter<ResourceGroup> exporter = exporterFactory.getExporter(ResourceGroup.class);
		try {
			exporter.export(group, exportInputAttrs);
			dirty = false;
			firePropertyChange(IEditorPart.PROP_DIRTY);
		} catch (ExportFailureException e) {
			Utility.reportError("Error while saving ResourceGroup", e);
		}
	}
	
	@Override
	public void doSaveAs() {
		// Not allowed
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void createPartControl(Composite c) {
		createControl(c);		
	}

	@Override
	public void setFocus() {
	}

	public void close() {
		Display display = getSite().getShell().getDisplay();
		display.asyncExec(new Runnable() {
			public void run() {
				if (getFormToolkit() != null) {}
					getSite().getPage().closeEditor(ResourceGroupEditor.this, false);
			}
		});
	}
}
