package com.tandbergtv.watchpoint.studio.ui.editor.expression;
 
import java.util.Arrays;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.eval.IEvaluationContext;
import org.eclipse.jdt.internal.core.util.SimpleDocument;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.text.java.AbstractJavaCompletionProposal;
import org.eclipse.jdt.internal.ui.text.java.ProposalInfo;
import org.eclipse.jdt.ui.text.java.CompletionProposalCollector;
import org.eclipse.jdt.ui.text.java.CompletionProposalComparator;
import org.eclipse.jdt.ui.text.java.IJavaCompletionProposal;
import org.eclipse.jface.fieldassist.ContentProposal;
import org.eclipse.jface.fieldassist.IContentProposal;
import org.eclipse.jface.fieldassist.IContentProposalProvider;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.BadPositionCategoryException;
import org.eclipse.jface.text.DefaultPositionUpdater;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IInformationControlCreator;
import org.eclipse.jface.text.IPositionUpdater;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;


/**
 *      Offers Java syntax completion for scripts, adding NodeDefinitions variables 
 * in the completion context.
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class SnippetCompletionProcessor implements IContentProposalProvider {
    
    public static final String DEFAULT_VARS = "org.jbpm.graph.exe.Token token = null; org.jbpm.graph.exe.ProcessInstance processInstance = null;";
    protected IProject project;
    protected CompletionProposalComparator fComparator = new CompletionProposalComparator();
    protected String contextVars;

    /**
     * Order the given proposals.
     */
    private ICompletionProposal[] order(IJavaCompletionProposal[] proposals) {
        Arrays.sort(proposals, fComparator);
        return proposals;   
    }   

    @Override
    public IContentProposal[] getProposals(String originalContents, int originalPosition) {
        try {
            // Invokes the java completion proposal collector
            IJavaProject javaProject = JavaCore.create(project);
            CompletionProposalCollector fCollector = new CompletionProposalCollector(javaProject);
            IEvaluationContext context = javaProject.newEvaluationContext();
            context.setImports(ExpressionUtil.DEFAULT_IMPORTS);

            String modifiedContents = originalContents;
            int modifiedPosition = originalPosition;
            if (contextVars != null) {
                modifiedContents = DEFAULT_VARS + contextVars + originalContents;
                modifiedPosition = DEFAULT_VARS.length() + contextVars.length() + originalPosition;
            }
            context.codeComplete(modifiedContents, modifiedPosition, fCollector);

            IJavaCompletionProposal[] javaCompletionResults = (IJavaCompletionProposal[]) order(fCollector
                    .getJavaCompletionProposals());

            return toIProposals(javaCompletionResults, modifiedContents, modifiedPosition);
        } catch (JavaModelException x) {
            throw new RuntimeException(x);
        }
    }

    private IContentProposal[] toIProposals(IJavaCompletionProposal[] javaCompletionResults, String modifiedContents,
            int modifiedPosition) {
        SimpleDocument sd = null;
        String replacement = null;
        int cursorPosition = 0;
        IContentProposal[] contentAssistResults = new IContentProposal[javaCompletionResults.length];
        ProposalDecorator proposalDecorator = null;
        for (int i = 0; i < javaCompletionResults.length; i++) {
            proposalDecorator = new ProposalDecorator((AbstractJavaCompletionProposal) javaCompletionResults[i]);

            // have to predict how the result will be. Simple way to do that is to simulate it.
            sd = new SimpleDocument(modifiedContents);
            proposalDecorator.apply(sd, (char) 0, modifiedPosition);

            // now, to extract the proposal apply, lets just remove the previous content
            replacement = sd.get().substring(DEFAULT_VARS.length() + contextVars.length());

            // checks if it ends in ")" to position the cursor properly
            cursorPosition = replacement.lastIndexOf(")");
            if (cursorPosition == (replacement.length() - 1)) {
                int tempCursorPosition = replacement.lastIndexOf("(");
                if (tempCursorPosition >= 0) {
                    tempCursorPosition += 1;
                    cursorPosition = tempCursorPosition;
                }
            } else {
                cursorPosition = replacement.length();
            }

            contentAssistResults[i] = new ContentProposal(replacement, proposalDecorator.getDisplayString(), null, cursorPosition);
        }
        return contentAssistResults;
    }

    public void setProject(IProject project) {
        this.project = project;
    }

    public void setContextVars(String contextVars) {
        this.contextVars = contextVars;
    }
}

/**
 * This decorator is only intended to solve an issue inside the {@link AbstractJavaCompletionProposal.#apply(IDocument, char, int)} method.
 */
class ProposalDecorator extends AbstractJavaCompletionProposal {
    private AbstractJavaCompletionProposal component;

    public ProposalDecorator(AbstractJavaCompletionProposal component) {
        this.component = component;
    }

    @Override
    public char[] getTriggerCharacters() {
        return component.getTriggerCharacters();
    }

    @Override
    public void setTriggerCharacters(char[] triggerCharacters) {
        component.setTriggerCharacters(triggerCharacters);
    }

    @Override
    public void setProposalInfo(ProposalInfo proposalInfo) {
        component.setProposalInfo(proposalInfo);
    }

    @Override
    public void setCursorPosition(int cursorPosition) {
        component.setCursorPosition(cursorPosition);
    }

    @Override
    public void apply(IDocument document, char trigger, int offset) {
        try {
            String replacement = getReplacementString();

            // reference position just at the end of the document change.
            int referenceOffset= getReplacementOffset() + (replacement == null ? 0 : replacement.length());
            
            final ReferenceTracker referenceTracker= new ReferenceTracker();
            referenceTracker.preReplace(document, referenceOffset);

            replace(document, getReplacementOffset(), getReplacementLength(), replacement);

            referenceOffset= referenceTracker.postReplace(document);
            setReplacementOffset(referenceOffset - (replacement == null ? 0 : replacement.length()));

        } catch (BadLocationException x) {
            // ignore
        }
    }

    @Override
    public void apply(ITextViewer viewer, char trigger, int stateMask, int offset) {
        component.apply(viewer, trigger, stateMask, offset);
    }

    @Override
    public Point getSelection(IDocument document) {
        return component.getSelection(document);
    }

    @Override
    public IContextInformation getContextInformation() {
        return component.getContextInformation();
    }

    @Override
    public void setContextInformation(IContextInformation contextInformation) {
        component.setContextInformation(contextInformation);
    }

    @Override
    public String getDisplayString() {
        return component.getDisplayString();
    }

    @Override
    public String getAdditionalProposalInfo() {
        return component.getAdditionalProposalInfo();
    }

    @Override
    public Object getAdditionalProposalInfo(IProgressMonitor monitor) {
        return component.getAdditionalProposalInfo(monitor);
    }

    @Override
    public int getContextInformationPosition() {
        return component.getContextInformationPosition();
    }

    @Override
    public int getReplacementOffset() {
        return component.getReplacementOffset();
    }

    @Override
    public void setReplacementOffset(int replacementOffset) {
        component.setReplacementOffset(replacementOffset);
    }

    @Override
    public int getPrefixCompletionStart(IDocument document, int completionOffset) {
        return component.getPrefixCompletionStart(document, completionOffset);
    }

    @Override
    public int getReplacementLength() {
        return component.getReplacementLength();
    }

    @Override
    public void setReplacementLength(int replacementLength) {
        component.setReplacementLength(replacementLength);
    }

    @Override
    public String getReplacementString() {
        return component.getReplacementString();
    }

    @Override
    public void setReplacementString(String replacementString) {
        component.setReplacementString(replacementString);
    }

    @Override
    public CharSequence getPrefixCompletionText(IDocument document, int completionOffset) {
        return component.getPrefixCompletionText(document, completionOffset);
    }

    @Override
    public Image getImage() {
        return component.getImage();
    }

    @Override
    public void setImage(Image image) {
        component.setImage(image);
    }

    @Override
    public boolean isValidFor(IDocument document, int offset) {
        return component.isValidFor(document, offset);
    }

    @Override
    public boolean validate(IDocument document, int offset, DocumentEvent event) {
        return component.validate(document, offset, event);
    }

    @Override
    public int getRelevance() {
        return component.getRelevance();
    }

    @Override
    public void setRelevance(int relevance) {
        component.setRelevance(relevance);
    }

    @Override
    public void selected(ITextViewer viewer, boolean smartToggle) {
        component.selected(viewer, smartToggle);
    }

    @Override
    public void unselected(ITextViewer viewer) {
        component.unselected(viewer);
    }

    @Override
    public IInformationControlCreator getInformationControlCreator() {
        return component.getInformationControlCreator();
    }

    @Override
    public String getSortString() {
        return component.getSortString();
    }

    @Override
    public StyledString getStyledDisplayString() {
        return component.getStyledDisplayString();
    }

    @Override
    public void setStyledDisplayString(StyledString text) {
        component.setStyledDisplayString(text);
    }

    @Override
    public String toString() {
        return component.toString();
    }

    @Override
    public IJavaElement getJavaElement() {
        return component.getJavaElement();
    }

    /**
     * This class was a "copy'n paste" from AbstractJavaCompletionProposal.ReferenceTracker due to accessibility constraints. If you find yourself "fixing" an issue
     * here, please keep in mind that it's class purpose is only to support the extension of AbstractJavaCompletionProposal by the {@link ProposalDecorator} class.
     */
    class ReferenceTracker {

        /** The reference position category name. */
        private static final String CATEGORY= "reference_position"; //$NON-NLS-1$
        /** The position updater of the reference position. */
        private final IPositionUpdater fPositionUpdater= new DefaultPositionUpdater(CATEGORY);
        /** The reference position. */
        private final Position fPosition= new Position(0);

        /**
         * Called before document changes occur. It must be followed by a call to postReplace().
         *
         * @param document the document on which to track the reference position.
         * @param offset the offset
         * @throws BadLocationException if the offset describes an invalid range in this document
         *
         */
        public void preReplace(IDocument document, int offset) throws BadLocationException {
            fPosition.setOffset(offset);
            try {
                document.addPositionCategory(CATEGORY);
                document.addPositionUpdater(fPositionUpdater);
                document.addPosition(CATEGORY, fPosition);

            } catch (BadPositionCategoryException e) {
                // should not happen
                JavaPlugin.log(e);
            }
        }

        /**
         * Called after the document changed occurred. It must be preceded by a call to preReplace().
         *
         * @param document the document on which to track the reference position.
         * @return offset after the replace
         */
        public int postReplace(IDocument document) {
            try {
                document.removePosition(CATEGORY, fPosition);
                document.removePositionUpdater(fPositionUpdater);
                document.removePositionCategory(CATEGORY);

            } catch (BadPositionCategoryException e) {
                // should not happen
                JavaPlugin.log(e);
            }
            return fPosition.getOffset();
        }
    }
}