/*
 * Created on Sep 19, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IPersistableElement;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.editor.input.IInputChangedListener;
import com.tandbergtv.watchpoint.studio.ui.editor.input.IWatchPointEditorInput;
import com.tandbergtv.watchpoint.studio.ui.editor.input.InputChangedEvent;
import com.tandbergtv.watchpoint.studio.ui.editor.input.InputChangedEventManager;
import com.tandbergtv.watchpoint.studio.ui.model.IEditableElement;

/**
 * Editor Input that contains a Resource Type object.
 * 
 * @author Vijay Silva
 */
public class ResourceTypeEditorInput implements IWatchPointEditorInput, IEditableElement
{
	private ResourceType resourceType;

	private ResourceType persistedResourceType;

	private Message message;

	private boolean isEditable = true;

	private InputChangedEventManager eventManager = new InputChangedEventManager();

	/**
	 * Default Constructor. The input is marked editable.
	 * 
	 * @param resourceType
	 *            The ResourceType that serves as the Editor Input
	 */
	public ResourceTypeEditorInput(ResourceType resourceType)
	{
		this(resourceType, null, true);
	}

	/**
	 * Constructor
	 * 
	 * @param resourceType
	 *            The ResourceType that serves as the Editor Input
	 * @param isEditable
	 *            flag to indicate if the Resource Type can be edited or not
	 */
	public ResourceTypeEditorInput(ResourceType resourceType, boolean isEditable)
	{
		this(resourceType, null, isEditable);
	}

	/**
	 * Constructor.
	 * 
	 * @param resourceType
	 *            The Resource Type to edit
	 * @param message
	 *            The currently selected message
	 */
	public ResourceTypeEditorInput(ResourceType resourceType, Message message)
	{
		this(resourceType, message, true);
	}

	/**
	 * Constructor.
	 * 
	 * @param resourceType
	 *            The Resource Type to edit
	 * @param message
	 *            The currently selected message
	 * @param isEditable
	 *            flag to indicate if the Resource Type can be edited or not
	 */
	public ResourceTypeEditorInput(ResourceType resourceType, Message message, boolean isEditable)
	{
		this.setPersistedResourceType(resourceType);
		this.message = message;
		this.isEditable = isEditable;
	}

	/**
	 * @see org.eclipse.ui.IEditorInput#exists()
	 */
	public boolean exists()
	{
		return false;
	}

	/**
	 * @see org.eclipse.ui.IEditorInput#getImageDescriptor()
	 */
	public ImageDescriptor getImageDescriptor()
	{
		return ImageDescriptor.getMissingImageDescriptor();
	}

	/**
	 * @see org.eclipse.ui.IEditorInput#getName()
	 */
	public String getName()
	{

		String name = this.resourceType.getName();
		name += " [" + this.resourceType.getSystemId() + "]";

		if (!this.isEditable)
			name += " (Read Only)";

		return name;
	}

	/**
	 * @see org.eclipse.ui.IEditorInput#getPersistable()
	 */
	public IPersistableElement getPersistable()
	{
		return null;
	}

	/**
	 * @see org.eclipse.ui.IEditorInput#getToolTipText()
	 */
	public String getToolTipText()
	{
		return this.getName();
	}

	/**
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapterClass)
	{
		Object adapter = null;

		if (adapterClass.isInstance(this.resourceType))
		{
			adapter = this.resourceType;
		}
		else if (adapterClass.isInstance(this.message))
		{
			adapter = this.message;
		}
		else if (adapterClass.isInstance(this))
		{
			return this;
		}
		else
		{
			adapter = Platform.getAdapterManager().getAdapter(this, adapterClass);
		}

		return adapter;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IEditableElement#canEdit()
	 */
	public boolean canEdit()
	{
		return this.isEditable;
	}

	/**
	 * Get the Message selected for editing
	 * 
	 * @return The resourceType
	 */
	public ResourceType getResourceType()
	{
		return this.resourceType;
	}

	/**
	 * Get the Message selected for editing
	 * 
	 * @return The message
	 */
	public Message getMessage()
	{
		return this.message;
	}

	/**
	 * Get the persisted Resource Type stored when the Editor Input is first created, or when the
	 * Resource Type is updated.
	 * 
	 * @return The persisted Resource Type
	 */
	public ResourceType getPersistedResourceType()
	{
		return this.persistedResourceType;
	}

	/*
	 * Set the Persisted Resource Type and update the copy maintained.
	 */
	private void setPersistedResourceType(ResourceType resourceType)
	{
		this.persistedResourceType = resourceType;
		this.resourceType = resourceType.clone();
	}

	/**
	 * Update the Resource Type with the newer data. Fires an input changed event when called.
	 * Should only be called after a sync with the Resource Type in the persistence layer.
	 * 
	 * @param updatedResourceType
	 *            The updated Resource Type
	 */
	public void updatePersistedResourceType(ResourceType updatedResourceType)
	{
		this.persistedResourceType = updatedResourceType;
		this.resourceType = updatedResourceType.clone();

		InputChangedEvent event = new InputChangedEvent(this);
		this.eventManager.fireInputChanged(event);
	}

	/**
	 * Add a new Message to the Resource Type
	 * 
	 * @param message
	 *            The Message that is added to
	 */
	public void addCreatedMessage(Message message)
	{
		this.persistedResourceType.addMessage(message);
		Message clonedMessage = message.clone();
		this.resourceType.addMessage(clonedMessage);

		InputChangedEvent event = new InputChangedEvent(this);
		this.eventManager.fireInputChanged(event);
	}

	/**
	 * Remove the message from the Resource Type
	 * 
	 * @param message
	 *            The message to remove
	 */
	public void removeDeletedMessage(Message message)
	{
		this.persistedResourceType.removeMessage(message);
		this.resourceType.removeMessage(message);

		InputChangedEvent event = new InputChangedEvent(this);
		this.eventManager.fireInputChanged(event);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.input.IWatchPointEditorInput#addInputChangedListener(com.tandbergtv.watchpoint.studio.ui.editor.input.IInputChangedListener)
	 */
	public void addInputChangedListener(IInputChangedListener listener)
	{
		this.eventManager.addInputChangedListener(listener);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.input.IWatchPointEditorInput#removeInputChangedListener(com.tandbergtv.watchpoint.studio.ui.editor.input.IInputChangedListener)
	 */
	public void removeInputChangedListener(IInputChangedListener listener)
	{
		this.eventManager.removeInputChangedListener(listener);
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode()
	{
		if (this.resourceType == null)
			return super.hashCode();

		int hashCode = 3 * this.resourceType.hashCode() - 421;
		return hashCode;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj)
	{
		if (!(obj instanceof ResourceTypeEditorInput))
			return false;

		ResourceTypeEditorInput input = (ResourceTypeEditorInput) obj;
		if (this.resourceType != null)
			return this.resourceType.equals(input.resourceType);

		return super.equals(obj);
	}
}
