package com.tandbergtv.watchpoint.studio.ui.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.jpdl.model.Description;

import com.tandbergtv.watchpoint.studio.ui.util.Utility;

public class DataCollectionReader {

    private final Map<String, String> messageNodeUrlByUid;
    private final Map<String, String> superstateUrlByName;
    private final Map<String, String> resourceTypeUrlByName;
    private final Map<String, ? super SemanticElement> samplesByUrl;

    private final SemanticElementFactory factory = new SemanticElementFactory(Utility.TEMPLATE_EDITOR_ID);
    
    public DataCollectionReader(Map<String, String> messageNodeUrlByUid, Map<String, String> superstateUrlByName,
            Map<String, String> resourceTypeUrlByName, Map<String, ? super SemanticElement> samplesByUrl) {
        this.messageNodeUrlByUid = messageNodeUrlByUid;
        this.superstateUrlByName = superstateUrlByName;
        this.resourceTypeUrlByName = resourceTypeUrlByName;
        this.samplesByUrl = samplesByUrl;
    }

    public NodeDefinition getNewSuperstate(String superstateName) {
        NodeDefinition nodeDefinition = getSampleSuperstateByName(superstateName);

        return nodeDefinition.cloneNodeDefinition();
    }

    public NodeDefinition getSampleSuperstateByName(String superstateName) {
        String nodeDefinitionUrl = superstateUrlByName.get(superstateName);
        NodeDefinition nodeDefinition = (NodeDefinition) samplesByUrl.get(nodeDefinitionUrl);
        if (nodeDefinitionUrl != null && nodeDefinition == null) {
            String content = Utility.readFile(nodeDefinitionUrl);

            nodeDefinition = (NodeDefinition) factory.createById(SemanticElementConstants.NODE_DEFN_SEID);
            XmlAdapter adapter = Utility.getAdapterFromString(content, factory);
            adapter.initialize(nodeDefinition);

            samplesByUrl.put(nodeDefinitionUrl, nodeDefinition);
        }
        return nodeDefinition;
    }

    public NodeDefinition getNewMessageNode(String uid) {
        NodeDefinition nodeDefinition = getSampleMessageNodeByUid(uid);

        return nodeDefinition.cloneNodeDefinition();
    }

    public NodeDefinition getSampleMessageNodeByUid(String uid) {
        String nodeDefinitionUrl = messageNodeUrlByUid.get(uid);
        NodeDefinition nodeDefinition = (NodeDefinition) samplesByUrl.get(nodeDefinitionUrl);
        if (nodeDefinitionUrl != null && nodeDefinition == null) {
            String content = Utility.readFile(nodeDefinitionUrl);

            nodeDefinition = (NodeDefinition) factory.createById(SemanticElementConstants.NODE_DEFN_SEID);
            XmlAdapter adapter = Utility.getAdapterFromString(content, factory);
            adapter.initialize(nodeDefinition);

            String description = extractDescription(nodeDefinitionUrl, uid);
            if (description != null) {
                nodeDefinition.setDescription((Description) factory.createById(SemanticElementConstants.DESCRIPTION_SEID));
                nodeDefinition.getDescription().setDescription(description);
            }

            samplesByUrl.put(nodeDefinitionUrl, nodeDefinition);
        }
        return nodeDefinition;
    }

    private String extractDescription(String nodeDefinitionUrl, String uid) {
        String projectName = extractProjectNameFromUrl(nodeDefinitionUrl);
        return  Utility.getDescriptionFromPluginXml(projectName, uid);
    }

    /**
     * Get the cache for message nodes.
     * 
     * @return
     */
    public Map<String, String> getMessageNodeUrlByUidMap() {
        return messageNodeUrlByUid;
    }

    /**
     * Get the URL of the message node with the given UID, null will be returned if the message node doesn't exist
     * 
     * @param uid
     *            UID of the message node
     * @return
     */
    public String getMessageNodeUrlByUid(String uid) {
        return messageNodeUrlByUid.get(uid);
    }

    public Map<String, String> getResourceTypeUrlByNameMap() {
        return resourceTypeUrlByName;
    }

    public List<String> getMessageNodeUidsByResourceType(String resourceTypeName) {
        List<String> result = new ArrayList<String>();
        String messageNodeUrl = null;
        for (String messageNodeUid : messageNodeUrlByUid.keySet()) {
            messageNodeUrl = messageNodeUrlByUid.get(messageNodeUid);
            String pluginUrl = new StringBuilder().append("/").append(extractProjectNameFromUrl(messageNodeUrl)).append("/plugin.xml").toString();
            if (resourceTypeUrlByName.get(resourceTypeName).equals(pluginUrl)) {
                result.add(messageNodeUid);
            }
        }

        return result;
    }

    public List<String> getSuperStateNamesByResourceType(String resourceTypeName) {
        List<String> result = new ArrayList<String>();
        String superstateUrl = null;
        for (String superstateName : superstateUrlByName.keySet()) {
            superstateUrl = superstateUrlByName.get(superstateName);
            String pluginUrl = new StringBuilder().append("/").append(extractProjectNameFromUrl(superstateUrl)).append("/plugin.xml").toString();
            if (resourceTypeUrlByName.get(resourceTypeName).equals(pluginUrl)) {
                result.add(superstateName);
            }
        }

        return result;
    }

    public String extractFileNameFromUrl(String url) {
        return url.substring(url.lastIndexOf("/") + 1, url.lastIndexOf("."));
    }

    public String getResourceTypeNameByMessageNodeUid(String uid) {
        String messageNodeUrl = messageNodeUrlByUid.get(uid);
        if (StringUtils.isBlank(messageNodeUrl)) {
            return null;
        }

        String pluginUrl = new StringBuilder().append("/").append(extractProjectNameFromUrl(messageNodeUrl)).append("/plugin.xml").toString();
        for (String resourceTypeName : resourceTypeUrlByName.keySet()) {
            if (resourceTypeUrlByName.get(resourceTypeName).equals(pluginUrl)) {
                return resourceTypeName;
            }
        }
        
        return null;
    }

    private String extractProjectNameFromUrl(String url) {
        return url.substring(1, url.indexOf("/", 1));
    }

    public Collection<String> getSuperstateNames() {
        return superstateUrlByName.keySet();
    }

    /**
     * Get the cache for superstate URLs
     * 
     * @return
     */
    public Map<String, String> getSuperstateUrlByNameMap() {
        return superstateUrlByName;
    }

    /**
     * Get the URL of the superstate with the given name, null will returned if the superstate doesn't exist.
     * 
     * @param name
     *            name of the superstate
     * @return
     */
    public String getSuperstateUrlByName(String name) {
        return superstateUrlByName.get(name);
    }
}