package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Controller;
import org.jbpm.gd.jpdl.model.MailElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.Subject;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.model.MailNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

public class MailElementMailInfoConfigurationComposite implements SelectionListener, FocusListener {
	
	public static MailElementMailInfoConfigurationComposite create(AbstractPropertySection parentPropertySection, TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		MailElementMailInfoConfigurationComposite result = new MailElementMailInfoConfigurationComposite();
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.parentPropertySection = parentPropertySection;
		result.create();
		return result;
	}

	private AbstractPropertySection parentPropertySection;
	private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;
	private MailNode mailElement;
	
	private Label destinationLabel;
	private CCombo destinationCombo;
	private Label attachmentLabel;
	private CCombo attachmentCombo;
	
	private Label subjectLabel;
	private Text subjectText;
	private Label bodyLabel;
	private Text bodyText;
	
	public void setMailElement(MailElement mailElement) {
		if (this.mailElement == mailElement) return;
		unhookListeners();
		this.mailElement = (MailNode) mailElement;
		if (mailElement == null) {
			clearControls();
		} else {
			updateControls();
			hookListeners();
		}
	}
	
	private void hookListeners() {
		destinationCombo.addFocusListener(this);
		attachmentCombo.addFocusListener(this);
		subjectText.addFocusListener(this);
		bodyText.addFocusListener(this);
	}
	
	private void unhookListeners() {
		destinationCombo.removeFocusListener(this);
		attachmentCombo.removeFocusListener(this);
		subjectText.removeFocusListener(this);
		bodyText.removeFocusListener(this);
	}
	
	private void clearControls() {
		destinationCombo.setText("");
		attachmentCombo.setText("");
		subjectText.setText("");
		bodyText.setText("");
	}
	
	private void updateControls() {
		fillCombos();
		destinationCombo.setText(mailElement.getTo() == null ? "" : mailElement.getTo());
		attachmentCombo.setText(mailElement.getAttachment() == null ? "" : mailElement.getAttachment());
		subjectText.setText(getSubjectText());
		bodyText.setText(getBodyText());
	}
	

	/*
	 * Fills the variable combo with node definition start variables.
	 */
	private void fillCombos() {
		destinationCombo.removeAll();
		NodeElementContainer elementContainer = Utility.getNodeElementContainer(this.parentPropertySection.getPart());

		List<? extends Variable> variables = Collections.emptyList();

		if (elementContainer instanceof WorkflowTemplate) {
			WorkflowTemplate template = (WorkflowTemplate) elementContainer;
			StartState startState = template.getStartState();
			if (startState != null && startState.getTask() != null) {
				Controller controller = startState.getTask().getController();
				variables = Arrays.asList(controller.getVariables());
			}
		} else if (elementContainer instanceof NodeDefinition) {
			NodeDefinition node = (NodeDefinition) elementContainer;
			variables = node.getVariables();
		}
		
		Collections.sort(variables, new Comparator<Variable>() {
			@Override
			public int compare(Variable var1, Variable var2) {
				String var1name = "";
				if (var1.getName() != null) {
					var1name = var1.getName();
				}
				String var2name = "";
				if (var2.getName() != null) {
					var2name = var2.getName();
				}
				return var1name.compareTo(var2name);
			}
		});
		
		this.destinationCombo.add("");
		this.attachmentCombo.add("");
		for (Variable element : variables) {
			WPVariable var = (WPVariable) element;
			String varType = var.getType();
			DataType type = DataType.STRING;
			if (varType != null && !varType.trim().isEmpty()) {
				type = DataType.valueOf(varType);
			}
			if (type == DataType.STRING || type == DataType.LIST) {
				this.destinationCombo.add(var.getName());
				this.attachmentCombo.add(var.getName());
			}
		}
	}
	
	private String getSubjectText() {
		String result = "";
		Subject subject = mailElement.getSubject();
		if (subject != null) {
			result = subject.getSubject() == null ? "" : subject.getSubject();
		}
		return result;
	}
	
	private String getBodyText() {
		String result = "";
		if (mailElement.getText() != null) {
			result = mailElement.getText().getText() == null ? "" : mailElement.getText().getText();
		}
		return result;
	}
	
	private void create() {
		destinationLabel = widgetFactory.createLabel(parent, "Destination");
		destinationCombo = widgetFactory.createCCombo(parent, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL);
		subjectLabel = widgetFactory.createLabel(parent, "Subject");
		subjectText = widgetFactory.createText(parent, "");
		attachmentLabel = widgetFactory.createLabel(parent, "Attachment");
		attachmentCombo = widgetFactory.createCCombo(parent, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL);
		bodyLabel = widgetFactory.createLabel(parent, "Body");
		bodyText = widgetFactory.createText(parent, "", SWT.MULTI | SWT.V_SCROLL);
		
		destinationLabel.setLayoutData(createDestinationLabelLayoutData());
		destinationCombo.setLayoutData(createDestinationComboLayoutData());
		subjectLabel.setLayoutData(createSubjectLabelLayoutData());
		subjectText.setLayoutData(createSubjectTextLayoutData());
		attachmentLabel.setLayoutData(createAttachmentLabelLayoutData());
		attachmentCombo.setLayoutData(createAttachmentComboLayoutData());
		bodyLabel.setLayoutData(createBodyLabelLayoutData());
		bodyText.setLayoutData(createBodyTextLayoutData());
	}
	
	private FormData createDestinationLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(0, 3);
		return data;
	}
	
	private FormData createDestinationComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(destinationLabel, 0);
		data.top = new FormAttachment(0, 2);
		data.right = new FormAttachment(100, 0);
		return data;
	}
	
	private FormData createSubjectLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(destinationCombo, 0);
		return data;
	}
	
	private FormData createSubjectTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(destinationCombo, 0);
		data.left.alignment = SWT.LEFT;
		data.top = new FormAttachment(destinationCombo, 0);
		data.right = new FormAttachment(100, 0);
		return data;
	}
	
	private FormData createAttachmentLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(subjectText, 0);
		return data;
	}
	
	private FormData createAttachmentComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(subjectText, 0);
		data.left.alignment = SWT.LEFT;
		data.top = new FormAttachment(subjectText, 0);
		data.right = new FormAttachment(100, 0);
		return data;
	}
	
	private FormData createBodyLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(attachmentCombo, 0);
		return data;
	}
	
	private FormData createBodyTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(attachmentCombo, 0);
		data.left.alignment = SWT.LEFT;
		data.top = new FormAttachment(attachmentCombo, 0);
		data.right = new FormAttachment(100, 0);
		data.bottom = new FormAttachment(100, 0);
		data.height = 50;
		return data;
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
	}
	
	private void updateSubject() {
		Subject subject = mailElement.getSubject();
		if (subject == null) {
			subject = (Subject)mailElement.getFactory().createById("org.jbpm.gd.jpdl.subject");
			mailElement.setSubject(subject);
		}
		subject.setSubject(subjectText.getText());
	}

	private void updateBody() {
		org.jbpm.gd.jpdl.model.Text text = mailElement.getText();
		if (text == null) {
			text = (org.jbpm.gd.jpdl.model.Text)mailElement.getFactory().createById("org.jbpm.gd.jpdl.text");
			mailElement.setText(text);
		}
		text.setText(bodyText.getText());
	}

	public void widgetSelected(SelectionEvent e) {
	}
	
	private void updateDestination() {
		String str = destinationCombo.getText();
		mailElement.setTo(str == null ? "" : str);
	}
	
	private void updateAttachment() {
		String str = attachmentCombo.getText();
		mailElement.setAttachment(str == null ? "" : str);
	}

	public void focusGained(FocusEvent e) {
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == destinationCombo) {
			updateDestination();
		} else if (e.widget == attachmentCombo) {
			updateAttachment();
		} else if (e.widget == subjectText) {
			updateSubject();
		} else if (e.widget == bodyText) {
			updateBody();
		}
	}
	
}
